﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using Tunynet.Repositories;

namespace Tunynet.Common
{
    /// <summary>
    /// 下载记录业务逻辑类
    /// </summary>
    public class AttachmentAccessRecordsService
    {
        private IAttachmentAccessRecordsRepository attachmentAccessRecordsRepository;

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="attachmentAccessRecordsRepository"></param>
        public AttachmentAccessRecordsService(IAttachmentAccessRecordsRepository attachmentAccessRecordsRepository)
        {
            this.attachmentAccessRecordsRepository = attachmentAccessRecordsRepository;
        }

        /// <summary>
        /// 创建或更新下载或浏览记录
        /// </summary>
        /// <param name="userId">下载用户UserId</param>
        /// <param name="userDisplayName">下载用户的名称</param>
        /// <param name="attachment">附件Id</param>
        /// <param name="accessType">附件记录访问类型</param>
        /// <returns>创建成功返回true，否则返回false</returns>
        public bool CreateOrUpdate(long userId, string userDisplayName, Attachment attachment, AccessType accessType = AccessType.Download)
        {
            if (attachment == null)
                return false;
            if ((accessType == AccessType.Download && IsDownloaded(userId, attachment.AttachmentId)) || (accessType == AccessType.Read && IsReaded(userId, attachment.AttachmentId)))
            {
                return attachmentAccessRecordsRepository.UpdateLastDownloadDate(userId, attachment.AttachmentId, accessType);
            }
            else
            {
                AttachmentAccessRecords record = AttachmentAccessRecords.New(attachment);
                record.UserId = userId;
                record.UserDisplayName = userDisplayName;
                record.AccessType = accessType;

                attachmentAccessRecordsRepository.Insert(record);

                return record.Id > 0;
            }
        }

        /// <summary>
        /// 更新最后下载时间
        /// </summary>
        /// <param name="userId">下载用户UserId</param>
        /// <param name="attachmentId">附件Id</param>
        /// <param name="accessType">访问类型</param>
        public bool UpdateLastDownloadDate(long userId, long attachmentId, AccessType accessType = AccessType.Download)
        {
            return attachmentAccessRecordsRepository.UpdateLastDownloadDate(userId, attachmentId, accessType);
        }

        /// <summary>
        /// 用户是否已经下载某个附件
        /// </summary>
        /// <param name="userId">下载用户UserId</param>
        /// <param name="attachmentId">附件Id</param>
        /// <returns>曾经下载过返回true，否则返回false</returns>
        public bool IsDownloaded(long userId, long attachmentId)
        {
            Dictionary<long, long> ids_AttachmentIds = attachmentAccessRecordsRepository.GetIds_AttachmentIdsByUser(userId, AccessType.Download);

            if (ids_AttachmentIds != null)
            {
                return ids_AttachmentIds.ContainsValue(attachmentId);
            }

            return false;
        }

        /// <summary>
        /// 用户是否已经浏览某个附件
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="attachmentId"></param>
        /// <returns></returns>
        public bool IsReaded(long userId, long attachmentId)
        {
            Dictionary<long, long> ids_AttachmentIds = attachmentAccessRecordsRepository.GetIds_AttachmentIdsByUser(userId, AccessType.Read);

            if (ids_AttachmentIds != null)
            {
                return ids_AttachmentIds.ContainsValue(attachmentId);
            }

            return false;
        }

        /// <summary>
        /// 获取附件的前topNumber条下载记录
        /// </summary>
        /// <param name="attachmentID">附件Id</param>
        /// <param name="topNumber">获取记录条数</param>
        public IEnumerable<AttachmentAccessRecords> GetTopsByAttachmentId(long attachmentID, int topNumber)
        {
            return attachmentAccessRecordsRepository.GetTopsByAttachmentId(attachmentID, topNumber);
        }

        /// <summary>
        /// 获取附件的下载记录分页显示
        /// </summary>
        /// <param name="attachmentID">附件Id</param>
        /// <param name="pageIndex">页码</param>
        public PagingDataSet<AttachmentAccessRecords> GetsByAttachmentId(long attachmentID, int pageIndex)
        {
            return attachmentAccessRecordsRepository.GetsByAttachmentId(attachmentID, pageIndex);
        }

        /// <summary>
        /// 获取用户的下载记录分页显示
        /// </summary>
        /// <param name="userId">下载用户UserId</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="needToBuy">是否需要购买</param>
        public PagingDataSet<AttachmentAccessRecords> GetsByUserId(long userId, int pageIndex, bool needToBuy = true)
        {
            return attachmentAccessRecordsRepository.GetsByUserId(userId, pageIndex, needToBuy);
        }

        /// <summary>
        /// 获取拥有者附件的下载记录分页显示
        /// </summary>
        /// <param name="ownerId">附件拥有者Id</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="needToBuy">是否需要够买</param>
        public PagingDataSet<AttachmentAccessRecords> GetsByOwnerId(long ownerId, int pageIndex, bool needToBuy = true)
        {
            return attachmentAccessRecordsRepository.GetsByOwnerId(ownerId, pageIndex, needToBuy);
        }

        /// <summary>
        /// 清理记录
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="attachmentId">附件Id</param>
        public void Clean(long userId = 0, long attachmentId = 0)
        {
            attachmentAccessRecordsRepository.Clean(userId, attachmentId);
        }
    }
}