﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Caching;
using Tunynet.Repositories;
using Tunynet.Utilities;

namespace Tunynet.Common
{
    /// <summary>
    /// Application仓储
    /// </summary>
    public class AttachmentRepository<T> : Repository<T>, IAttachmentRepository<T> where T : Attachment
    {

        /// <summary>
        /// 删除AssociateId相关的附件
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="associateId">附件关联Id（例如：博文Id、贴子Id）</param>
        public void DeletesByAssociateId(string tenantTypeId, long associateId)
        {
            var sql = Sql.Builder;
            sql.Append("DELETE  FROM tn_Attachments").Where("TenantTypeId=@0 and AssociateId=@1", tenantTypeId, associateId);

            int affectCount = CreateDAO().Execute(sql);
            if (affectCount > 0)
            {
                //更新缓存
                RealTimeCacheHelper.IncreaseAreaVersion("AssociateId", associateId);
                IEnumerable<object> attachmentIds = GetIdsByAssociateId(tenantTypeId, associateId);
                foreach (var id in attachmentIds.Cast<long>())
                {
                    RealTimeCacheHelper.IncreaseEntityCacheVersion(id);
                }
            }
        }

        /// <summary>
        /// 删除UserId相关的附件
        /// </summary>
        /// <param name="tenantTypeId"></param>
        /// <param name="userId"></param>
        public void DeletesByUserId(string tenantTypeId, long userId)
        {
            var sql = Sql.Builder;
            sql.Append("DELETE  FROM tn_Attachments").Where("TenantTypeId=@0 and UserId=@1", tenantTypeId, userId);

            int affectCount = CreateDAO().Execute(sql);
            if (affectCount > 0)
            {
                //更新缓存
                IEnumerable<object> attachmentIds = GetIdsByUserId(tenantTypeId, userId);
                foreach (var id in attachmentIds.Cast<long>())
                {
                    RealTimeCacheHelper.IncreaseEntityCacheVersion(id);
                }
            }
        }

        /// <summary>
        /// 依据UserId获取附件列表（用于UserId与附件一对多关系）
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="userId">附件关联Id</param>
        /// <returns>附件列表</returns>
        public IEnumerable<T> GetsByUserId(string tenantTypeId, long userId)
        {
            IEnumerable<object> attachmentIds = GetIdsByUserId(tenantTypeId, userId);
            return PopulateEntitiesByEntityIds(attachmentIds);
        }

        /// <summary>
        /// 依据UserId获取附件Id列表（用于UserId与附件一对多关系）
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="userId">附件关联Id</param>
        /// <returns>附件Id列表</returns>
        private IEnumerable<object> GetIdsByUserId(string tenantTypeId, long userId)
        {
            var sql = Sql.Builder;
            sql.Select("AttachmentId")
               .From("tn_Attachments")
               .Where("TenantTypeId=@0", tenantTypeId)
               .Where("UserId=@0", userId);

            List<object> attachmentIds = CreateDAO().FetchFirstColumn(sql).ToList();

            return attachmentIds;
        }

        /// <summary>
        /// 依据AssociateId获取附件列表（用于AssociateId与附件一对多关系）
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="associateId">附件关联Id</param>
        /// <returns>附件列表</returns>
        public IEnumerable<T> GetsByAssociateId(string tenantTypeId, long associateId)
        {
            IEnumerable<object> attachmentIds = GetIdsByAssociateId(tenantTypeId, associateId);
            return PopulateEntitiesByEntityIds(attachmentIds);
        }

        /// <summary>
        /// 依据AssociateId获取附件Id列表（用于AssociateId与附件一对多关系）
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="associateId">附件关联Id</param>
        /// <returns>附件Id列表</returns>
        private IEnumerable<object> GetIdsByAssociateId(string tenantTypeId, long associateId)
        {
            var sql = Sql.Builder;
            sql.Select("AttachmentId")
               .From("tn_Attachments")
               .Where("TenantTypeId=@0", tenantTypeId)
               .Where("AssociateId=@0", associateId)
               .OrderBy("AttachmentId");
            var attachmentIds = CreateDAO().FetchFirstColumn(sql).ToList();

            return attachmentIds;
        }

        /// <summary>
        /// 搜索附件并分页显示
        /// </summary>
        /// <param name="tenantTypeId">附件租户类型</param>
        /// <param name="keyword">搜索关键词</param>
        /// <param name="pageSize">每页大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <returns></returns>
        public PagingDataSet<T> Gets(string tenantTypeId, string keyword, int pageSize, int pageIndex)
        {
            var sql = Sql.Builder;

            if (!String.IsNullOrEmpty(tenantTypeId))
                sql.Where("TenantTypeId = @0", tenantTypeId);
            if (!String.IsNullOrEmpty(keyword))
                sql.Where("FriendlyFileName like @0", "%" + StringUtility.StripSQLInjection(keyword) + "%");

            sql.OrderBy("AttachmentId  ASC");

            return GetPagingEntities(pageSize, pageIndex, sql);
        }


        /// <summary>
        /// 删除垃圾数据
        /// </summary>
        public void DeleteTrashDatas()
        {
            IEnumerable<TenantType> tenantTypes = new TenantTypeRepository().Gets(MultiTenantServiceKeys.Instance().Attachment());
            //2017-5-17 在MySql中 不能先select出同一表中的某些值，再update 或delete 这个表(在同一语句中) 已改正
            List<Sql> sqls = new List<Sql>();
            sqls.Add(Sql.Builder.Append("delete from tn_AttachmentAccessRecords where not exists  (select 1 from  tn_Users  where tn_Users.UserId = tn_AttachmentAccessRecords.UserId)"));

            sqls.Add(Sql.Builder.Append("delete from tn_AttachmentAccessRecords where not exists  (select 1  from  tn_attachments  where tn_attachments.AttachmentId = tn_AttachmentAccessRecords.AttachmentId)"));

            //修改删除附件垃圾数据时不将tn_Attachments.UserId字段设置为0
            //例如：当用户新建一个内容项时，此时AssociateId为0
            //      然后被当作垃圾附件处理，tn_Attachments.UserId 被设置为0
            //      然后再次被当作垃圾附件处理
            //      附件转换时并不会对tn_Attachments.UserId 做处理，然后导致这个附件会一直被当作垃圾附件
            sqls.Add(Sql.Builder.Append("update tn_Attachments set tn_Attachments.AssociateId = 0,tn_Attachments.OwnerId = 0")
            .Where("not exists (select 1  from tn_Users  where tn_Attachments.UserId = tn_Users.UserId) "));

            foreach (var tenantType in tenantTypes)
            {
                Type type = Type.GetType(tenantType.ClassType);
                if (type == null)
                    continue;

                var pd = TableInfo.FromPoco(type);
                sqls.Add(Sql.Builder.Append("update tn_Attachments set tn_Attachments.AssociateId = 0,tn_Attachments.OwnerId = 0")
                                    .Where("not exists  (select 1 from  " + pd.TableName + " where tn_Attachments.AssociateId = " + pd.TableName + "." + pd.PrimaryKey + ")  and tn_Attachments.TenantTypeId = @0"
                                    , tenantType.TenantTypeId));
            }

            CreateDAO().Execute(sqls);
        }

        /// <summary>
        /// 附件转换
        /// </summary>
        /// <param name="attachmentIds">最终的所有附件Id集合</param>
        /// <param name="associateId">需要将附件关联到的内容Id</param>
        /// <param name="tenantTypeId">租户类型</param>
        public void ToggleTemporaryAttachments(IEnumerable<long> attachmentIds, long associateId, string tenantTypeId)
        {
            var oldAttachmentIds = GetsByAssociateId(tenantTypeId, associateId)?.Select(n => n.AttachmentId);
            IEnumerable<long> needToDeleteAttachmentIds = oldAttachmentIds;
            IEnumerable<long> needToToggleAttachmentIds = attachmentIds;

            needToToggleAttachmentIds = attachmentIds?.Except(oldAttachmentIds ?? new List<long>());
            needToDeleteAttachmentIds = oldAttachmentIds?.Except(attachmentIds ?? new List<long>());

            var sqls = new List<Sql>();

            var sql = Sql.Builder;

            if (needToToggleAttachmentIds != null && needToToggleAttachmentIds.Any())
            {
                sqls.Add(Sql.Builder.Append("UPDATE tn_Attachments SET tn_Attachments.AssociateId=@0 WHERE tn_Attachments.AttachmentId IN (@attachmentIds)", associateId, new { attachmentIds = needToToggleAttachmentIds }));
            }

            if (needToDeleteAttachmentIds != null && needToDeleteAttachmentIds.Any())
            {
                sqls.Add(Sql.Builder.Append("UPDATE tn_Attachments SET tn_Attachments.AssociateId=@0 WHERE tn_Attachments.AttachmentId IN (@attachmentIds)", 0, new { attachmentIds = needToDeleteAttachmentIds }));
            }

            CreateDAO().Execute(sqls);

            if (needToToggleAttachmentIds != null && needToToggleAttachmentIds.Any())
            {
                //更新缓存
                foreach (var id in needToToggleAttachmentIds)
                {
                    var entity = Get(id);
                    entity.AssociateId = associateId;
                    //更新缓存
                    base.OnUpdated(entity);
                }
            }
        }

        /// <summary>
        /// 获取需删除的垃圾附件
        /// </summary>
        /// <param name="beforeDays">多少天之前的附件</param>
        public IEnumerable<T> GetTrashTemporaryAttachments(int beforeDays)
        {
            var sql = Sql.Builder;

            sql.Select("AttachmentId")
               .From("tn_Attachments")
               .Where("AssociateId=0")
               .Where("DateCreated <= @0 ", DateTime.Now.AddDays(-beforeDays));
            IEnumerable<object> attachmentIds = CreateDAO().FetchFirstColumn(sql);

            return PopulateEntitiesByEntityIds(attachmentIds);
        }

        /// <summary>
        /// 删除垃圾临时附件
        /// </summary>
        /// <param name="beforeDays">多少天之前的附件</param>
        public void DeleteTrashTemporaryAttachments(int beforeDays)
        {
            var sql = Sql.Builder;
            sql.Append("DELETE  FROM tn_Attachments ")
            .Where("AssociateId=0")
            .Where("DateCreated <= @0 ", DateTime.Now.AddDays(-beforeDays));
            CreateDAO().Execute(sql);
        }

        #region OwnerId相关(已废弃)

        ///// <summary>
        ///// 删除OwnerId相关的附件
        ///// </summary>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="ownerId">拥有者Id</param>
        //public void DeletesByOwnerId(string tenantTypeId, long ownerId)
        //{
        //    var sql = Sql.Builder;
        //    sql.Append("DELETE  FROM tn_Attachments ").Where("TenantTypeId=@0 and OwnerId=@1", tenantTypeId, ownerId);

        //    int affectCount = CreateDAO().Execute(sql);

        //    if (affectCount > 0)
        //    {
        //        //获取被影响的Id集合
        //        IEnumerable<object> attachmentIds = GetIdsByOwnerId(ownerId, tenantTypeId);
        //        foreach (var id in attachmentIds.Cast<long>())
        //        {
        //            RealTimeCacheHelper.IncreaseEntityCacheVersion(id);
        //        }
        //    }
        //}

        ///// <summary>
        ///// 获取拥有者的所有附件或者拥有者一种租户类型的附件
        ///// </summary>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="ownerId">拥有者Id</param>
        ///// <returns>附件列表</returns>
        //public IEnumerable<T> Gets(long ownerId, string tenantTypeId)
        //{
        //    IEnumerable<object> attachmentIds = GetIdsByOwnerId(ownerId, tenantTypeId);
        //    return PopulateEntitiesByEntityIds(attachmentIds);
        //}

        ///// <summary>
        ///// 获取拥有者的所有附件或者拥有者一种租户类型的附件
        ///// </summary>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="ownerId">拥有者Id</param>
        ///// <returns>附件Id列表</returns>
        //private IEnumerable<object> GetIdsByOwnerId(long ownerId, string tenantTypeId)
        //{
        //    var sql = Sql.Builder;
        //    sql.Select("AttachmentId")
        //       .From("tn_Attachments")
        //       .Where("OwnerId=@0", ownerId);
        //    if (!string.IsNullOrEmpty(tenantTypeId))
        //        sql.Where("TenantTypeId = @0", tenantTypeId);

        //    List<object> attachmentIds = CreateDAO().FetchFirstColumn(sql).ToList();

        //    return attachmentIds;
        //}


        #endregion

        #region 临时附件相关(已废弃)

        ///// <summary>
        ///// 获取拥有者一种租户类型的临时附件
        ///// </summary>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="ownerId">拥有者Id</param>
        //public IEnumerable<T> GetTemporaryAttachments(long ownerId, string tenantTypeId)
        //{
        //    List<object> attachmentIds;
        //    var sql = Sql.Builder;
        //    sql.Select("AttachmentId")
        //       .From("tn_Attachments")
        //       .Where("OwnerId=@0", ownerId)
        //       .Where("AssociateId=0");

        //    if (!string.IsNullOrEmpty(tenantTypeId))
        //        sql.Where("TenantTypeId = @0", tenantTypeId);
        //    sql.OrderBy("AttachmentId");

        //    attachmentIds = CreateDAO().FetchFirstColumn(sql).ToList();

        //    return PopulateEntitiesByEntityIds(attachmentIds);
        //}

        ///// <summary>
        ///// 把临时附件转成正常附件
        ///// </summary>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="ownerId">拥有者Id</param>
        ///// <param name="associateId">附件关联Id</param>
        ///// <param name="attachmentIds">待转换的附件Id</param>
        //public void ToggleTemporaryAttachments(long ownerId, string tenantTypeId, long associateId, IEnumerable<long> attachmentIds)
        //{
        //    var sql = Sql.Builder;
        //    sql.Append("UPDATE tn_Attachments Set AssociateId=@0", associateId)
        //    .Where("TenantTypeId = @0", tenantTypeId)
        //    .Where("OwnerId = @0", ownerId);

        //    if (attachmentIds != null && attachmentIds.Count() > 0)
        //    {
        //        sql.Where("AttachmentId in (@attachmentIds)", new { attachmentIds = attachmentIds });
        //    }
        //    else
        //    {
        //        sql.Where("AssociateId=0");
        //    }

        //    CreateDAO().Execute(sql);
        //    RealTimeCacheHelper.IncreaseAreaVersion("AssociateId", associateId);
        //    var attachmentId = Gets(ownerId, tenantTypeId).Where(n => n.AssociateId == associateId || n.AssociateId == 0).Select(n => n.AttachmentId);
        //    foreach (var id in attachmentId.Cast<long>())
        //    {
        //        RealTimeCacheHelper.IncreaseEntityCacheVersion(id);
        //    }
        //}

        ///// <summary>
        ///// 把临时附件转成正常附件
        ///// </summary>
        ///// <param name="ownerId">拥有者Id</param>
        ///// <param name="tenantTypeId">租户类型Id</param>
        ///// <param name="associateId">附件关联Id</param>
        ///// <param name="generatorAssociateId">临时附件关联Id</param>
        //public void ToggleTemporaryAttachments(long ownerId, string tenantTypeId, long associateId, long generatorAssociateId)
        //{
        //    Sql sql = Sql.Builder;
        //    sql.Append("UPDATE tn_Attachments Set AssociateId=@0", associateId)
        //    .Where("TenantTypeId = @0", tenantTypeId)
        //    .Where("AssociateId = @0", generatorAssociateId)
        //    .Where("OwnerId = @0", ownerId);

        //    CreateDAO().Execute(sql);
        //    RealTimeCacheHelper.IncreaseAreaVersion("AssociateId", associateId);
        //    var attachmentId = Gets(ownerId, tenantTypeId).Where(n => n.AssociateId == associateId || n.AssociateId == 0).Select(n => n.AttachmentId);
        //    foreach (var id in attachmentId.Cast<long>())
        //    {
        //        RealTimeCacheHelper.IncreaseEntityCacheVersion(id);
        //    }
        //}

        #endregion
    }
}