﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System.Collections.Generic;
using Tunynet.Repositories;

namespace Tunynet.Common.Repositories
{
    /// <summary>
    /// 分类和内容关联项，需要的数据服务接口
    /// </summary>
    public interface IItemInCategoryRepository : IRepository<ItemInCategory>
    {
        /// <summary>
        /// 批量为内容项设置类别
        /// </summary>
        /// <param name="itemIds">内容项Id集合</param>
        /// <param name="categoryId">类别Id</param>
        int AddItemsToCategory(IEnumerable<long> itemIds, long categoryId);

        /// <summary>
        /// 为内容项批量设置类别
        /// </summary>
        /// <param name="categoryIds">类别Id集合</param>
        /// <param name="itemId">内容项Id</param>
        void AddCategoriesToItem(IEnumerable<long> categoryIds, long itemId);

        /// <summary>
        /// 清除内容项的所有类别
        /// </summary>
        /// <param name="itemId">内容项Id</param>
        /// <param name="ownerId">分类所有者</param>
        /// <param name="tenantTypeId">租户Id</param>
        int ClearCategoriesFromItem(long itemId, long? ownerId, string tenantTypeId);

        /// <summary>
        /// 删除分类下的所有的关联项
        /// </summary>
        /// <param name="categoryId">分类Id</param>
        int ClearItemsFromCategory(long categoryId);

        /// <summary>
        /// 获取类别的内容项集合
        /// </summary>
        /// <param name="categoryId">分类的Id集合</param>
        /// <param name="categorieIds"></param>
        /// <returns>内容项的ID集合</returns>
        IEnumerable<long> GetItemsOfCategory(long categoryId, IEnumerable<long> categorieIds);

        /// <summary>
        /// 获取类别的内容项集合
        /// </summary>
        /// <param name="categoryId">当前分类的Id</param>
        /// <param name="ids">分类的Id集合</param>
        /// <param name="pageSize">页面大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <param name="totalRecords">输出参数：总记录数</param>
        /// <returns>当页内容项的ID集合</returns>
        IEnumerable<long> GetItemIdsOfCategory(long categoryId, IEnumerable<long> ids, int pageSize, int pageIndex, out long totalRecords);

        /// <summary>
        /// 获取内容项的所有类别Id集合
        /// </summary>
        /// <param name="itemId">内容项Id</param>
        /// <param name="ownerId">分类所有者</param>
        /// <param name="tenantTypeId">租户Id</param>
        /// <returns>返回内容项的类别Id集合</returns>
        IEnumerable<long> GetCategoriesOfItem(long itemId, long? ownerId, string tenantTypeId);

        /// <summary>
        /// 删除分类同内容的关联项
        /// </summary>
        /// <param name="categoryId">分类Id</param>
        /// <param name="itemId">内容项Id</param>
        int DeleteItemInCategory(long categoryId, long itemId);

        /// <summary>
        /// 根据ItemId获取分类
        /// </summary>
        /// <param name="itemId">成员Id</param>
        /// <param name="tenantTypeId">租户ID</param>
        /// <returns></returns>
        ItemInCategory GetItems(long itemId, string tenantTypeId);

        /// <summary>
        /// 获取分类及其成员项的关系集合
        /// </summary>
        /// <param name="itemId">成员Id</param>
        /// <param name="categoryId">分类Id</param>
        /// <returns></returns>
        IEnumerable<ItemInCategory> GetItemsInCategories(long itemId, long categoryId);
    }
}