﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using Tunynet.Common;
using Tunynet.Repositories;

namespace Tunynet.Post
{
    /// <summary>
    /// 贴子仓储接口
    /// </summary>
    public interface IThreadRepository : IRepository<Thread>
    {
        /// <summary>
        /// 获取某个贴吧下的所有贴子（用于删除贴子）
        /// </summary>
        /// <param name="sectionId"></param>
        /// <returns></returns>
        IEnumerable<Thread> GetAllThreadsOfSection(long sectionId);

        /// <summary>
        /// 获取BarThread内容
        /// </summary>
        /// <param name="threadId"></param>
        string GetBody(long threadId);

        /// <summary>
        /// 获取解析过正文
        /// </summary>
        /// <returns></returns>
        string GetResolvedBody(long threadId);

        /// <summary>
        /// 获取用户的主题贴分页集合
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="ignoreAudit">是否忽略审核状态（作者或管理员查看时忽略审核状态）</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns>主题贴列表</returns>
        PagingDataSet<Thread> GetUserThreads(string tenantTypeId, long userId, bool ignoreAudit, int pageSize, int pageIndex, long? sectionId);

        /// <summary>
        /// 获取主题贴的排行数据
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="topNumber">前多少条</param>
        /// <param name="sortBy">主题贴排序依据</param>
        /// <returns></returns>
        IEnumerable<Thread> GetTops(string tenantTypeId, int topNumber, SortBy_BarThread sortBy);

        /// <summary>
        /// 根据标签名获取主题贴排行分页集合
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="tagName">标签名</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <returns>主题贴列表</returns>
        PagingDataSet<Thread> Gets(string tenantTypeId, string tagName, SortBy_BarThread sortBy, int pageIndex, int pageSize);

        /// <summary>
        /// 根据贴吧获取主题贴分页集合
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴吧分类Id</param>
        /// <param name="orderBySticky">是否按置顶排序</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="BarDate">贴子显示时间范围</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <returns>主题贴列表</returns>
        PagingDataSet<Thread> Gets(long? sectionId, long? ownerId, long? categoryId, bool? orderBySticky, SortBy_BarThread sortBy, int pageSize, int pageIndex, SortBy_BarDateThread BarDate, string keyword);

        /// <summary>
        /// 根据associateId获取贴子
        /// </summary>
        /// <param name="associateId">关联Id(活动,投票)</param>
        /// <param name="threadType">贴子类型</param>
        /// <returns></returns>
        Thread GetByAssociateId(long associateId, ThreadType threadType);

        /// <summary>
        /// 管理员获取贴子分页列表（不根据后台设置的审核状态过滤）
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴子分类Id</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <param name="auditStatus">审核状态</param>
        /// <param name="StartDate">发贴时间起始时间</param>
        /// <param name="EndDate">发贴时间截止时间</param>
        /// <param name="threadType">贴子类型</param>
        /// <param name="tenantTypeId">租户Id</param>
        /// <returns></returns>
        PagingDataSet<Thread> GetsForAdmin(long? sectionId = null, long? ownerId = null, long? categoryId = null, int pageSize = 20, int pageIndex = 1, string keyword = "", AuditStatus? auditStatus = null, DateTime? StartDate = null, DateTime? EndDate = null, ThreadType? threadType = null, string tenantTypeId = "");

        /// <summary>
        /// 贴子计数获取(后台用)
        /// </summary>
        /// <param name="approvalStatus">审核状态</param>
        /// <param name="is24Hours">是否24小时之内</param>
        /// <returns></returns>
        int GetThreadCount(AuditStatus? approvalStatus, bool is24Hours);

        /// <summary>
        /// 获取贴子计数（前台用）
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="categoryId"></param>
        /// <returns></returns>
        int GetThreadCount(long? sectionId = null, long? categoryId = null);

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        void UpdateThreadAuthor(long userId, string displayName);

        /// <summary>
        ///  删除用户发布的贴子
        /// </summary>
        /// <remarks>
        /// 供用户删除时处理用户相关信息时调用
        /// </remarks>
        /// <param name="userId">UserId</param>
        /// <returns></returns>
        int DeleteUserThreads(long userId);
    }
}