﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Caching;
using Tunynet.Common;
using Tunynet.Repositories;
using Tunynet.Utilities;

namespace Tunynet.Post
{
    /// <summary>
    ///贴吧Repository
    /// </summary>
    public class SectionRepository : Repository<Section>, ISectionRepository
    {
        private CategoryService categoryService = DIContainer.Resolve<CategoryService>();

        /// <summary>
        /// 获取拥有者的贴吧列表
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="userId">吧主UserId</param>
        /// <returns>贴吧列表</returns>
        public IEnumerable<Section> GetsByUserId(string tenantTypeId, long userId)
        {
            List<long> sectionIds = new List<long>();
            var sql = Sql.Builder;
            sql.Select("SectionId")
               .From("tn_Sections")
               .Where("UserId=@0", userId);

            if (!string.IsNullOrEmpty(tenantTypeId))
                sql.Where("TenantTypeId=@0", tenantTypeId);

            sectionIds = CreateDAO().Fetch<long>(sql);

            return PopulateEntitiesByEntityIds<long>(sectionIds);
        }
      
        /// <summary>
        /// 获取贴吧分页集合
        /// </summary>
        /// <param name="tenantTypeId">租户id</param>
        /// <param name="categoryId">分类id</param>
        /// <param name="userId">吧主id</param>
        /// <param name="isEnabled">是否启用（前台只显示启用贴吧）</param>
        /// <param name="keyword">搜索关键词（名称或描述）</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public PagingDataSet<Section> Gets(string tenantTypeId, long? categoryId = null, long? userId = null, bool? isEnabled = null, string keyword = "", int pageSize = 20, int pageIndex = 1)
        {
            var sql = Sql.Builder;
            sql.Select("tn_Sections.*")
            .From("tn_Sections");

            if (categoryId.HasValue && categoryId.Value > 0)
            {
                IEnumerable<Category> categories = categoryService.GetCategoriesOfDescendants(categoryId.Value);
                List<long> categoryIds = new List<long> { categoryId.Value };
                if (categories != null && categories.Count() > 0)
                    categoryIds.AddRange(categories.Select(n => n.CategoryId));
                sql.InnerJoin("tn_ItemsInCategories")
               .On("tn_Sections.SectionId = tn_ItemsInCategories.ItemId")
               .Where("tn_ItemsInCategories.CategoryId in(@categoryIds)", new { categoryIds = categoryIds });
            }

            sql.Where("TenantTypeId = @0", tenantTypeId);

            if (userId.HasValue && userId.Value > 0)
                sql.Where("UserId = @0", userId.Value);

            if (isEnabled != null)
                sql.Where("IsEnabled = @0", isEnabled.Value);

            if (!string.IsNullOrEmpty(keyword))
                sql.Where("Name like @0 or Description like @0 ", "%" + StringUtility.StripSQLInjection(keyword) + "%");

            sql.OrderBy("DisplayOrder,SectionId desc");

            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        ///  删除用户创建的贴吧
        /// </summary>
        /// <remarks>
        /// 供用户删除时处理用户相关信息时调用
        /// </remarks>
        /// <param name="userId">UserId</param>
        /// <returns></returns>
        public int DeleteUserSections(long userId)
        {
            var sql = Sql.Builder;

            sql.Append("DELETE FROM tn_Sections");

            sql.Where("UserId=@0", userId);
            int rows = CreateDAO().Execute(sql);

            RealTimeCacheHelper.IncreaseGlobalVersion();

            return rows;
        }

        #region CacheKey

        /// <summary>
        /// 获取贴吧缓存Key
        /// </summary>
        /// <param name="ownerId"></param>
        /// <param name="tenantTypeId"></param>
        /// <returns></returns>
        private string GetCacheKey_BarSection(long ownerId, string tenantTypeId)
        {
            return string.Format("BarSection::O-{0}:T-{1}", ownerId, tenantTypeId);
        }

        #endregion CacheKey
    }
}