﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Caching;
using Tunynet.Common;
using Tunynet.Repositories;
using Tunynet.Settings;
using Tunynet.Utilities;

namespace Tunynet.Post
{
    /// <summary>
    /// 贴子仓储
    /// </summary>
    public class ThreadRepository : Repository<Thread>, IThreadRepository
    {
        private IBodyProcessor barBodyProcessor = DIContainer.ResolveNamed<IBodyProcessor>(TenantTypeIds.Instance().Section());
        private ISettingsManager<SiteSettings> siteSettings = DIContainer.Resolve<ISettingsManager<SiteSettings>>();

        /// <summary>
        /// 更新贴子
        /// </summary>
        /// <param name="entity"></param>
        public override void Update(Thread entity)
        {
            base.Update(entity);
            //更新解析正文缓存
            string cacheKey = GetCacheKeyOfResolvedBody(entity.ThreadId);
            string resolveBody = cacheService.Get<string>(cacheKey);
            if (resolveBody != null)
            {
                resolveBody = entity.GetBody();

                resolveBody = barBodyProcessor.Process(resolveBody, TenantTypeIds.Instance().Thread(), entity.ThreadId, entity.UserId);
                cacheService.Set(cacheKey, resolveBody, CachingExpirationType.SingleObject);
            }
        }

        /// <summary>
        /// 获取某个贴吧下的所有贴子（用于删除贴子）
        /// </summary>
        /// <param name="sectionId"></param>
        /// <returns></returns>
        public IEnumerable<Thread> GetAllThreadsOfSection(long sectionId)
        {
            var sql = Sql.Builder;
            sql.Select("*")
            .From("tn_Threads")
            .Where("SectionId=@0", sectionId);
            IEnumerable<Thread> threads = CreateDAO().Fetch<Thread>(sql);
            return threads;
        }

        /// <summary>
        /// 获取解析的正文
        /// </summary>
        /// <param name="threadId"></param>
        /// <returns></returns>
        public string GetResolvedBody(long threadId)
        {
            Thread barThread = Get(threadId);
            if (barThread == null)
                return string.Empty;

            string cacheKey = GetCacheKeyOfResolvedBody(threadId);
            string resolveBody = cacheService.Get<string>(cacheKey);
            if (resolveBody == null)
            {
                resolveBody = barThread.GetBody();
                resolveBody = barBodyProcessor.Process(resolveBody, TenantTypeIds.Instance().Thread(), barThread.ThreadId, barThread.UserId);
                cacheService.Set(cacheKey, resolveBody, CachingExpirationType.SingleObject);
            }
            return resolveBody;
        }

        /// <summary>
        /// 获取BarThread内容
        /// </summary>
        /// <param name="threadId"></param>
        /// <returns></returns>
        public string GetBody(long threadId)
        {
            string cacheKey = RealTimeCacheHelper.GetCacheKeyOfEntityBody(threadId);
            string body = cacheService.Get<string>(cacheKey);
            if (body == null)
            {
                Thread barThread = CreateDAO().SingleOrDefault<Thread>("Where ThreadId = @0", threadId);
                body = barThread?.Body ?? string.Empty;
                cacheService.Set(cacheKey, body, CachingExpirationType.SingleObject);
            }
            return body;
        }

        /// <summary>
        /// 获取用户的主题贴分页集合
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="ignoreAudit">是否忽略审核状态（作者或管理员查看时忽略审核状态）</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> GetUserThreads(string tenantTypeId, long userId, bool ignoreAudit, int pageSize, int pageIndex, long? sectionId)
        {
            //不必筛选审核状态
            //缓存周期：对象集合，需要维护即时性
            //排序：发布时间（倒序）
            var sql = Sql.Builder;
            sql.Select("ThreadId")
            .From("tn_Threads")
            .Where("UserId = @0", userId)
            .Where("TenantTypeId = @0", tenantTypeId);
            if (sectionId.HasValue)
                sql.Where("SectionId=@0", sectionId.Value);
            //过滤审核状态
            if (!ignoreAudit)
                sql = AuditSqls(sql);
            sql.OrderBy("ThreadId desc");
            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        /// 获取主题贴的排行数据
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="topNumber">前多少条</param>
        /// <param name="sortBy">主题贴排序依据</param>
        /// <returns></returns>
        public IEnumerable<Thread> GetTops(string tenantTypeId, int topNumber, SortBy_BarThread sortBy)
        {
            //只获取可对外显示审核状态的主题贴
            //缓存周期：常用对象集合，不用维护即时性

            IEnumerable<Thread> threads = null;
            var cacheKey = GetCacheKeyOfTopThreads(tenantTypeId, topNumber, sortBy);
            if (cacheService.TryGetValue(cacheKey, out threads))
            {
                var sql = Sql.Builder;
                var whereSql = Sql.Builder;
                var orderSql = Sql.Builder;
                sql.Select("tn_Threads.*")
                .From("tn_Threads");
                whereSql.Where("tn_Threads.TenantTypeId = @0", tenantTypeId);

                //审核
                whereSql = AuditSqls(whereSql);

                switch (sortBy)
                {
                    case SortBy_BarThread.DateCreated_Desc:
                        orderSql.OrderBy("ThreadId desc");
                        break;

                    case SortBy_BarThread.LastModified_Desc:
                        orderSql.OrderBy("LastModified desc");
                        break;

                    case SortBy_BarThread.HitTimes:
                        sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", CountTypes.Instance().HitTimes()))
                        .On("ThreadId = c.ObjectId");
                        orderSql.OrderBy("c.StatisticsCount desc");
                        break;

                    case SortBy_BarThread.StageHitTimes:
                        StageCountTypeManager stageCountTypeManager = StageCountTypeManager.Instance(TenantTypeIds.Instance().Thread());
                        int stageCountDays = stageCountTypeManager.GetMaxDayCount(CountTypes.Instance().HitTimes());
                        string stageCountType = stageCountTypeManager.GetStageCountType(CountTypes.Instance().HitTimes(), stageCountDays);
                        sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", stageCountType))
                        .On("ThreadId = c.ObjectId");
                        orderSql.OrderBy("c.StatisticsCount desc");
                        break;

                    default:
                        orderSql.OrderBy("ThreadId desc");
                        break;
                }
                sql.Append(whereSql).Append(orderSql);
                threads = GetTopEntities(topNumber, sql);
                cacheService.Set(cacheKey, threads, CachingExpirationType.UsualObjectCollection);
            }

            return threads;
        }

        /// <summary>
        /// 根据标签名获取主题贴排行分页集合
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="tagName">标签名</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> Gets(string tenantTypeId, string tagName, SortBy_BarThread sortBy, int pageIndex, int pageSize)
        {
            //只获取可对外显示审核状态的主题贴
            //缓存周期：对象集合，不用维护即时性
            var sql = Sql.Builder;
            sql.Select("tn_Threads.*")
            .From("tn_Threads");
            var whereSql = Sql.Builder;
            var orderSql = Sql.Builder;

            whereSql.Where("tn_Threads.TenantTypeId = @0", tenantTypeId);
            if (!string.IsNullOrEmpty(tagName))
            {
                sql.InnerJoin("tn_ItemsInTags")
                .On("tn_Threads.ThreadId = tn_ItemsInTags.ItemId");

                whereSql.Where("tn_ItemsInTags.TagName=@0", tagName)
                .Where("tn_ItemsInTags.TenantTypeId = @0", TenantTypeIds.Instance().Thread());
            }

            //审核
            whereSql = AuditSqls(whereSql);

            CountService countService = new CountService(TenantTypeIds.Instance().Thread());

            switch (sortBy)
            {
                case SortBy_BarThread.DateCreated_Desc:
                    orderSql.OrderBy("ThreadId desc");
                    break;

                case SortBy_BarThread.LastModified_Desc:
                    orderSql.OrderBy("LastModified desc");
                    break;

                case SortBy_BarThread.HitTimes:
                    sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", CountTypes.Instance().HitTimes()))
                              .On("ThreadId = c.ObjectId");
                    orderSql.OrderBy("c.StatisticsCount desc");
                    break;

                case SortBy_BarThread.StageHitTimes:
                    StageCountTypeManager stageCountTypeManager = StageCountTypeManager.Instance(TenantTypeIds.Instance().Thread());
                    int stageCountDays = stageCountTypeManager.GetMaxDayCount(CountTypes.Instance().HitTimes());
                    string stageCountType = stageCountTypeManager.GetStageCountType(CountTypes.Instance().HitTimes(), stageCountDays);
                    sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", stageCountType))
                    .On("ThreadId = c.ObjectId");
                    orderSql.OrderBy("c.StatisticsCount desc");
                    break;

                default:
                    orderSql.OrderBy("ThreadId desc");
                    break;
            }
            sql.Append(whereSql).Append(orderSql);
            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        /// 根据贴吧获取主题贴分页集合
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴吧分类Id</param>
        /// <param name="orderBySticky">是否按置顶排序</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="BarDate">贴子显示时间范围</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> Gets(long? sectionId, long? ownerId, long? categoryId, bool? orderBySticky, SortBy_BarThread sortBy, int pageSize, int pageIndex, SortBy_BarDateThread BarDate, string keyword)
        {
            //只获取可对外显示审核状态的主题贴
            //缓存周期：对象集合，需要维护即时性
            var sql = Sql.Builder;
            var whereSql = Sql.Builder;
            var orderSql = Sql.Builder;
            sql.Select("tn_Threads.*")
            .From("tn_Threads");

            if (sectionId.HasValue && sectionId > 0)
            {
                whereSql.Where("SectionId = @0", sectionId);
            }
            if (ownerId.HasValue)
            {
                if (ownerId.Value == -1)
                {
                    whereSql.Where("ownerId != 0");
                }
                else
                {
                    whereSql.Where("ownerId = @0", ownerId.Value);
                }
            }

            if (!string.IsNullOrEmpty(keyword))
                whereSql.Where("Subject like @0 or Author like @0", "%" + StringUtility.StripSQLInjection(keyword) + "%");

            //审核
            whereSql = AuditSqls(whereSql);
            if (categoryId.HasValue && categoryId.Value > 0)
            {
                sql.InnerJoin("tn_ItemsInCategories")
                .On("ThreadId = tn_ItemsInCategories.ItemId");
                whereSql.Where("tn_ItemsInCategories.CategoryId=@0", categoryId.Value);
            }

            switch (BarDate)
            {
                case SortBy_BarDateThread.ThreeDay:
                    whereSql.Where("tn_Threads.DateCreated>@0", DateTime.Now.AddDays(-3));
                    break;

                case SortBy_BarDateThread.SevenDay:
                    whereSql.Where("tn_Threads.DateCreated>@0", DateTime.Now.AddDays(-7));
                    break;

                case SortBy_BarDateThread.AMonth:
                    whereSql.Where("tn_Threads.DateCreated>@0", DateTime.Now.AddMonths(-1));
                    break;
            }
            //置顶排序
            if (orderBySticky.HasValue && orderBySticky.Value)
                orderSql.OrderBy("IsSticky desc");

            switch (sortBy)
            {
                case SortBy_BarThread.DateCreated_Desc:
                    orderSql.OrderBy("ThreadId desc");
                    break;

                case SortBy_BarThread.LastModified_Desc:
                    orderSql.OrderBy("LastModified desc");
                    break;

                case SortBy_BarThread.HitTimes:
                    sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", CountTypes.Instance().HitTimes()))
                              .On("ThreadId = c.ObjectId");
                    orderSql.OrderBy("c.StatisticsCount desc");
                    break;

                case SortBy_BarThread.StageHitTimes:
                    StageCountTypeManager stageCountTypeManager = StageCountTypeManager.Instance(TenantTypeIds.Instance().Thread());
                    int stageCountDays = stageCountTypeManager.GetMaxDayCount(CountTypes.Instance().HitTimes());
                    string stageCountType = stageCountTypeManager.GetStageCountType(CountTypes.Instance().HitTimes(), stageCountDays);
                    sql.LeftJoin(string.Format("(select * from tn_Counts WHERE (tn_Counts.CountType = '{0}')) c", stageCountType))
                    .On("ThreadId = c.ObjectId");
                    orderSql.OrderBy("c.StatisticsCount desc");
                    break;

                default:
                    orderSql.OrderBy("ThreadId desc");
                    break;
            }

            sql.Append(whereSql).Append(orderSql);
            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        /// 根据associateId获取贴子
        /// </summary>
        /// <param name="associateId">关联Id(活动,投票)</param>
        /// <param name="threadType">贴子类型</param>
        /// <returns></returns>
        public Thread GetByAssociateId(long associateId, ThreadType threadType)
        {
            var sql = Sql.Builder.Select("*").From("tn_Threads").Where("AssociateId=@0 AND ThreadType=@1", associateId, threadType);

            return CreateDAO().SingleOrDefault<Thread>(sql);
        }

        /// <summary>
        /// 管理员获取贴子分页列表（不根据后台设置的审核状态过滤）
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴子分类Id</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <param name="auditStatus">审核状态</param>
        /// <param name="StartDate">发贴时间起始时间</param>
        /// <param name="EndDate">发贴时间截止时间</param>
        /// <param name="threadType">贴子类型</param>
        /// <param name="tenantTypeId">租户Id</param>
        /// <returns></returns>
        public PagingDataSet<Thread> GetsForAdmin(long? sectionId = null, long? ownerId = null, long? categoryId = null, int pageSize = 20, int pageIndex = 1, string keyword = "", AuditStatus? auditStatus = null, DateTime? StartDate = null, DateTime? EndDate = null, ThreadType? threadType = null, string tenantTypeId = "")
        {
            var sql = Sql.Builder;
            sql.Select("*")
            .From("tn_Threads");

            if (categoryId.HasValue && categoryId.Value > 0)
            {
                sql.InnerJoin("tn_ItemsInCategories")
                .On("ThreadId = tn_ItemsInCategories.ItemId");
                sql.Where("tn_ItemsInCategories.CategoryId=@0", categoryId.Value);
            }
            if (threadType.HasValue)
            {
                sql.Where("ThreadType=@0", threadType.Value);
            }

            if (auditStatus.HasValue)
            {
                sql.Where("ApprovalStatus=@0", auditStatus.Value);
            }

            if (!string.IsNullOrEmpty(tenantTypeId))
                sql.Where("TenantTypeId = @0", tenantTypeId);

            if (sectionId != null && sectionId > 0)
                sql.Where("SectionId = @0", sectionId.Value);

            if (!string.IsNullOrEmpty(keyword))
                sql.Where("Subject like @0 or Author like @0", "%" + StringUtility.StripSQLInjection(keyword) + "%");

            if (StartDate.HasValue)
                sql.Where("DateCreated >= @0", StartDate.Value);
            if (EndDate.HasValue)
                sql.Where("DateCreated < @0", EndDate.Value.AddDays(1));

            sql.OrderBy("ThreadId desc");

            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        /// 贴子计数获取(后台用)
        /// </summary>
        /// <param name="approvalStatus">审核状态</param>
        /// <param name="is24Hours">是否24小时之内</param>
        /// <returns></returns>
        public int GetThreadCount(AuditStatus? approvalStatus, bool is24Hours)
        {
            Sql sql = Sql.Builder;
            sql.Select(" count(tn_Threads.ThreadId )").From("tn_Threads");
            if (approvalStatus.HasValue)
                sql.Where("tn_Threads.ApprovalStatus=@0", (int)approvalStatus.Value);
            if (is24Hours)
                sql.Where("tn_Threads.DateCreated>@0", DateTime.Now.AddHours(-24));
            return CreateDAO().SingleOrDefault<int>(sql);
        }

        /// <summary>
        /// 获取贴子计数（前台用）
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="categoryId"></param>
        /// <returns></returns>
        public int GetThreadCount(long? sectionId = null, long? categoryId = null)
        {
            Sql sql = Sql.Builder;
            sql.Select(" count(tn_Threads.ThreadId )").From("tn_Threads");

            if (categoryId.HasValue && categoryId.Value > 0)
            {
                sql.InnerJoin("tn_ItemsInCategories")
                .On("ThreadId = tn_ItemsInCategories.ItemId");
                sql.Where("tn_ItemsInCategories.CategoryId=@0", categoryId.Value);
            }
            if (sectionId != null && sectionId > 0)
                sql.Where("SectionId = @0", sectionId.Value);

            sql = AuditSqls(sql);

            return CreateDAO().SingleOrDefault<int>(sql);
        }

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        public void UpdateThreadAuthor(long userId, string displayName)
        {
            var sql = Sql.Builder.Append("Update tn_Threads SET Author=@1 WHERE UserId=@0", userId, displayName);
            CreateDAO().Execute(sql);

            RealTimeCacheHelper.IncreaseGlobalVersion();
        }

        /// <summary>
        ///  删除用户发布的贴子
        /// </summary>
        /// <remarks>
        /// 供用户删除时处理用户相关信息时调用
        /// </remarks>
        /// <param name="userId">UserId</param>
        /// <returns></returns>
        public int DeleteUserThreads(long userId)
        {
            var sql = Sql.Builder;

            sql.Append("DELETE FROM tn_Threads");

            sql.Where("UserId=@0", userId);
            int rows = CreateDAO().Execute(sql);

            RealTimeCacheHelper.IncreaseGlobalVersion();

            return rows;
        }

        /// <summary>
        /// 审核语句组装
        /// </summary>
        /// <param name="wheresql">wheresql</param>
        /// <returns></returns>
        private Sql AuditSqls(Sql wheresql)
        {
            var setting = siteSettings.Get();
            if (setting.AuditStatus == PubliclyAuditStatus.Success)
                wheresql.Where("ApprovalStatus=@0", setting.AuditStatus);
            else if (setting.AuditStatus == PubliclyAuditStatus.Again_GreaterThanOrEqual)
                wheresql.Where("ApprovalStatus>@0 ", PubliclyAuditStatus.Pending);
            else if (setting.AuditStatus == PubliclyAuditStatus.Pending_GreaterThanOrEqual)
                wheresql.Where("ApprovalStatus>@0 ", PubliclyAuditStatus.Fail);
            return wheresql;
        }

        /// <summary>
        /// 获取解析正文缓存Key
        /// </summary>
        /// <param name="threadId"></param>
        /// <returns></returns>
        private string GetCacheKeyOfResolvedBody(long threadId)
        {
            return "BarThreadResolvedBody" + threadId;
        }

        /// <summary>
        /// 获取主题帖排行缓存key
        /// </summary>
        /// <param name="tenantTypeId"></param>
        /// <param name="topNumber"></param>
        /// <param name="sortBy"></param>
        /// <returns></returns>
        private string GetCacheKeyOfTopThreads(string tenantTypeId, int topNumber, SortBy_BarThread sortBy)
        {
            return $"TopThreadsCacheKey:tenantTypeId:{tenantTypeId}:topNumber:{topNumber}:sortBy:{(int)sortBy}";
        }
    }
}