﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System.Collections.Generic;
using System.Linq;
using Tunynet.Common;
using Tunynet.Events;

namespace Tunynet.Post
{
    /// <summary>
    /// 贴吧业务逻辑类
    /// </summary>
    public class SectionService
    {
        #region ctor

        private ISectionRepository barSectionRepository;
        private IUserService userService;
        private CategoryService categoryService;
        private CategoryManagerService categoryManagerService;

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="barSectionRepository"></param>
        /// <param name="userService"></param>
        /// <param name="categoryService"></param>
        /// <param name="categoryManagerService"></param>
        public SectionService(ISectionRepository barSectionRepository, IUserService userService, CategoryService categoryService,
            CategoryManagerService categoryManagerService)
        {
            this.barSectionRepository = barSectionRepository;
            this.userService = userService;
            this.categoryService = categoryService;
            this.categoryManagerService = categoryManagerService;
        }

        #endregion

        #region 维护贴吧

        /// <summary>
        /// 创建贴吧
        /// </summary>
        /// <param name="section">贴吧</param>
        /// <param name="managerIds">管理员用户Id</param>
        /// <returns>是否创建成功</returns>
        public bool Create(Section section, IEnumerable<long> managerIds)
        {
            EventBus<Section>.Instance().OnBefore(section, new CommonEventArgs(EventOperationType.Instance().Create()));
            barSectionRepository.Insert(section);
            new AttachmentService(TenantTypeIds.Instance().Section()).ToggleTemporaryAttachments(new List<long>() { section.FeaturedImageAttachmentId }, section.SectionId);
            if (section.SectionId > 0)
            {
                if (managerIds != null && managerIds.Count() > 0)
                {
                    List<long> mangagerIds_list = managerIds.ToList();
                    mangagerIds_list.Remove(section.UserId);
                    managerIds = mangagerIds_list;
                    categoryManagerService.UpdateManagerIds(TenantTypeIds.Instance().Section(), section.SectionId, managerIds);
                }
                section.DisplayOrder = section.SectionId;
                barSectionRepository.Update(section);
                EventBus<Section>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Create()));
                //记录操作日志
                EventBus<Section, CommonEventArgs>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Create()));
            }
            return section.SectionId > 0;
        }

        /// <summary>
        /// 更新贴吧
        /// </summary>
        /// <param name="section">贴吧</param>
        /// <param name="managerIds">管理员用户Id</param>
        public void Update(Section section, IEnumerable<long> managerIds)
        {
            EventBus<Section>.Instance().OnBefore(section, new CommonEventArgs(EventOperationType.Instance().Update()));

            barSectionRepository.Update(section);
            new AttachmentService(TenantTypeIds.Instance().Section()).ToggleTemporaryAttachments(new List<long>() { section.FeaturedImageAttachmentId }, section.SectionId);
            if (managerIds != null && managerIds.Count() > 0)
            {
                List<long> mangagerIds_list = managerIds.ToList();
                mangagerIds_list.Remove(section.UserId);
                managerIds = mangagerIds_list;
            }
            categoryManagerService.UpdateManagerIds(TenantTypeIds.Instance().Section(), section.SectionId, managerIds);

            EventBus<Section>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Update()));
            //记录操作日志
            EventBus<Section, CommonEventArgs>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除贴吧
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        public void Delete(long sectionId)
        {
            Section section = barSectionRepository.Get(sectionId);
            if (section == null)
                return;
            EventBus<Section>.Instance().OnBefore(section, new CommonEventArgs(EventOperationType.Instance().Delete()));
            int affectCount = barSectionRepository.Delete(section);
            if (affectCount > 0)
            {
                EventBus<Section>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Delete()));
                //记录操作日志
                EventBus<Section, CommonEventArgs>.Instance().OnAfter(section, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        #endregion 维护贴吧

        #region 获取贴吧

        /// <summary>
        /// 获取单个贴吧实体
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns>贴吧</returns>
        public Section Get(long sectionId)
        {
            return barSectionRepository.Get(sectionId);
        }

        /// <summary>
        /// 是否为吧主
        /// </summary>
        /// <param name="userId">被验证用户Id</param>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns></returns>
        public bool IsSectionOwner(long userId, long sectionId)
        {
            Section barSection = Get(sectionId);
            if (barSection == null)
                return false;
            return barSection.UserId == userId;
        }

        /// <summary>
        /// 获取贴吧的管理员列表
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns>贴吧</returns>
        public IEnumerable<User> GetSectionManagers(long sectionId)
        {
            bool isReference = false;
            return userService.GetFullUsers(categoryManagerService.GetCategoryManagerIds(TenantTypeIds.Instance().Section(), sectionId, out isReference));
        }

        /// <summary>
        /// 获取拥有者的贴吧列表
        /// </summary>
        /// <param name="tenantTypeId">租户id</param>
        /// <param name="userId">吧主UserId</param>
        /// <returns>贴吧列表</returns>
        public IEnumerable<Section> GetsByUserId(string tenantTypeId, long userId)
        {
            return barSectionRepository.GetsByUserId(tenantTypeId, userId);
        }

        /// <summary>
        /// 获取贴吧分页集合
        /// </summary>
        /// <param name="tenantTypeId">租户id</param>
        /// <param name="categoryId">分类id</param>
        /// <param name="userId">吧主id</param>
        /// <param name="isEnabled">是否启用（前台只显示启用贴吧）</param>
        /// <param name="keyword">搜索关键词（名称或描述）</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public PagingDataSet<Section> Gets(string tenantTypeId, long? categoryId = null, long? userId = null, bool? isEnabled = null, string keyword = "", int pageSize = 20, int pageIndex = 1)
        {
            return barSectionRepository.Gets(tenantTypeId, categoryId, userId, isEnabled, keyword, pageSize, pageIndex);
        }

        /// <summary>
        /// 根据贴吧Id集合组装贴吧实体集合
        /// </summary>
        /// <param name="sectionIds">贴吧Id集合</param>
        /// <returns>贴吧实体集合</returns>
        public IEnumerable<Section> GetBarSections(IEnumerable<long> sectionIds)
        {
            return barSectionRepository.PopulateEntitiesByEntityIds(sectionIds);
        }

        /// <summary>
        ///  删除用户创建的贴吧
        /// </summary>
        /// <remarks>
        /// 供用户删除时处理用户相关信息时调用
        /// </remarks>
        /// <param name="userId">UserId</param>
        /// <returns></returns>
        public int DeleteUserSections(long userId)
        {
            return barSectionRepository.DeleteUserSections(userId);
        }

        #endregion 获取贴吧
    }
}