﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using Newtonsoft.Json;
using PetaPoco;
using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Caching;
using Tunynet.Common;
using Tunynet.Utilities;

namespace Tunynet.Post
{
    /// <summary>
    /// 贴子
    /// </summary>
    [Serializable]
    [TableName("tn_Threads")]
    [PrimaryKey("ThreadId", autoIncrement = true)]
    [CacheSetting(true, PropertyNameOfBody = "Body")]
    public class Thread : SerializablePropertiesBase, IEntity, IAuditable
    {
        /// <summary>
        /// 新建实体时使用
        /// </summary>
        public static Thread New()
        {
            Thread barThread = new Thread()
            {
                Author = string.Empty,
                Subject = string.Empty,
                IP = WebUtility.GetIP(),
                DateCreated = DateTime.Now,
                LastModified = DateTime.Now
            };
            return barThread;
        }

        #region 需持久化属性

        /// <summary>
        ///ThreadId
        /// </summary>
        public long ThreadId { get; set; }

        /// <summary>
        ///所属贴吧Id
        /// </summary>
        public long SectionId { get; set; }

        /// <summary>
        /// 租户类型Id
        /// </summary>
        public string TenantTypeId { get; set; }

        /// <summary>
        ///所属贴吧拥有者Id（例如：群组Id）
        /// </summary>
        public long OwnerId { get; set; }

        /// <summary>
        ///主题作者用户Id
        /// </summary>
        public long UserId { get; set; }

        /// <summary>
        ///主题作者
        /// </summary>
        public string Author { get; set; }

        /// <summary>
        ///贴子标题
        /// </summary>
        public string Subject { get; set; }

        /// <summary>
        ///贴子内容
        /// </summary>
        public string Body { get; set; }

        /// <summary>
        ///是否锁定
        /// </summary>
        public bool IsLocked { get; set; }

        /// <summary>
        ///是否置顶
        /// </summary>
        public bool IsSticky { get; set; }

        /// <summary>
        ///审批状态
        /// </summary>
        public AuditStatus ApprovalStatus { get; set; }

        /// <summary>
        ///发贴人IP
        /// </summary>
        public string IP { get; set; }

        /// <summary>
        ///创建时间
        /// </summary>
        public DateTime DateCreated { get; set; }

        /// <summary>
        ///最后更新日期（被回复时也需要更新时间）
        /// </summary>
        public DateTime LastModified { get; set; }

        /// <summary>
        /// 贴子类型
        /// </summary>
        public ThreadType ThreadType { get; set; }

        /// <summary>
        /// 关联 Id(活动,投票)
        /// </summary>
        public long AssociateId { get; set; }

        #endregion 需持久化属性

        #region IAuditable 实现

        /// <summary>
        /// 审核项Key
        /// </summary>
        public string AuditItemKey
        {
            get { return AuditItemKeys.Instance().Post(); }
        }

        #endregion IAuditable 实现

        #region IEntity 成员

        object IEntity.EntityId { get { return this.ThreadId; } }

        bool IEntity.IsDeletedInDatabase { get; set; }

        #endregion IEntity 成员

        #region 序列化属性

        /// <summary>
        /// 是否匿名评论
        /// </summary>
        [Ignore]
        public bool IsAnonymous
        {
            get { return GetExtendedProperty<bool>("IsAnonymous"); }
            set { SetExtendedProperty("IsAnonymous", value); }
        }

        /// <summary>
        /// 是否允许手机端编辑
        /// </summary>
        [Ignore]
        public bool IsAllowMobileEdit
        {
            get
            {
                return GetExtendedProperty<bool>("IsAllowMobileEdit");
            }
            set
            {
                SetExtendedProperty("IsAllowMobileEdit", value);
            }
        }

        #endregion 序列化属性

        #region 扩展属性

        /// <summary>
        /// 所属贴吧
        /// </summary>
        [Ignore]
        [JsonIgnore]
        public Section BarSection
        {
            get
            {
                SectionService sectionService = DIContainer.Resolve<SectionService>();
                return sectionService.Get(this.SectionId);
            }
        }

        /// <summary>
        /// 所属本贴子的
        /// </summary>
        public IEnumerable<Attachment> Attachments(string tenantTypeId = "")
        {
            return new AttachmentService(string.IsNullOrEmpty(tenantTypeId) ? TenantTypeIds.Instance().Thread() : tenantTypeId).GetsByAssociateId(tenantTypeId == TenantTypeIds.Instance().Thread() ? this.ThreadId : this.AssociateId) ?? new List<Attachment>();
        }

        /// <summary>
        /// 贴子所属分类
        /// </summary>
        [Ignore]
        [JsonIgnore]
        public Category ThreadCategory
        {
            get
            {
                CategoryService categoryService = DIContainer.Resolve<CategoryService>();
                var category = categoryService.GetItems(this.ThreadId, this.TenantTypeId);
                if (category != null)
                {
                    return categoryService.Get(category.CategoryId);
                }
                return null;
            }
        }

        #endregion 扩展属性

        #region 提交表单扩展

        /// <summary>
        /// 最终的附件Id集合
        /// </summary>
        [Ignore]
        public IEnumerable<long> AttachmentIdsFinal { get; set; }

        #endregion 提交表单扩展

        #region 计数

        /// <summary>
        /// 评论数
        /// </summary>
        [Ignore]
        public int CommentCount
        {
            get
            {
                CountService countService = new CountService(TenantTypeIds.Instance().Thread());
                return countService.Get(CountTypes.Instance().CommentCount(), this.ThreadId);
            }
        }

        /// <summary>
        /// 浏览数
        /// </summary>
        [Ignore]
        public int HitTimes
        {
            get
            {
                CountService countService = new CountService(TenantTypeIds.Instance().Thread());
                return countService.Get(CountTypes.Instance().HitTimes(), this.ThreadId);
            }
        }

        #endregion 计数

        #region 方法

        /// <summary>
        /// 获取BarThread的Body
        /// </summary>
        /// <remarks>
        /// 由于使用独立的实体内容缓存，Body属性已经置为null
        /// </remarks>
        /// <returns></returns>
        public string GetBody()
        {
            return new ThreadRepository().GetBody(this.ThreadId);
        }

        /// <summary>
        /// 获取Thread的解析过的内容(在web呈现)
        /// </summary>
        public string GetResolvedBody()
        {
            return new ThreadRepository().GetResolvedBody(this.ThreadId);
        }

        /// <summary>
        /// 获取手机端BarThread的Body
        /// </summary>
        /// <returns>(返回的img添加对应的标识符，用于预览)</returns>
        public string GetMobileBody()
        {
            return new ThreadRepository().GetBody(this.ThreadId).Replace("<img ", "<img  data-preview-src=\"\" data-preview-group=\"1\" ").Replace("src=\"/img/Emotions", "src=\"../../img/Emotions");
        }

        /// <summary>
        /// 最后回贴人ID
        /// </summary>
        [Ignore]
        public long LastModifiedUserId
        {
            get
            {
                CommentService commentService = DIContainer.Resolve<CommentService>();
                var lastcomment = commentService.GetObjectComments(TenantTypeIds.Instance().Thread(), this.ThreadId, 10, 1, SortBy_Comment.DateCreatedDesc).FirstOrDefault();
                if (lastcomment != null)
                {
                    return lastcomment.UserId;
                }
                return 0;
            }
        }

        #endregion 方法
    }
}