﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Common;
using Tunynet.Events;
using Tunynet.Settings;

namespace Tunynet.Post
{
    /// <summary>
    /// 主题贴业务逻辑类
    /// </summary>
    public class ThreadService
    {
        #region ctor

        private IUserService userService;
        private IThreadRepository barThreadRepository;
        private SiteSettings siteSetting;
        private CommentService commentService;
        private SectionService barSectionService;
        private CategoryService categoryService;
        private AuditService auditService;

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="barThreadRepository"></param>
        /// <param name="commentService"></param>
        /// <param name="userService"></param>
        /// <param name="barSectionService"></param>
        /// <param name="categoryService"></param>
        /// <param name="auditService"></param>
        /// <param name="siteSettings"></param>
        public ThreadService(IThreadRepository barThreadRepository, CommentService commentService, IUserService userService, SectionService barSectionService, CategoryService categoryService, AuditService auditService, ISettingsManager<SiteSettings> siteSettings)
        {
            this.barThreadRepository = barThreadRepository;
            this.commentService = commentService;
            this.userService = userService;
            this.barSectionService = barSectionService;
            this.categoryService = categoryService;
            this.auditService = auditService;
            this.siteSetting = siteSettings.Get();
        }

        #endregion ctor

        #region 维护主题贴

        /// <summary>
        /// 创建主题贴
        /// </summary>
        /// <param name="thread">主题贴</param>
        /// <param name="isMobile">是否手机端</param>
        /// <param name="isManager">是否有权限管理</param>
        public bool Create(Thread thread, bool isManager = false, bool isMobile = false)
        {
            //防灌水设置
            if (!isManager)
            {
                if (!userService.CanCreateNewContent(thread.UserId, TenantTypeIds.Instance().Thread()))
                    return false;
                //设置上次发布时间
                userService.SetNewContentLastCreated(thread.UserId, TenantTypeIds.Instance().Thread(), thread.DateCreated);
            }
            EventBus<Thread>.Instance().OnBefore(thread, new CommonEventArgs(EventOperationType.Instance().Create()));
            auditService.ChangeAuditStatusForCreate(thread.UserId, thread, isManager);

            barThreadRepository.Insert(thread);

            new AttachmentService(TenantTypeIds.Instance().Thread()).ToggleTemporaryAttachments(thread.AttachmentIdsFinal, thread.ThreadId);

            if (thread.ThreadId > 0)
            {
                //执行事件
                EventBus<Thread>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Create()));
                //发布通过审核的贴子增加积分
                EventBus<Thread, AuditEventArgs>.Instance().OnAfter(thread, new AuditEventArgs(null, thread.ApprovalStatus, EventOperationType.Instance().Create()));
                //记录操作日志
                EventBus<Thread, CommonEventArgs>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Create()));
                EventBus<Thread, AttachmentEventArgs>.Instance().OnAfter(thread, new AttachmentEventArgs(EventOperationType.Instance().Create(), thread.TenantTypeId, isMobile));
            }
            return thread.ThreadId > 0;
        }

        /// <summary>
        /// 更新主题贴
        /// </summary>
        /// <param name="thread">主题贴</param>
        /// <param name="operatorUserId">当前操作人</param>
        /// <param name="isManager">是否有权限管理</param>
        /// <param name="isNeedToggleTemporaryAttachment">是否需要处理附件</param>
        public void Update(Thread thread, long operatorUserId, bool isManager = false, bool isNeedToggleTemporaryAttachment = true)
        {
            EventBus<Thread>.Instance().OnBefore(thread, new CommonEventArgs(EventOperationType.Instance().Update()));
            AuditStatus oldAuditStatus = thread.ApprovalStatus;

            auditService.ChangeAuditStatusForUpdate(thread.UserId, thread, isManager);

            barThreadRepository.Update(thread);

            if (isNeedToggleTemporaryAttachment)
                new AttachmentService(TenantTypeIds.Instance().Thread()).ToggleTemporaryAttachments(thread.AttachmentIdsFinal, thread.ThreadId);

            EventBus<Thread>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Update()));
            //积分
            EventBus<Thread, AuditEventArgs>.Instance().OnAfter(thread, new AuditEventArgs(oldAuditStatus, thread.ApprovalStatus, EventOperationType.Instance().Update()));
            //记录操作日志
            EventBus<Thread, CommonEventArgs>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 设置置顶
        /// </summary>
        /// <param name="threadId">待操作的主题贴Id</param>
        /// <param name="isSticky">待更新至的置顶状态</param>
        public bool SetSticky(long threadId, bool isSticky)
        {
            Thread thread = barThreadRepository.Get(threadId);
            if (thread == null)
                return false;
            if (thread.IsSticky != isSticky)
            {
                thread.IsSticky = isSticky;
                barThreadRepository.Update(thread);
            }
            return true;
        }

        /// <summary>
        /// 更新审核状态
        /// </summary>
        /// <param name="threadId">待被更新的主题贴Id</param>
        /// <param name="isApproved">是否通过审核</param>
        public void UpdateAuditStatus(long threadId, bool isApproved)
        {
            Thread thread = barThreadRepository.Get(threadId);
            AuditStatus auditStatus = isApproved ? AuditStatus.Success : AuditStatus.Fail;
            if (thread.ApprovalStatus == auditStatus)
                return;
            AuditStatus oldAuditStatus = thread.ApprovalStatus;
            thread.ApprovalStatus = auditStatus;
            barThreadRepository.Update(thread);

            EventBus<Thread>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Update()));
            //待审核贴子通过审核增加积分
            EventBus<Thread, AuditEventArgs>.Instance().OnAfter(thread, new AuditEventArgs(oldAuditStatus, auditStatus, isApproved ? EventOperationType.Instance().Approved() : EventOperationType.Instance().Disapproved()));
            //记录操作日志
            EventBus<Thread, CommonEventArgs>.Instance().OnAfter(thread, new CommonEventArgs(isApproved ? EventOperationType.Instance().Approved() : EventOperationType.Instance().Disapproved()));
        }

        /// <summary>
        /// 删除主题贴
        /// </summary>
        /// <param name="threadId">主题贴Id</param>
        public void Delete(long threadId)
        {
            Thread thread = barThreadRepository.Get(threadId);
            if (thread == null)
                return;
            EventBus<Thread>.Instance().OnBefore(thread, new CommonEventArgs(EventOperationType.Instance().Delete()));
            int affectCount = barThreadRepository.Delete(thread);
            if (affectCount > 0)
            {
                EventBus<Thread>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Delete()));
                //积分处理
                EventBus<Thread, AttachmentEventArgs>.Instance().OnAfter(thread, new AttachmentEventArgs(EventOperationType.Instance().Delete(), TenantTypeIds.Instance().Thread()));
                //相关联删除处理
                EventBus<Thread, AuditEventArgs>.Instance().OnAfter(thread, new AuditEventArgs(thread.ApprovalStatus, null, EventOperationType.Instance().Delete()));
                //记录操作日志
                EventBus<Thread, CommonEventArgs>.Instance().OnAfter(thread, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 删除贴吧下的所有主题贴
        /// </summary>
        /// <param name="sectionId"></param>
        public void DeletesBySectionId(long sectionId)
        {
            IEnumerable<Thread> barThreads = barThreadRepository.GetAllThreadsOfSection(sectionId);
            foreach (var barThread in barThreads)
            {
                Delete(barThread.ThreadId);
            }
        }

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        public void UpdateThreadAuthor(long userId, string displayName)
        {
            barThreadRepository.UpdateThreadAuthor(userId, displayName);
        }

        /// <summary>
        ///  删除用户发布的贴子
        /// </summary>
        /// <remarks>
        /// 供用户删除时处理用户相关信息时调用
        /// </remarks>
        /// <param name="userId">UserId</param>
        /// <returns></returns>
        public int DeleteUserThreads(long userId)
        {
            return barThreadRepository.DeleteUserThreads(userId);
        }

        #endregion 维护主题贴

        #region 获取主题贴

        /// <summary>
        /// 获取单个主题贴实体
        /// </summary>
        /// <param name="threadId">主题贴Id</param>
        /// <returns>主题贴</returns>
        public Thread Get(long threadId)
        {
            return barThreadRepository.Get(threadId);
        }

        /// <summary>
        /// 获取主题贴内容
        /// </summary>
        /// <param name="threadId">主题贴Id</param>
        /// <returns>主题贴内容</returns>
        public string GetBody(long threadId)
        {
            return barThreadRepository.GetBody(threadId);
        }

        /// <summary>
        /// 获取排序的列表ID集合去获取贴子实体
        /// </summary>
        /// <param name="threadIds">主题贴Id集合</param>
        /// <returns>主题贴</returns>
        public IEnumerable<Thread> Gets(IEnumerable<long> threadIds)
        {
            var threads = barThreadRepository.PopulateEntitiesByEntityIds(threadIds);
            //根据站点设置过滤审核状态
            return threads.Where(n => (int)n.ApprovalStatus >= (int)siteSetting.AuditStatus).ToList();
        }

        /// <summary>
        /// 获取用户的主题贴分页集合
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="ignoreAudit">是否忽略审核状态（比如查看自己的用户空间时就需要忽略审核状态，查看别人用户空间则不忽略）</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="sectionId">贴吧Id</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> GetUserThreads(string tenantTypeId, long userId, bool ignoreAudit, int pageSize = 10, int pageIndex = 1, long? sectionId = null)
        {
            //不必筛选审核状态
            //缓存周期：对象集合，需要维护即时性
            //排序：发布时间（倒序）
            return barThreadRepository.GetUserThreads(tenantTypeId, userId, ignoreAudit, pageSize, pageIndex, sectionId);
        }

        /// <summary>
        /// 获取主题贴的排行数据
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="topNumber">前多少条</param>
        /// <param name="sortBy">主题贴排序依据</param>
        /// <remarks>
        /// 只获取可对外显示审核状态的主题贴
        /// 缓存周期：常用对象集合，不用维护即时性
        /// </remarks>
        /// <returns></returns>
        public IEnumerable<Thread> GetTops(string tenantTypeId, int topNumber, SortBy_BarThread sortBy = SortBy_BarThread.StageHitTimes)
        {
            return barThreadRepository.GetTops(tenantTypeId, topNumber, sortBy);
        }

        /// <summary>
        /// 根据标签名获取主题贴排行分页集合
        /// </summary>
        /// <param name="tenantTypeId">租户类型Id</param>
        /// <param name="tagName">标签名</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">分页大小</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> GetsByTagName(string tenantTypeId, string tagName, SortBy_BarThread sortBy = SortBy_BarThread.StageHitTimes, int pageIndex = 1, int pageSize = 10)
        {
            return barThreadRepository.Gets(tenantTypeId, tagName, sortBy, pageIndex, pageSize);
        }

        /// <summary>
        /// 根据贴吧获取主题贴分页集合
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴吧分类Id</param>
        /// <param name="orderBySticky">是否按置顶排序</param>
        /// <param name="sortBy">贴子排序依据</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="BarDate">贴子显示时间范围</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <returns>主题贴列表</returns>
        public PagingDataSet<Thread> Gets(long? sectionId, long? ownerId = null, long? categoryId = null, bool? orderBySticky = null, SortBy_BarThread sortBy = SortBy_BarThread.LastModified_Desc, int pageSize = 20, int pageIndex = 1, SortBy_BarDateThread BarDate = SortBy_BarDateThread.All, string keyword = "")
        {
            return barThreadRepository.Gets(sectionId, ownerId, categoryId, orderBySticky, sortBy, pageSize, pageIndex, BarDate, keyword);
        }

        /// <summary>
        /// 根据associateId获取贴子（获取活动或投票关联的贴子时使用）
        /// </summary>
        /// <param name="associateId">关联Id(活动,投票)</param>
        /// <param name="threadType">贴子类型</param>
        /// <returns></returns>
        public Thread GetByAssociateId(long associateId, ThreadType threadType)
        {
            return barThreadRepository.GetByAssociateId(associateId, threadType);
        }

        /// <summary>
        /// 贴子计数获取(后台用)
        /// </summary>
        /// <param name="approvalStatus">审核状态</param>
        /// <param name="is24Hours">是否24小时之内</param>
        /// <returns></returns>
        public int GetThreadCount(AuditStatus? approvalStatus = null, bool is24Hours = false)
        {
            return barThreadRepository.GetThreadCount(approvalStatus, is24Hours);
        }

        /// <summary>
        /// 获取贴子计数（前台用）
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="categoryId"></param>
        /// <returns></returns>
        public int GetThreadCount(long? sectionId = null, long? categoryId = null)
        {
            return barThreadRepository.GetThreadCount(sectionId, categoryId);
        }

        /// <summary>
        /// 管理员获取贴子分页列表（不根据后台设置的审核状态过滤）
        /// </summary>
        /// <param name="sectionId">贴吧Id</param>
        /// <param name="ownerId">所属贴吧拥有者Id</param>
        /// <param name="categoryId">贴子分类Id</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="keyword">标题或作者关键字</param>
        /// <param name="auditStatus">审核状态</param>
        /// <param name="startDate">发贴时间起始时间</param>
        /// <param name="endDate">发贴时间截止时间</param>
        /// <param name="threadType">贴子类型</param>
        /// <param name="tenantTypeId">租户Id</param>
        /// <returns></returns>
        public PagingDataSet<Thread> GetsForAdmin(long? sectionId = null, long? ownerId = null, long? categoryId = null, int pageSize = 20, int pageIndex = 1, string keyword = "", AuditStatus? auditStatus = null, DateTime? startDate = null, DateTime? endDate = null, ThreadType? threadType = null, string tenantTypeId = "")
        {
            return barThreadRepository.GetsForAdmin(sectionId, ownerId, categoryId, pageSize, pageIndex, keyword, auditStatus, startDate, endDate, threadType, tenantTypeId);
        }

        #endregion 获取主题贴
    }
}