﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Repositories;

namespace Tunynet.Common.Repositories
{
    /// <summary>
    /// 账户绑定数据仓储
    /// </summary>
    public class AccountBindingRepository : Repository<AccountBinding>, IAccountBindingRepository
    {
        /// <summary>
        /// 创建第三方帐号绑定
        /// </summary>
        /// <param name="account"></param>
        public void CreateAccountBinding(AccountBinding account)
        {
            //设计说明:
            //插入前，需要检查UserId+AccountTypeKey唯一
            Sql sql = Sql.Builder;
            sql.Select("*")
                .From("tn_AccountBindings")
                .Where("UserId = @0 and AccountTypeKey = @1", account.UserId, account.AccountTypeKey);
            AccountBinding localAccountBinding = CreateDAO().FirstOrDefault<AccountBinding>(sql);

            Sql.Builder.Select("*")
               .From("tn_AccountBindings")
               .Where("Identification = @0 and AccountTypeKey = @1", account.Identification, account.AccountTypeKey);
            AccountBinding remoteAccountBinding = CreateDAO().FirstOrDefault<AccountBinding>(sql);

            if (localAccountBinding == null && remoteAccountBinding == null)
            {
                base.Insert(account);
            }
        }

        /// <summary>
        /// 更新授权凭据
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="accountTypeKey"></param>
        /// <param name="identification"></param>
        /// <param name="accessToken"></param>
        /// <param name="expires_in"></param>
        public void UpdateAccessToken(long userId, string accountTypeKey, string identification, string accessToken, int expires_in)
        {
            Sql sql = Sql.Builder;
            sql.Select("*")
                .From("tn_AccountBindings")
                .Where("UserId = @0 and AccountTypeKey = @1 and identification = @2", userId, accountTypeKey, identification);
            AccountBinding accountBinding = CreateDAO().FirstOrDefault<AccountBinding>(sql);
            if (accountBinding != null)
            {
                accountBinding.AccessToken = accessToken;
                if (expires_in > 0)
                    accountBinding.ExpiredDate = DateTime.Now.AddSeconds(expires_in);
                base.Update(accountBinding);
            }
        }

        /// <summary>
        /// 删除第三方帐号绑定
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="accountTypeKey">第三方帐号类型Key</param>
        public void DeleteAccountBinding(long userId, string accountTypeKey)
        {
            AccountBinding accountBinding = this.GetAccountBinding(userId, accountTypeKey);

            Sql sql = Sql.Builder;
            sql.Append("delete from tn_AccountBindings")
                .Where("UserId=@0 and AccountTypeKey=@1", userId, accountTypeKey);
            CreateDAO().Execute(sql);

            OnDeleted(accountBinding);
        }

        /// <summary>
        /// 获取单个第三方帐号绑定
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="accountTypeKey">第三方帐号类型Key</param>
        /// <returns></returns>
        public AccountBinding GetAccountBinding(long userId, string accountTypeKey)
        {
            IEnumerable<AccountBinding> accountBindings = GetAccountBindings(userId);
            if (accountBindings != null && accountBindings.Count() > 0)
                return accountBindings.Where(ab => ab.AccountTypeKey.Equals(accountTypeKey, StringComparison.OrdinalIgnoreCase)).FirstOrDefault();
            else
                return null;
        }

        /// <summary>
        /// 获取某用户的所有第三方帐号绑定
        /// </summary>
        /// <param name="userId"></param>
        /// <returns>若没有，则返回空集合</returns>
        public IEnumerable<AccountBinding> GetAccountBindings(long userId)
        {
            Sql sql = Sql.Builder;
            sql.Select("Id")
                .From("tn_AccountBindings")
                .Where("UserId=@0", userId);

            IEnumerable<AccountBinding> accountBindings = Gets(sql);

            return accountBindings;
        }

        /// <summary>
        /// 获取用户Id
        /// </summary>
        /// <param name="accountTypeKey">第三方帐号类型Key</param>
        /// <param name="Identification">第三方帐号标识</param>
        /// <returns>用户Id</returns>
        public long GetUserId(string accountTypeKey, string Identification)
        {
            //设计说明:
            //无需缓存
            Sql sql = Sql.Builder;
            sql.Select("UserId")
                .From("tn_AccountBindings")
                .Where("AccountTypeKey= @0 and Identification= @1", accountTypeKey, Identification);
            long userId = CreateDAO().FirstOrDefault<long>(sql);
            return userId;
        }
    }
}