﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------
using System;
using Tunynet.Events;

namespace Tunynet.Common
{
    /// <summary>
    /// 举报业务逻辑
    /// </summary>
    public class ImpeachReportService
    {
        private IImpeachReportRepository impeachReportRepository;

        #region 构造器

        /// <summary>
        /// 构造函数
        /// </summary>
        public ImpeachReportService()
            : this(new ImpeachReportRepository())
        {
        }

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="impeachReportRepository">数据访问层实例</param>
        public ImpeachReportService(IImpeachReportRepository impeachReportRepository)
        {
            this.impeachReportRepository = impeachReportRepository;
        }

        #endregion 构造器

        #region 处理方法

        /// <summary>
        /// 创建举报
        /// </summary>
        /// <param name="report">要创建的举报实体</param>
        /// <returns></returns>
        public bool Create(ImpeachReport report)
        {
            impeachReportRepository.Insert(report);
            return report.Id > 0;
        }

        /// <summary>
        /// 标记为已处理
        /// </summary>
        /// <param name="reportId">要处理的举报Id</param>
        /// <returns></returns>
        public void Dispose(long reportId)
        {
            ImpeachReport report = impeachReportRepository.Get(reportId);
            if (report == null)
                return;
            report.Status = true;
            impeachReportRepository.Update(report);
            EventBus<ImpeachReport, CommonEventArgs>.Instance().OnAfter(report, new CommonEventArgs(EventOperationType.Instance().Approved()));
        }

        /// <summary>
        /// 删除举报
        /// </summary>
        /// <param name="reportId">举报Id集合</param>
        public void Delete(long reportId)
        {
            impeachReportRepository.DeleteByEntityId(reportId);
        }

        /// <summary>
        /// 获取举报的分页集合
        /// </summary>
        /// <param name="isDisposed">是否已处理</param>
        /// <param name="impeachReason">举报原因</param>
        /// <param name="startTtime">开始时间</param>
        /// <param name="endTtime">结束时间</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <returns>举报分页集合</returns>
        public PagingDataSet<ImpeachReport> GetsForAdmin(bool? isDisposed, ImpeachReasonEnum? impeachReason, DateTime? startTtime, DateTime? endTtime, int pageSize = 20, int pageIndex = 1)
        {
            return impeachReportRepository.GetsForAdmin(isDisposed, impeachReason, startTtime, endTtime, pageSize, pageIndex);
        }

        /// <summary>
        /// 获取单个举报实体
        /// </summary>
        /// <param name="reportId">举报Id</param>
        /// <returns></returns>
        public ImpeachReport GetReport(long reportId)
        {
            return impeachReportRepository.Get(reportId);
        }

        /// <summary>
        /// 删除垃圾数据
        /// </summary>
        public void DeleteTrashDatas()
        {
            impeachReportRepository.DeleteTrashDatas();
        }

        #endregion 处理方法
    }
}