﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Repositories;

namespace Tunynet.Common
{
    /// <summary>
    /// 勋章用户关联数据库仓储
    /// </summary>
    public class MedalToUserRepository : Repository<MedalToUser>, IMedalToUserRepository
    {
        /// <summary>
        /// 获取勋章授予记录（后台管理）
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="medalId">勋章Id</param>
        /// <param name="userAwardStatus">授予勋章状态</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <returns></returns>
        public PagingDataSet<MedalToUser> GetsMedalToUser(List<string> userId = null, long medalId = 0, UserAwardStatus? userAwardStatus = null, int pageSize = 20, int pageIndex = 1)
        {
            var sql = Sql.Builder;
            if (userId != null && userId.Count > 0)
                sql.Where("UserId in (@userId)", new { @userId = userId });
            if (medalId > 0)
                sql.Where("MedalId=@0", medalId);
            if (userAwardStatus.HasValue)
                sql.Where("UserAwardStatus=@0", userAwardStatus);
            sql.OrderBy("DateCreated desc");
            return GetPagingEntities(pageSize, pageIndex, sql);
        }

        /// <summary>
        /// 获取勋章授予记录（前台勋章获取记录）
        /// </summary>
        /// <param name="topNumber">前几条</param>
        /// <returns></returns>
        public IEnumerable<MedalToUser> GetTopsMedalToUser(int topNumber)
        {
            var sql = Sql.Builder;
            sql.Where("UserAwardStatus=@0", UserAwardStatus.AlreadyAward);
            sql.OrderBy("DateCreated desc");
            return GetTopEntities(topNumber, sql);
        }

        /// <summary>
        /// 获取用户某互斥组下的勋章授予记录Id
        /// </summary>
        /// <param name="groupId">互斥组Id</param>
        /// <param name="userId">用户Id</param>
        /// <returns></returns>
        public long GetMedaltoUserByGroupId(long groupId, long userId)
        {
            var sql = Sql.Builder;
            sql.Select("tn_MedalToUsers.*").From("tn_MedalToUsers");
            sql.LeftJoin("tn_MedalInGroups").On("tn_MedalToUsers.MedalId=tn_MedalInGroups.MedalId");
            sql.Where("tn_MedalInGroups.GroupId=@0", groupId).Where("tn_MedalToUsers.UserId=@0", userId);
            sql.Where("tn_MedalToUsers.UserAwardStatus=@0", UserAwardStatus.AlreadyAward);
            sql.OrderBy("tn_MedalToUsers.DateCreated");
            var medaltoUser = CreateDAO().Fetch<MedalToUser>(sql);
            if (medaltoUser.Count >= 1)
                return medaltoUser.FirstOrDefault().Id;
            else
                return 0;
        }

        /// <summary>
        /// 删除勋章授予记录
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        public void DeleteMedaltoUserByMedalId(long medalId)
        {
            var sql = Sql.Builder;
            sql.Append("Delete from tn_MedalToUsers");
            if (medalId > 0)
            {
                sql.Where("MedalId=@0", medalId);
                CreateDAO().Execute(sql);
            }
        }

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        public void UpdateMedalToUserAuthor(long userId, string displayName)
        {
            var sql = Sql.Builder.Append("Update tn_MedalToUsers SET UserDisplayName=@1 WHERE UserId=@0", userId, displayName);
            CreateDAO().Execute(sql);

            RealTimeCacheHelper.IncreaseGlobalVersion();
        }
    }
}