﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Tunynet.CMS;
using Tunynet.Events;
using Tunynet.Post;

namespace Tunynet.Common
{
    /// <summary>
    /// 勋章业务逻辑
    /// </summary>
    public class MedalService
    {
        #region define

        private IMedalRepository medalRepository;
        private IMedalConditionRepository medalConditionRepository;
        private IMedalGroupRepository medalGroupRepository;
        private IMedalInConditionRepository medalInConditionRepository;
        private IMedalInGroupRepository medalInGroupRepository;
        private IMedalToUserRepository medalToUserRepository;
        private ThreadService threadService;
        private ContentItemService contentItemService;

        /// <summary>
        /// 勋章业务逻辑
        /// </summary>
        public MedalService(IMedalRepository medalRepository, IMedalConditionRepository medalConditionRepository, IMedalGroupRepository medalGroupRepository,
                            IMedalInConditionRepository medalInConditionRepository, IMedalInGroupRepository medalInGroupRepository, IMedalToUserRepository medalToUserRepository,
                            ThreadService threadService, ContentItemService contentItemService)
        {
            this.medalRepository = medalRepository;
            this.medalConditionRepository = medalConditionRepository;
            this.medalGroupRepository = medalGroupRepository;
            this.medalInConditionRepository = medalInConditionRepository;
            this.medalInGroupRepository = medalInGroupRepository;
            this.medalToUserRepository = medalToUserRepository;
            this.threadService = threadService;
            this.contentItemService = contentItemService;
        }

        #endregion define
        #region Create&Update&Delete

        #region 勋章

        /// <summary>
        /// 添加勋章
        /// </summary>
        /// <param name="medal">勋章实体</param>
        /// <returns></returns>
        public bool Create(Medal medal)
        {
            medalRepository.Insert(medal);
            new AttachmentService(TenantTypeIds.Instance().Medal()).ToggleTemporaryAttachments(medal.AttachmentIdsFinal, medal.MedalId);
            if (medal.MedalId > 0)
            {
                EventBus<Medal>.Instance().OnAfter(medal, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章
        /// </summary>
        /// <param name="medal">勋章实体</param>
        public void Update(Medal medal)
        {
            medalRepository.Update(medal);
            new AttachmentService(TenantTypeIds.Instance().Medal()).ToggleTemporaryAttachments(medal.AttachmentIdsFinal, medal.MedalId);
            EventBus<Medal>.Instance().OnAfter(medal, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        public void Delete(long medalId)
        {
            Medal medal = medalRepository.Get(medalId);
            if (medal != null)
            {
                medalRepository.Delete(medal);
                //执行事件
                EventBus<Medal>.Instance().OnAfter(medal, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 添加勋章用户关联
        /// </summary>
        /// <param name="medalToUser">勋章用户关联实体</param>
        /// <returns></returns>
        public bool CreateMedalToUser(MedalToUser medalToUser)
        {
            medalToUserRepository.Insert(medalToUser);
            if (medalToUser.Id > 0)
            {
                EventBus<MedalToUser>.Instance().OnAfter(medalToUser, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章用户关联
        /// </summary>
        /// <param name="medalToUser">勋章用户关联实体</param>
        public void UpdateMedalToUser(MedalToUser medalToUser)
        {
            medalToUserRepository.Update(medalToUser);
            EventBus<MedalToUser>.Instance().OnAfter(medalToUser, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章用户关联
        /// </summary>
        /// <param name="Id">勋章用户关联Id</param>
        public void DeleteMedalToUser(long Id)
        {
            MedalToUser medalToUser = medalToUserRepository.Get(Id);
            if (medalToUser != null)
            {
                medalToUserRepository.Delete(medalToUser);
                //执行事件
                EventBus<MedalToUser>.Instance().OnAfter(medalToUser, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 删除勋章授予记录
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        public void DeleteMedaltoUserByMedalId(long medalId)
        {
            medalToUserRepository.DeleteMedaltoUserByMedalId(medalId);
        }

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        public void UpdateMedalToUserAuthor(long userId, string displayName)
        {
            medalToUserRepository.UpdateMedalToUserAuthor(userId, displayName);
        }

        #endregion 勋章

        #region 勋章互斥组

        /// <summary>
        /// 添加勋章互斥组
        /// </summary>
        /// <param name="medalGroup">勋章互斥组实体</param>
        /// <returns></returns>
        public bool CreateGroup(MedalGroup medalGroup)
        {
            medalGroupRepository.Insert(medalGroup);
            if (medalGroup.GroupId > 0)
            {
                EventBus<MedalGroup>.Instance().OnAfter(medalGroup, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章互斥组
        /// </summary>
        /// <param name="medalGroup">勋章互斥组实体</param>
        public void UpdateGroup(MedalGroup medalGroup)
        {
            medalGroupRepository.Update(medalGroup);
            EventBus<MedalGroup>.Instance().OnAfter(medalGroup, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章互斥组
        /// </summary>
        /// <param name="medalGroupId">勋章互斥组Id</param>
        public void DeleteGroup(long medalGroupId)
        {
            MedalGroup medalGroup = medalGroupRepository.Get(medalGroupId);
            if (medalGroup != null)
            {
                medalGroupRepository.Delete(medalGroup);
                //执行事件
                EventBus<MedalGroup>.Instance().OnAfter(medalGroup, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 添加勋章互斥组关联
        /// </summary>
        /// <param name="medalInGroup">勋章互斥组关联实体</param>
        /// <returns></returns>
        public bool CreateMedalInGroup(MedalInGroup medalInGroup)
        {
            medalInGroupRepository.Insert(medalInGroup);
            if (medalInGroup.Id > 0)
            {
                EventBus<MedalInGroup>.Instance().OnAfter(medalInGroup, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章互斥组关联
        /// </summary>
        /// <param name="medalInGroup">勋章互斥组关联实体</param>
        public void UpdateMedalInGroup(MedalInGroup medalInGroup)
        {
            medalInGroupRepository.Update(medalInGroup);
            EventBus<MedalInGroup>.Instance().OnAfter(medalInGroup, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章互斥组关联
        /// </summary>
        /// <param name="Id">勋章互斥组关联Id</param>
        public void DeleteMedalInGroup(long Id)
        {
            MedalInGroup medalInGroup = medalInGroupRepository.Get(Id);
            if (medalInGroup != null)
            {
                medalInGroupRepository.Delete(medalInGroup);
                //执行事件
                EventBus<MedalInGroup>.Instance().OnAfter(medalInGroup, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 删除勋章的互斥组设置
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        public void DeleteMedalInGroupByMedalId(long medalId)
        {
            medalInGroupRepository.DeleteMedalInGroupByMedalId(medalId);
        }

        #endregion 勋章互斥组

        #region 勋章条件

        /// <summary>
        /// 添加勋章条件
        /// </summary>
        /// <param name="medalCondition">勋章条件实体</param>
        /// <returns></returns>
        public bool CreateCondition(MedalCondition medalCondition)
        {
            medalConditionRepository.Insert(medalCondition);
            if (medalCondition.ConditionId > 0)
            {
                EventBus<MedalCondition>.Instance().OnAfter(medalCondition, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章条件
        /// </summary>
        /// <param name="medalCondition">勋章条件实体</param>
        public void UpdateCondition(MedalCondition medalCondition)
        {
            medalConditionRepository.Update(medalCondition);
            EventBus<MedalCondition>.Instance().OnAfter(medalCondition, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章条件
        /// </summary>
        /// <param name="Id">勋章条件Id</param>
        public void DeleteCondition(long Id)
        {
            MedalCondition medalCondition = medalConditionRepository.Get(Id);
            if (medalCondition != null)
            {
                medalConditionRepository.Delete(medalCondition);
                //执行事件
                EventBus<MedalCondition>.Instance().OnAfter(medalCondition, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 添加勋章条件关联
        /// </summary>
        /// <param name="medalInCondition">勋章条件关联实体</param>
        /// <returns></returns>
        public bool CreateMedalInCondition(MedalInCondition medalInCondition)
        {
            medalInConditionRepository.Insert(medalInCondition);
            if (medalInCondition.Id > 0)
            {
                EventBus<MedalInCondition>.Instance().OnAfter(medalInCondition, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新勋章条件关联
        /// </summary>
        /// <param name="medalInCondition">勋章条件关联实体</param>
        public void UpdateMedalInCondition(MedalInCondition medalInCondition)
        {
            medalInConditionRepository.Update(medalInCondition);
            EventBus<MedalInCondition>.Instance().OnAfter(medalInCondition, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除勋章条件关联
        /// </summary>
        /// <param name="Id">勋章条件关联Id</param>
        public void DeleteMedalInCondition(long Id)
        {
            MedalInCondition medalInCondition = medalInConditionRepository.Get(Id);
            if (medalInCondition != null)
            {
                medalInConditionRepository.Delete(medalInCondition);
                //执行事件
                EventBus<MedalInCondition>.Instance().OnAfter(medalInCondition, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 删除勋章的条件设置
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        public void DeleteMedalInConditionByMedalId(long medalId)
        {
            medalInConditionRepository.DeleteMedalInConditionByMedalId(medalId);
        }

        #endregion 勋章条件

        #endregion Create&Update&Delete

        #region Get&Gets

        #region 勋章

        /// <summary>
        /// 获取勋章
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        /// <returns>勋章实体</returns>
        public Medal Get(long medalId)
        {
            return medalRepository.Get(medalId);
        }

        /// <summary>
        /// 获取勋章列表（后台管理）
        /// </summary>
        /// <param name="awardStatus">授予状态</param>
        /// <param name="userId">用户Id</param>
        /// <returns></returns>
        public IEnumerable<Medal> Gets(AwardStatus? awardStatus = null, long userId = 0)
        {
            return medalRepository.Gets(awardStatus, userId);
        }

        /// <summary>
        /// 获取用户授予勋章实体
        /// </summary>
        /// <param name="Id">用户授予勋章Id</param>
        /// <returns>户授予勋章实体</returns>
        public MedalToUser GetMedalToUser(long Id)
        {
            return medalToUserRepository.Get(Id);
        }

        /// <summary>
        /// 获取勋章授予记录（后台管理、我的/他的勋章）
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="medalId">勋章Id</param>
        /// <param name="userAwardStatus">授予勋章状态</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <returns></returns>
        public PagingDataSet<MedalToUser> GetsMedalToUser(List<string> userId = null, long medalId = 0, UserAwardStatus? userAwardStatus = null, int pageSize = 20, int pageIndex = 1)
        {
            return medalToUserRepository.GetsMedalToUser(userId, medalId, userAwardStatus, pageSize, pageIndex);
        }

        /// <summary>
        /// 获取勋章授予记录（前台勋章获取记录）
        /// </summary>
        /// <param name="topNumber">前几条</param>
        /// <returns></returns>
        public IEnumerable<MedalToUser> GetTopsMedalToUser(int topNumber = 10)
        {
            return medalToUserRepository.GetTopsMedalToUser(topNumber);
        }

        /// <summary>
        /// 获取用户某互斥组下的勋章授予记录Id
        /// </summary>
        /// <param name="groupId">互斥组Id</param>
        /// <param name="userId">用户Id</param>
        /// <returns></returns>
        public long GetMedaltoUserByGroupId(long groupId, long userId)
        {
            return medalToUserRepository.GetMedaltoUserByGroupId(groupId, userId);
        }

        #endregion 勋章

        #region 勋章互斥组

        /// <summary>
        /// 获取勋章互斥组
        /// </summary>
        /// <param name="groupId">勋章互斥组Id</param>
        /// <returns>勋章互斥组实体</returns>
        public MedalGroup GetGroup(long groupId)
        {
            return medalGroupRepository.Get(groupId);
        }

        /// <summary>
        /// 获取所有勋章互斥组
        /// </summary>
        /// <returns>所有勋章互斥组实体</returns>
        public IEnumerable<MedalGroup> GetGroups()
        {
            return medalGroupRepository.GetAll();
        }

        /// <summary>
        /// 获取互斥组Id
        /// </summary>
        /// <returns>互斥组Id</returns>
        public long GetGroupId(long medalId)
        {
            return medalInGroupRepository.GetGroupId(medalId);
        }

        /// <summary>
        /// 获取勋章互斥组关联
        /// </summary>
        /// <returns>勋章互斥组关联</returns>
        public IEnumerable<MedalInGroup> GetMedalInGroup()
        {
            return medalInGroupRepository.GetAll();
        }

        /// <summary>
        /// 获取互斥组勋章数量
        /// </summary>
        /// <returns>互斥组勋章数量</returns>
        /// <param name="groupId">互斥组Id</param>
        /// <returns></returns>
        public int GetMedalCount(long groupId)
        {
            return medalInGroupRepository.GetMedalCount(groupId);
        }

        #endregion 勋章互斥组

        #region 勋章条件

        /// <summary>
        /// 获取勋章条件
        /// </summary>
        /// <param name="conditionId">勋章条件Id</param>
        /// <returns>勋章条件实体</returns>
        public MedalCondition GetCondition(long conditionId)
        {
            return medalConditionRepository.Get(conditionId);
        }

        /// <summary>
        /// 获取所有勋章条件
        /// </summary>
        /// <returns>勋章条件实体</returns>
        public IEnumerable<MedalCondition> GetConditions()
        {
            return medalConditionRepository.GetAll("DisplayOrder");
        }

        /// <summary>
        /// 获取勋章关系关联
        /// </summary>
        /// <returns>勋章关系关联</returns>
        public IEnumerable<MedalInCondition> GetMedalInCondition()
        {
            return medalInConditionRepository.GetAll();
        }

        /// <summary>
        /// 获取某勋章的条件信息
        /// </summary>
        /// <param name="medalId">勋章Id</param>
        /// <returns>勋章关系关联</returns>
        public IEnumerable<MedalInCondition> GetMedalInConditionBymedalId(long medalId)
        {
            return medalInConditionRepository.GetMedalInConditionBymedalId(medalId);
        }

        /// <summary>
        /// 获取勋章某条件的最小值
        /// </summary>
        /// <returns>勋章条件的最小值</returns>
        /// <param name="conditionId">条件Id</param>
        /// <param name="medalId">勋章Id</param>
        /// <returns></returns>
        public int GetMinCondition(long conditionId, long medalId)
        {
            return medalInConditionRepository.GetMinCondition(conditionId, medalId);
        }

        /// <summary>
        /// 更改勋章显示顺序
        /// </summary>
        /// <param name="fromMedalId"></param>
        /// <param name="toMedalId"></param>
        public void ChangeMedalDisplayOrder(long fromMedalId, long toMedalId)
        {
            var fromMedal = Get(fromMedalId);
            var toMedal = Get(toMedalId);
            //更改栏目显示顺序
            var temp = fromMedal.DisplayOrder;
            fromMedal.DisplayOrder = toMedal.DisplayOrder;
            medalRepository.Update(fromMedal);
            toMedal.DisplayOrder = temp;
            medalRepository.Update(toMedal);
        }

        #endregion 勋章条件

        #endregion Get&Gets

        #region OtherMethod

        #region 勋章

        /// <summary>
        /// 用户等级检查
        /// </summary>
        /// <param name="user">检查用户</param>
        /// <returns></returns>
        public int UserRankCheck(IUser user)
        {
            return user.Rank;
        }

        /// <summary>
        /// 咨询投稿检查
        /// </summary>
        /// <param name="user">检查用户</param>
        /// <returns></returns>
        public int CreateCMSNumCheck(IUser user)
        {
            return Convert.ToInt32(contentItemService.GetContentItems(null, null, user.UserId).TotalRecords);
        }

        /// <summary>
        /// 回答问题数检查
        /// </summary>
        /// <param name="user">检查用户</param>
        /// <returns></returns>
        public int AnwerNumCheck(IUser user)
        {
            return medalRepository.GetAnwerNum(user.UserId);
        }

        /// <summary>
        /// 评论回复数检查
        /// </summary>
        /// <param name="user">检查用户</param>
        /// <returns></returns>
        public int CommentNumCheck(IUser user)
        {
            return medalRepository.GetCommentNum(user.UserId);
        }

        /// <summary>
        /// 通过方法名，反射调用函数(Medal)
        /// </summary>
        /// <param name="methodName">方法名</param>
        /// <param name="user">用户</param>
        /// <returns></returns>
        public int MedalRunMethod(string methodName, IUser user)
        {
            Type type = typeof(MedalService);
            MethodInfo[] mi = type.GetMethods();
            foreach (MethodInfo m in mi)
            {
                if (m.Name == methodName)
                {
                    object[] para = new object[1];
                    para[0] = user;
                    return Convert.ToInt32(m.Invoke(this, para));
                }
            }
            return 0;
        }

        #endregion 勋章

        #endregion OtherMethod
    }
}