﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using Tunynet.Events;

namespace Tunynet.Common
{
    /// <summary>
    /// 私信逻辑类
    /// </summary>
    public class MessageService
    {
        private IMessageRepository messageRepository;
        private IMessageSessionRepository messageSessionRepository;
        private IMessageInSessionRepository messageInSessionRepository;

        /// <summary>
        /// 构造器
        /// </summary>
        public MessageService(IMessageRepository messageRepository, IMessageSessionRepository messageSessionRepository, IMessageInSessionRepository messageInSessionRepository)
        {
            this.messageRepository = messageRepository;
            this.messageSessionRepository = messageSessionRepository;
            this.messageInSessionRepository = messageInSessionRepository;
        }

        #region 私信

        /// <summary>
        /// 创建私信
        /// </summary>
        /// <param name="message">待创建的私信实体</param>
        /// <param name="sessionId">会话Id</param>
        /// <returns>是否删除成功：true-成功，false-不成功</returns>
        public bool Create(Message message, long sessionId = 0)
        {
            if (message == null)
                return false;

            EventBus<Message>.Instance().OnBefore(message, new CommonEventArgs(EventOperationType.Instance().Create()));

            messageRepository.Create(message, sessionId);
            EventBus<Message>.Instance().OnAfter(message, new CommonEventArgs(EventOperationType.Instance().Create()));

            return message.MessageId > 0;
        }

        /// <summary>
        /// 删除私信
        /// </summary>
        /// <param name="messageId">私信Id</param>
        /// <param name="sessionId">私信会话Id</param>
        /// <returns>是否删除成功：true-成功，false-不成功</returns>
        public bool Delete(long messageId, long sessionId)
        {
            Message message = messageRepository.Get(messageId);
            if (message == null)
                return false;

            EventBus<Message>.Instance().OnBefore(message, new CommonEventArgs(EventOperationType.Instance().Delete()));
            int affectCount = messageRepository.Delete(message, sessionId);
            EventBus<Message>.Instance().OnAfter(message, new CommonEventArgs(EventOperationType.Instance().Delete()));

            return affectCount > 0;
        }

        /// <summary>
        /// 获取私信实体
        /// </summary>
        /// <param name="messageId">私信Id</param>
        public Message Get(long messageId)
        {
            return messageRepository.Get(messageId);
        }

        /// <summary>
        /// 获取主会话Id
        /// </summary>
        ///<param name="sessionId">会话Id</param>
        public MessageSession GetSenderSessionId(long sessionId)
        {
            return messageSessionRepository.GetSenderSessionId(sessionId);
        }

        /// <summary>
        /// 设置私信为已读
        /// </summary>
        /// <param name="sessionId">私信会话Id</param>
        /// <param name="userId">会话拥有者Id</param>
        /// <returns>是否更新成功</returns>
        public bool SetIsRead(long sessionId, long userId)
        {
            bool isRead = messageRepository.SetIsRead(sessionId, userId);
            return isRead;
        }

        /// <summary>
        /// 获取用户前N个私信
        /// </summary>
        ///<param name="userId">UserId</param>
        ///<param name="sortBy">私信排序字段</param>
        ///<param name="topNumber">要获取的记录数</param>
        public IEnumerable<Message> GetTops(long userId, SortBy_Message? sortBy, int topNumber)
        {
            return messageRepository.GetTopMessagesOfUser(userId, sortBy, topNumber);
        }

        /// <summary>
        ///获取会话中前N条私信
        /// </summary>
        /// <param name="sessionId">会话Id</param>
        /// <param name="topNumber">要获取的记录数</param>
        public IEnumerable<Message> GetTops(long sessionId, int topNumber)
        {
            IEnumerable<object> sessionids = messageInSessionRepository.GetMessageIds(sessionId, topNumber);
            IEnumerable<Message> messages;
            if (sessionids.Count() > topNumber)
            {
                messages = messageRepository.PopulateEntitiesByEntityIds(sessionids.Take(topNumber));
            }
            else
            {
                messages = messageRepository.PopulateEntitiesByEntityIds(sessionids);
            }

            return messages;
        }

        /// <summary>
        ///获取会话中所有私信
        /// </summary>
        /// <param name="sessionId">会话Id</param>
        /// <param name="topNumber">获取记录条数</param>
        public IEnumerable<Message> GetMessagesOfSession(long sessionId, int topNumber)
        {
            IEnumerable<object> sessionids = messageInSessionRepository.GetMessageIds(sessionId, topNumber);
            return messageRepository.PopulateEntitiesByEntityIds(sessionids);
        }

        /// <summary>
        /// 获取会话下的某条私信之前的20条私信Id(移动端使用)
        /// </summary>
        /// <param name="sessionId">会话Id</param>
        /// <param name="oldMessageId">会话Id</param>
        public IEnumerable<Message> GetMessagesOfSession(long sessionId, long oldMessageId)
        {
            IEnumerable<object> sessionids = messageInSessionRepository.GetMessageIds(sessionId, oldMessageId);
            IEnumerable<Message> messages;
            if (sessionids.Count() > 20)
                messages = messageRepository.PopulateEntitiesByEntityIds(sessionids.Take(20));
            else
                messages = messageRepository.PopulateEntitiesByEntityIds(sessionids);

            return messages;
        }

        /// <summary>
        /// 获取未读私信数
        /// </summary>
        /// <param name="userId">私信拥有者UserId</param>
        /// <returns></returns>
        public int GetUnreadCount(long userId)
        {
            return messageRepository.GetUnReadCount(userId);
        }

        /// <summary>
        /// 批量更改author字段
        /// </summary>
        public void UpdateMessageAuthor(long userId, string displayName)
        {
            messageRepository.UpdateMessageAuthor(userId, displayName);
        }

        #endregion 私信

        #region 私信会话

        /// <summary>
        ///
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="otherUserId"></param>
        /// <returns></returns>
        public MessageSession GetMessageSession(long userId, long otherUserId)
        {
            return messageSessionRepository.Get(userId, otherUserId);
        }

        /// <summary>
        /// 获取私信实体
        /// </summary>
        /// <param name="messageId">私信Id</param>
        public MessageSession GetSession(long messageId)
        {
            return messageSessionRepository.Get(messageId);
        }

        /// <summary>
        /// 删除私信会话
        /// </summary>
        /// <param name="sessionId">私信会话Id</param>
        public bool DeleteSession(long sessionId)
        {
            MessageSession messageSession = messageSessionRepository.Get(sessionId);
            if (messageSession != null)
            {
                long otherSessionId = messageSessionRepository.GetSessionsId(messageSession.UserId, messageSession.OtherUserId);
                int count = messageSessionRepository.DeleteByEntityId(sessionId);
                if (count > 0)
                {
                    return messageSessionRepository.DeleteByEntityId(otherSessionId) > 0;
                }
            }
            return false;
        }

        /// <summary>
        /// 获取用户私信会话
        /// </summary>
        ///<param name="userId">用户Id</param>
        ///<param name="pageIndex">页码</param>
        ///<param name="pageSize">每页大小</param>
        public PagingDataSet<MessageSession> GetSessionsOfUser(long userId, int pageIndex, int pageSize = 20)
        {
            return messageSessionRepository.GetSessionsOfUser(userId, pageIndex, pageSize);
        }

        /// <summary>
        /// 获取私信会话分页数据（后台用）
        /// </summary>
        /// <param name="type">私信类型</param>
        /// <param name="userId">用户Id（用来搜索用户Id相关的私信会话）</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="asAnonymous">是否匿名</param>
        /// <param name="pageSize">每页大小</param>
        /// <returns>私信会话分页数据</returns>
        public PagingDataSet<MessageSession> GetSessions(MessageType? type, long? userId, int pageIndex, bool asAnonymous = false, int pageSize = 20)
        {
            return messageSessionRepository.GetSessions(type, userId, asAnonymous, pageIndex, pageSize);
        }

        /// <summary>
        /// 清除用户的私信会话
        /// </summary>
        /// <param name="userId">用户Id</param>
        public void ClearSessionsFromUser(long userId)
        {
            messageSessionRepository.ClearSessionsFromUser(userId);
        }

        /// <summary>
        /// 获取前N个私信会话
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="topNumber">获取记录条数</param>
        /// <param name="hasUnread">是否仅获取未读会话</param>
        public IEnumerable<MessageSession> GetTopSessions(long userId, int topNumber, bool hasUnread = false)
        {
            return messageSessionRepository.GetTopSessions(userId, topNumber, hasUnread);
        }

        /// <summary>
        /// 获取会话最早的Message
        /// </summary>
        /// <param name="sessionId"></param>
        /// <returns></returns>
        public Message GetFirstMessageOfSession(long sessionId)
        {
            return messageRepository.GetFirstMssageOfSession(sessionId);
        }

        #endregion 私信会话

        #region 客服消息

        /// <summary>
        /// 客服消息
        /// </summary>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">每页记录数</param>
        /// <param name="userName">发件人</param>
        /// <param name="roleId">角色</param>
        /// <param name="minRank">最小等级</param>
        /// <param name="maxRank">最大等级</param>
        /// <returns></returns>
        public PagingDataSet<MessageSession> GetCustomerMessages(int pageIndex, int pageSize, string userName, long roleId, long minRank = 0, long maxRank = 0)
        {
            return messageSessionRepository.GetCustomerMessages(pageIndex, pageSize, userName, roleId, minRank, maxRank);
        }

        #endregion 客服消息
    }
}