﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System.Collections.Generic;
using Tunynet.Repositories;

namespace Tunynet.Common
{  /// <summary>
   /// 私信会话数据访问接口
   /// </summary>
    public interface IMessageSessionRepository : IRepository<MessageSession>
    {
        /// <summary>
        /// 根据我跟其他人的id获取我们之间的会话（如果不存在，创建一个空会话）
        /// </summary>
        /// <param name="userId">用户id</param>
        /// <param name="otherUserId">其他用户的id</param>
        /// <returns>会话</returns>
        MessageSession Get(long userId, long otherUserId);

        /// <summary>
        /// 获取用户私信会话
        /// </summary>
        ///<param name="userId">用户Id</param>
        ///<param name="pageIndex">当前页码</param>
        ///<param name="pageSize">每页大小</param>
        PagingDataSet<MessageSession> GetSessionsOfUser(long userId, int pageIndex, int pageSize);

        /// <summary>
        /// 获取私信会话分页数据（后台用）
        /// </summary>
        /// <param name="type">私信类型</param>
        /// <param name="userId">用户Id（用来搜索用户Id相关的私信会话）</param>
        /// <param name="asAnonymous">是否匿名</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">每页大小</param>
        /// <returns>私信会话分页数据</returns>
        PagingDataSet<MessageSession> GetSessions(MessageType? type, long? userId, bool asAnonymous, int pageIndex, int pageSize);

        /// <summary>
        /// 获取主会话Id
        /// </summary>
        ///<param name="sessionId">会话Id</param>
        MessageSession GetSenderSessionId(long sessionId);

        /// <summary>
        /// 获取前N个私信会话
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="topNumber">获取记录条数</param>
        /// <param name="hasUnread">是否仅获取未读会话</param>
        IEnumerable<MessageSession> GetTopSessions(long userId, int topNumber, bool hasUnread = false);

        /// <summary>
        /// 清除用户所有私信会话
        /// </summary>
        /// <param name="userId">用户Id</param>
        void ClearSessionsFromUser(long userId);

        /// <summary>
        /// 获取客服消息
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <param name="pageSize"></param>
        /// <param name="userName"></param>
        /// <param name="roleId">角色</param>
        /// <param name="minRank">最小等级</param>
        /// <param name="maxRank">最大等级</param>
        /// <returns></returns>
        PagingDataSet<MessageSession> GetCustomerMessages(int pageIndex, int pageSize, string userName, long roleId, long minRank, long maxRank);

        /// <summary>
        /// 获取私信会话ID
        /// </summary>
        /// <param name="userId">用户ID</param>
        /// <param name="otherUserId">其他人的ID</param>
        long GetSessionsId(long userId, long otherUserId);
    }
}