﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using PetaPoco;
using System;
using Tunynet.Utilities;

namespace Tunynet.Common
{
    /// <summary>
    /// 通知的实体类
    /// </summary>
    [TableName("tn_Notices")]
    [PrimaryKey("Id", autoIncrement = true)]
    [Serializable]
    public class Notice : SerializablePropertiesBase, IEntity
    {
        /// <summary>
        /// 新建实体时使用
        /// </summary>
        public static Notice New()
        {
            Notice notice = new Notice();
            notice.Body = string.Empty;
            notice.NoticeTypeKey = string.Empty;
            notice.ReceiverId = 0;
            notice.LeadingActorUserId = 0;
            notice.LeadingActor = string.Empty;
            notice.RelativeObjectName = string.Empty;
            notice.RelativeObjectId = 0;
            notice.RelativeObjectUrl = string.Empty;
            notice.Status = 0;
            notice.DateCreated = DateTime.Now;
            notice.LastSendDate = DateTime.Now;
            notice.Times = 0;

            return notice;
        }

        #region 需持久化属性

        /// <summary>
        /// ID
        /// </summary>
        public long Id { get; set; }

        /// <summary>
        /// 通知类型 Key
        /// </summary>
        public string NoticeTypeKey { get; set; }

        /// <summary>
        /// 通知接收人
        /// </summary>
        public long ReceiverId { get; set; }

        /// <summary>
        /// 主角 UserId
        /// </summary>
        public long LeadingActorUserId { get; set; }

        /// <summary>
        /// 主角
        /// </summary>
        public string LeadingActor { get; set; }

        /// <summary>
        /// 相关项对象名称
        /// </summary>
        public string RelativeObjectName { get; set; }

        /// <summary>
        /// 相关项对象 Id
        /// </summary>
        public long RelativeObjectId { get; set; }

        /// <summary>
        /// 相关项对象链接地址
        /// </summary>
        public string RelativeObjectUrl { get; set; }

        /// <summary>
        /// 触发通知的对象Id
        /// </summary>
        public long ObjectId { get; set; }

        /// <summary>
        /// 内容
        /// </summary>
        public string Body { get; set; }

        /// <summary>
        /// 处理状态   0=Unhandled:未处理;1=Readed  知道了;  2=Accepted 接受；3=Refused 拒绝；
        /// </summary>
        public NoticeStatus Status { get; set; }

        /// <summary>
        /// 创建日期
        /// </summary>
        public DateTime DateCreated { get; set; }

        /// <summary>
        /// 上次发送时间
        /// </summary>
        public DateTime LastSendDate { get; set; }

        /// <summary>
        /// 通知发送次数
        /// </summary>
        public int Times { get; set; }

        #endregion 需持久化属性

        #region 拓展属性

        /// <summary>
        /// 获取解析过的内容
        /// </summary>
        [Ignore]
        public string ResolvedBody
        {
            get
            {
                //if (!string.IsNullOrEmpty(Body))
                //    return Body;
                return NoticeBuilder.Instance().Resolve(this);
            }
        }

        /// <summary>
        /// 获取手机模板解析内容
        /// </summary>
        [Ignore]
        public string MobileResolvedBody
        {
            get
            {
                //if (!string.IsNullOrEmpty(Body))
                //    return Body;
                return NoticeBuilder.Instance().MobileResolve(this);
            }
        }

        /// <summary>
        /// 个推内容的title
        /// </summary>
        [Ignore]
        public string GetuiTitle
        {
            get
            {
                var title = string.Empty;

                switch (this.NoticeTypeKey)
                {
                    case "NewThreadReply":
                    case "NewCommentReply":
                    case "NewArticleReply":
                    case "NewImageReply":
                    case "NewVideoReply":
                    case "NewCMSReply":
                    case "NewAskAnswerComment":
                    case "NewAskQuestionComment":
                    case "NewAskComment":
                    case "NewDocumentComment":
                        title = "您有一条新回复";
                        break;
                    case "ThreadApproved":
                    case "CMSArticleApproved":
                    case "ThreadDisapproved":
                    case "CMSArticleDisapproved":
                    case "DocumentManagerApproved":
                    case "DocumentManagerDisapproved":
                    case "EventApproved":
                    case "EventDisapproved":
                    case "NewParticipate":
                    case "EventManagerApproved":
                    case "MedalApproved":
                    case "MedalDisapproved":
                    case "TaskApproved":
                    case "TaskDisapproved":
                    case "MedalAward":
                    case "MedalRecovered":
                        title = "您有一条新提醒";
                        break;

                    case "FollowUser":
                        title = "您有一条关注信息";
                        break;

                    case "NewAnswer":
                        title = "您的问题有新的回答";
                        break;

                    case "SetBestAnswer":
                        title = "您的回答被设为了最佳回答";
                        break;
                    case "CancelOrder":
                        title = string.Format("您的“{0}”兑换已被取消", RelativeObjectName);
                        break;
                    case "ConfirmOrder":
                        title = string.Format("您的积分兑换礼品“{0}”已被管理员确认", RelativeObjectName);
                        break;
                    case "CompleteOrder":
                        title = string.Format("您的积分兑换礼品“{0}”已完成兑换", RelativeObjectName);
                        break;

                    default:
                        break;
                }

                return title;
            }
        }

        /// <summary>
        /// 个推内容的Text
        /// </summary>
        [Ignore]
        public string GetuiText
        {
            get
            {
                var text = string.Empty;

                switch (this.NoticeTypeKey)
                {
                    case "NewThreadReply":
                    case "NewCommentReply":
                    case "NewArticleReply":
                    case "NewImageReply":
                    case "NewVideoReply":
                    case "NewCMSReply":
                    case "NewAskAnswerComment":
                    case "NewAskQuestionComment":
                    case "NewAskComment":
                    case "NewDocumentComment":
                        text = string.Format("{0}在'{1}'中回复了您", LeadingActor, RelativeObjectName);
                        break;

                    case "ThreadApproved":
                        text = string.Format("您的贴子'{0}'通过了审核", RelativeObjectName);
                        break;

                    case "CMSArticleApproved":
                        text = string.Format("您的文章'{0}'通过了审核", RelativeObjectName);
                        break;
                    case "DocumentManagerApproved":
                        text = string.Format("您的文档'{0}'通过了审核", RelativeObjectName);
                        break;
                    case "EventManagerApproved":
                        text = string.Format("您的活动'{0}'通过了审核", RelativeObjectName);
                        break;
                    case "TaskApproved":
                        text = string.Format("您的任务'{0}'已通过审核，设置为完成", RelativeObjectName);
                        break;
                    case "MedalApproved":
                        text = string.Format("您申请的'{0}'通过批准", RelativeObjectName);
                        break;
                    case "MedalDisapproved":
                        text = string.Format("您申请的'{0}'已被拒绝", RelativeObjectName);
                        break;
                    case "EventApproved":
                        text = string.Format("您在'{0}'的报名信息已被审核通过", RelativeObjectName);
                        break;
                    case "EventDisapproved":
                        text = string.Format("您在'{0}'的报名信息没有通过审核", RelativeObjectName);
                        break;
                    case "NewParticipate":
                        text = string.Format("您的活动'{0}'，有新报名人'{1}'", RelativeObjectName,LeadingActor);
                        break;
                    case "MedalAward":
                        text = string.Format("您被授予新的勋章{0}", RelativeObjectName);
                        break;
                    case "MedalRecovered":
                        text = string.Format("您的勋章'{0}'被收回", RelativeObjectName);
                        break;
                    case "ThreadDisapproved":
                        text = string.Format("您的贴子'{0}'没有通过审核", RelativeObjectName);
                        break;
                    case "CMSArticleDisapproved":
                        text = string.Format("您的文章'{0}'没有通过审核", RelativeObjectName);
                        break;
                    case "DocumentManagerDisapproved":
                        text = string.Format("您的文档'{0}'没有通过审核", RelativeObjectName);
                        break;
                    case "TaskDisapproved":
                        text = string.Format("您的任务'{0}' 未通过审核，反馈意见为：'{1}'", RelativeObjectName, Body);
                        break;
                    case "FollowUser":
                        text = string.Format("{0}关注了您", LeadingActor);
                        break;

                    case "NewAnswer":
                        text = string.Format("{0}在'{1}'中回答了您", LeadingActor, RelativeObjectName);
                        break;

                    case "AskQuestionApproved":
                        text = string.Format("您的提问{0}通过了审核", RelativeObjectName);
                        break;

                    case "AskAnswerApproved":
                        text = string.Format("您的回答{0}通过了审核", RelativeObjectName);
                        break;

                    case "SetBestAnswer":
                        text = string.Format("您在'{0}'的回答被设为了最佳回答", RelativeObjectName);
                        break;
                    case "CancelOrder":
                        text = string.Format("您的积分兑换礼品“{0}”，{1}。", RelativeObjectName, Body);
                        break;
                    case "ConfirmOrder":
                        text = string.Format("您的积分兑换礼品“{0}”，已被管理员确认。 {0}", RelativeObjectName, Body);
                        break;
                    case "CompleteOrder":
                        text = string.Format("您的“{0}”兑换已完成兑换。{1}", RelativeObjectName, Body);
                        break;
                   

                    default:
                        break;
                }

                return HtmlUtility.StripHtml(text, false, false);
            }
        }

        /// <summary>
        /// 主角空间链接
        /// </summary>
        [Ignore]
        public string LeadingActorUrl
        {
            get
            {
                return GetExtendedProperty<string>("LeadingActorUrl");
            }
            set
            {
                SetExtendedProperty("LeadingActorUrl", value);
            }
        }

        #endregion 拓展属性

        #region IEntity 成员

        object IEntity.EntityId { get { return this.Id; } }
        bool IEntity.IsDeletedInDatabase { get; set; }

        #endregion IEntity 成员
    }
}