﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Dynamic;
using System.Linq;

namespace Tunynet.Common
{
    /// <summary>
    /// 积分充值订单管理
    /// </summary>
    public class PointRechargeOrderService
    {
        private IPointRechargeOrderRepository pointRechargeOrderRepository;
        private IKvStore kvStore;

        /// <summary>
        /// 构造函数注入
        /// </summary>
        public PointRechargeOrderService(IPointRechargeOrderRepository pointRechargeOrderRepository, IKvStore kvStore)
        {
            this.pointRechargeOrderRepository = pointRechargeOrderRepository;
            this.kvStore = kvStore;
        }

        /// <summary>
        /// 创建积分充值订单
        /// </summary>
        /// <param name="pointRechargeOrder"></param>
        public void Create(PointRechargeOrder pointRechargeOrder)
        {
            pointRechargeOrderRepository.Insert(pointRechargeOrder);
        }

        /// <summary>
        /// 更新积分充值订单
        /// </summary>
        /// <param name="pointRechargeOrder"></param>
        public void Update(PointRechargeOrder pointRechargeOrder)
        {
            pointRechargeOrderRepository.Update(pointRechargeOrder);
        }

        /// <summary>
        /// 删除积分充值订单
        /// </summary>
        /// <param name="id"></param>
        public void Delete(long id)
        {
            pointRechargeOrderRepository.DeleteByEntityId(id);
        }

        /// <summary>
        /// 获取积分充值订单详情
        /// </summary>
        /// <param name="orderId"></param>
        public PointRechargeOrder Get(long orderId)
        {
            return pointRechargeOrderRepository.Get(orderId);
        }

        /// <summary>
        /// 获取积分充值订单列表
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="startDate">开始时间</param>
        /// <param name="endDate">结束时间</param>
        /// <param name="pageSize">个数</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public PagingDataSet<PointRechargeOrder> Gets(long userId = 0, DateTime? startDate = null, DateTime? endDate = null, int pageSize = 20, int pageIndex = 1)
        {
            return pointRechargeOrderRepository.Gets(userId, startDate, endDate, pageSize, pageIndex);
        }

        /// <summary>
        /// 保存积分充值类型
        /// </summary>
        /// <param name="pointRechargeSettings">积分充值设置</param>
        public void SetPointRechargeSetting(IDictionary<float, float> pointRechargeSettings)
        {
            List<dynamic> PointRechargeSettings = new List<dynamic>();
            foreach (var item in pointRechargeSettings)
            {
                dynamic PointRechargeSetting = new ExpandoObject();
                PointRechargeSetting.TotalPrice = item.Key;
                PointRechargeSetting.TradePoints = item.Value;
                PointRechargeSettings.Add(PointRechargeSetting);
            }
            kvStore.Set(KvKeys.Instance().PointRechargeSetting(), PointRechargeSettings);
        }

        /// <summary>
        /// 删除积分充值类型
        /// </summary>
        /// <param name="totalPrice">金额</param>
        public void RemovePointRechargeSetting(float totalPrice)
        {
            List<dynamic> PointRechargeSettings = new List<dynamic>();
            kvStore.TryGet<List<dynamic>>(KvKeys.Instance().PointRechargeSetting(), out PointRechargeSettings);
            PointRechargeSettings.Remove(PointRechargeSettings.Select(n => n.TotalPrice == totalPrice).FirstOrDefault());
            kvStore.Set(KvKeys.Instance().PointRechargeSetting(), PointRechargeSettings);
        }

        /// <summary>
        /// 获取所有充值类型
        /// </summary>
        public IEnumerable<dynamic> GetPointRechargeSettings()
        {
            List<dynamic> PointRechargeSettings = new List<dynamic>();
            kvStore.TryGet<List<dynamic>>(KvKeys.Instance().PointRechargeSetting(), out PointRechargeSettings);
            return PointRechargeSettings ?? new List<dynamic>();
        }
    }
}