﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------
using Newtonsoft.Json;
using PetaPoco;
using System;
using System.Collections.Generic;
using System.Linq;
using Tunynet.Caching;

namespace Tunynet.Common
{
    /// <summary>
    /// 积分任务实体
    /// </summary>
    [TableName("tn_PointTasks")]
    [PrimaryKey("TaskId", autoIncrement = true)]
    [CacheSetting(true)]
    [Serializable]
    public class PointTask : IEntity
    {
        /// <summary>
        /// 初始化数据
        /// </summary>
        /// <returns></returns>
        public static PointTask New()
        {
            PointTask pointTask = new PointTask
            {
                TaskName = string.Empty,
                Description = string.Empty,
                TasksSettings = string.Empty,
                Deadline = DateTime.Now.AddDays(30),
                DateCreated = DateTime.Now
            };
            return pointTask;
        }

        #region 需持久化属性

        /// <summary>
        /// 积分任务Id
        /// </summary>
        public long TaskId { get; set; }

        /// <summary>
        /// 积分任务种类Id
        /// </summary>
        public long TypeId { get; set; }

        /// <summary>
        /// 积分任务名
        /// </summary>
        public string TaskName { get; set; }

        /// <summary>
        /// 描述
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// 奖励积分值
        /// </summary>
        public int AwardPoints { get; set; }

        /// <summary>
        /// 奖励金币数
        /// </summary>
        public int AwardGolds { get; set; }

        /// <summary>
        /// 申请所需最小用户等级
        /// </summary>
        public int MinUserRank { get; set; }

        /// <summary>
        /// 最小条件（人工审核任务时为0）
        /// </summary>
        public int MinCondition { get; set; }

        /// <summary>
        /// 任务内容（人工审核任务、分享任务url）（多个任务内容json存储）
        /// </summary>
        public string TasksSettings { get; set; }

        /// <summary>
        /// 截止时间
        /// </summary>
        public DateTime Deadline { get; set; }

        /// <summary>
        /// 任务状态（正常、已禁用、已超期）
        /// </summary>
        public PointTaskStatus Status { get; set; }

        /// <summary>
        /// 创建时间
        /// </summary>
        public DateTime DateCreated { get; set; }

        #endregion 需持久化属性



        #region 扩展属性/方法

        /// <summary>
        /// 任务种类
        /// </summary>
        [Ignore]
        [JsonIgnore]
        public PointTaskType TaskType
        {
            get
            {
                return DIContainer.Resolve<PointTaskService>().GetType(TypeId);
            }
        }

        /// <summary>
        /// 获取积分任务用户领取数量
        /// </summary>
        /// <returns></returns>
        public int GetTaskToUserNum()
        {
            int taskToUserNum = 0;
            DIContainer.Resolve<IKvStore>().TryGet(KvKeys.Instance().PointTasktoUser(TaskId), out taskToUserNum);
            return taskToUserNum;
        }

        /// <summary>
        /// 获取用户该任务的记录
        /// </summary>
        /// <returns></returns>
        public PointTaskRecord GetTaskToUser(long userId, TaskRecordStatus? taskRecordStatus = null)
        {
            return DIContainer.Resolve<PointTaskService>().GetRecordsBy(taskRecordStatus, TaskId, userId);
        }

        /// <summary>
        /// 获取最新领取此任务的用户
        /// </summary>
        /// <returns></returns>
        public List<IUser> GetNewUserOfTask(int topNumber)
        {
            var records = DIContainer.Resolve<PointTaskService>().GetRecords(null, TaskId, null, topNumber, 1);
            List<IUser> users = new List<IUser>();
            foreach (var record in records)
            {
                if (record.Status != TaskRecordStatus.Abandoned)
                    users.Add(DIContainer.Resolve<UserService>().GetFullUser(record.UserId));
            }
            return users;
        }

        /// <summary>
        /// 获取任务进度
        /// </summary>
        /// <returns></returns>
        public int CheckUserTask(IUser user)
        {
            if (!string.IsNullOrEmpty(TaskType.CheckMethodName))
            {
                var beginTime = GetTaskToUser(user.UserId, TaskRecordStatus.Doing)?.DateCreated ?? DateTime.Now;
                return DIContainer.Resolve<PointTaskService>().PointTaskRunMethod(TaskType.CheckMethodName, user, MinCondition, beginTime, this);
            }
            else
                return 0;
        }

        #endregion 扩展属性/方法

        #region IEntity 成员

        object IEntity.EntityId { get { return this.TaskId; } }

        bool IEntity.IsDeletedInDatabase { get; set; }

        #endregion IEntity 成员
    }
}