﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Reflection;
using Tunynet.Events;

namespace Tunynet.Common
{
    /// <summary>
    /// 积分任务业务逻辑
    /// </summary>
    public class PointTaskService
    {
        #region define

        private IPointTaskRepository pointTaskRepository;
        private IPointTaskTypeRepository pointTaskTypeRepository;
        private IPointTaskRecordRepository pointTaskRecordRepository;
        private IKvStore kvStore;
        private UserSignInService userSignInService;
        private InviteFriendService inviteFriendService;

        /// <summary>
        /// 积分任务业务逻辑
        /// </summary>
        public PointTaskService(IPointTaskRepository pointTaskRepository, IPointTaskTypeRepository pointTaskTypeRepository, IPointTaskRecordRepository pointTaskRecordRepository, IKvStore kvStore, UserSignInService userSignInService, InviteFriendService inviteFriendService)
        {
            this.pointTaskRepository = pointTaskRepository;
            this.pointTaskRecordRepository = pointTaskRecordRepository;
            this.pointTaskTypeRepository = pointTaskTypeRepository;
            this.kvStore = kvStore;
            this.userSignInService = userSignInService;
            this.inviteFriendService = inviteFriendService;
        }

        #endregion define

        #region Create&Update&Delete

        #region 积分任务

        /// <summary>
        /// 创建积分任务
        /// </summary>
        /// <param name="task">任务实体</param>
        /// <returns></returns>
        public bool Create(PointTask task)
        {
            pointTaskRepository.Insert(task);
            if (task.TaskId > 0)
            {
                EventBus<PointTask>.Instance().OnAfter(task, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新积分任务
        /// </summary>
        /// <param name="task">任务实体</param>
        /// <returns></returns>
        public void Update(PointTask task)
        {
            pointTaskRepository.Update(task);
            EventBus<PointTask>.Instance().OnAfter(task, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除积分任务
        /// </summary>
        /// <param name="taskId">任务id</param>
        /// <returns></returns>
        public void Delete(long taskId)
        {
            PointTask task = pointTaskRepository.Get(taskId);
            if (task != null)
            {
                pointTaskRepository.Delete(task);
                //执行事件
                EventBus<PointTask>.Instance().OnAfter(task, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        #endregion 积分任务

        #region 任务领取记录

        /// <summary>
        /// 创建领取记录
        /// </summary>
        /// <param name="taskRecord">领取记录实体</param>
        /// <returns></returns>
        public bool CreateRecord(PointTaskRecord taskRecord)
        {
            pointTaskRecordRepository.Insert(taskRecord);
            if (taskRecord.TaskId > 0)
            {
                EventBus<PointTaskRecord>.Instance().OnAfter(taskRecord, new CommonEventArgs(EventOperationType.Instance().Create()));
                return true;
            }
            return false;
        }

        /// <summary>
        /// 更新领取记录
        /// </summary>
        /// <param name="taskRecord">领取记录实体</param>
        /// <returns></returns>
        public void UpdateRecord(PointTaskRecord taskRecord)
        {
            pointTaskRecordRepository.Update(taskRecord);
            EventBus<PointTaskRecord>.Instance().OnAfter(taskRecord, new CommonEventArgs(EventOperationType.Instance().Update()));
        }

        /// <summary>
        /// 删除领取记录
        /// </summary>
        /// <param name="taskRecordId">领取记录id</param>
        /// <returns></returns>
        public void DeleteRecord(long taskRecordId)
        {
            PointTaskRecord taskRecord = pointTaskRecordRepository.Get(taskRecordId);
            if (taskRecord != null)
            {
                pointTaskRecordRepository.Delete(taskRecord);
                //执行事件
                EventBus<PointTaskRecord>.Instance().OnAfter(taskRecord, new CommonEventArgs(EventOperationType.Instance().Delete()));
            }
        }

        /// <summary>
        /// 根据任务ID删除任务领取记录
        /// </summary>
        /// <param name="taskId"></param>
        public void DeleteTasktoUserByTaskId(long taskId)
        {
            pointTaskRecordRepository.DeleteTasktoUserByTaskId(taskId);
        }

        #endregion 任务领取记录

        #endregion Create&Update&Delete

        #region Get&Gets

        #region 积分任务

        /// <summary>
        /// 获取积分任务
        /// </summary>
        /// <param name="taskId">任务id</param>
        /// <returns></returns>
        public PointTask Get(long taskId)
        {
            return pointTaskRepository.Get(taskId);
        }

        /// <summary>
        /// 获取积分任务列表
        /// </summary>
        /// <param name="pointTaskStatus">积分任务状态</param>
        /// <returns></returns>
        public IEnumerable<PointTask> Gets(PointTaskStatus? pointTaskStatus = null)
        {
            return pointTaskRepository.Gets(pointTaskStatus);
        }

        #endregion 积分任务

        #region 任务领取记录

        /// <summary>
        /// 获取积分任务领取记录
        /// </summary>
        /// <param name="recordId">记录id</param>
        /// <returns></returns>
        public PointTaskRecord GetRecord(long recordId)
        {
            return pointTaskRecordRepository.Get(recordId);
        }

        /// <summary>
        /// 获取积分任务领取记录列表
        /// </summary>
        /// <param name="taskRecordStatus">领取状态</param>
        /// <param name="taskId">任务Id</param>
        /// <param name="userId">用户Id</param>
        /// <param name="pageSize">页面大小</param>
        /// <param name="pageIndex">页码</param>
        /// <returns></returns>
        public PagingDataSet<PointTaskRecord> GetRecords(TaskRecordStatus? taskRecordStatus = null, long taskId = 0, List<string> userId = null, int pageSize = 20, int pageIndex = 1)
        {
            return pointTaskRecordRepository.GetRecords(taskRecordStatus, taskId, userId, pageSize, pageIndex);
        }

        /// <summary>
        /// 获取积分任务领取记录列表
        /// </summary>
        /// <param name="taskRecordStatus">领取状态</param>
        /// <param name="taskId">任务Id</param>
        /// <param name="userId">用户Id</param>
        /// <returns></returns>
        public PointTaskRecord GetRecordsBy(TaskRecordStatus? taskRecordStatus = null, long taskId = 0, long userId = 0)
        {
            return pointTaskRecordRepository.GetRecordsBy(taskRecordStatus, taskId, userId);
        }

        #endregion 任务领取记录

        #region 任务种类

        /// <summary>
        /// 获取积分任务种类
        /// </summary>
        /// <param name="typeId">种类id</param>
        /// <returns></returns>
        public PointTaskType GetType(long typeId)
        {
            return pointTaskTypeRepository.Get(typeId);
        }

        /// <summary>
        /// 获取全部积分任务种类
        /// </summary>
        /// <returns></returns>
        public IEnumerable<PointTaskType> GetTypes(bool? canAddTask = null)
        {
            return pointTaskTypeRepository.GetTypes(canAddTask);
        }

        #endregion 任务种类

        #endregion Get&Gets

        #region OtherMethod

        #region 积分任务//判断方法

        /// <summary>
        /// 检验用户资料填写
        /// </summary>
        /// <returns></returns>
        public int UserInfoCheck(IUser user, int minCondition, DateTime beginTime, PointTask task)
        {
            var userInfo = pointTaskRepository.GetUserInfo(user.UserId);
            int result = 0;
            if (userInfo != null)
            {
                if (userInfo.Gender != GenderType.NotSet)
                    result += 25;
                if (!string.IsNullOrEmpty(userInfo.NowAreaCode))
                    result += 25;
                if (!string.IsNullOrEmpty(userInfo.Introduction))
                    result += 25;
                if (user.HasCover > 0)
                    result += 25;
            }
            return result;
        }

        /// <summary>
        /// 检验用户手机端登录
        /// </summary>
        /// <returns></returns>
        public int AppLoginCheck(IUser user, int minCondition, DateTime beginTime, PointTask task)
        {
            KvValue AppLoginNum = kvStore.Get("AppLogin_UserId_" + user.UserId);
            if (AppLoginNum != null)
                return 100;
            else
                return 0;
        }

        /// <summary>
        /// 检验连续签到天数
        /// </summary>
        /// <returns></returns>
        public int UserSignInCheck(IUser user, int minCondition, DateTime beginTime, PointTask task)
        {
            if (minCondition == 0)
                return 0;
            UserSignIn userSignIn = userSignInService.GetByUserId(user.UserId);
            if (userSignIn != null)
            {
                var userSignInTime = userSignInService.GetSignInTime(user.UserId, beginTime);
                var taskNum = (DateTime.Now - beginTime).Days + (userSignInTime > DateTime.Now ? 0 : (userSignInTime > beginTime ? 1 : 0));
                int continuedSignCount = userSignIn.ContinuedSignCount >= taskNum ? taskNum : userSignIn.ContinuedSignCount;
                int checkNum = Convert.ToInt32(Math.Round(Convert.ToDouble(continuedSignCount) / Convert.ToDouble(minCondition), 2) * 100);
                return checkNum > 100 ? 100 : checkNum;
            }
            else
                return 0;
        }

        /// <summary>
        /// 检验邀请朋友
        /// </summary>
        /// <returns></returns>
        public int InviteFriendCheck(IUser user, int minCondition, DateTime beginTime, PointTask task)
        {
            if (minCondition == 0)
                return 0;
            long totalRecords = 0;
            //邀请朋友Id列表
            IEnumerable<long> friendsIds = inviteFriendService.GetMyInviteFriendRecords(user.UserId, 9999, 1, out totalRecords, beginTime);
            int checkNum = Convert.ToInt32(Math.Round(Convert.ToDouble(totalRecords) / Convert.ToDouble(minCondition), 2) * 100);
            return checkNum > 100 ? 100 : checkNum;
        }

        /// <summary>
        /// 检验站外分享
        /// </summary>
        /// <returns></returns>
        public int OutShareCheck(IUser user, int minCondition, DateTime beginTime, PointTask task)
        {
            KvValue AppLoginNum = kvStore.Get("ShareOut_UserId_" + user.UserId + "_TaskId_" + task.TaskId);
            if (AppLoginNum != null)
                return 100;
            else
                return 0;
        }

        /// <summary>
        /// 通过方法名，反射调用函数(task)
        /// </summary>
        /// <param name="methodName">方法名</param>
        /// <param name="user">用户</param>
        /// <param name="minCondition">最小条件</param>
        /// <param name="beginTime">任务开始时间</param>
        /// <param name="task">任务</param>
        /// <returns></returns>
        public int PointTaskRunMethod(string methodName, IUser user, int minCondition = 0, DateTime? beginTime = null, PointTask task = null)
        {
            Type type = typeof(PointTaskService);
            MethodInfo[] mi = type.GetMethods();
            foreach (MethodInfo m in mi)
            {
                if (m.Name == methodName)
                {
                    object[] para = new object[4];
                    para[0] = user;
                    para[1] = minCondition;
                    para[2] = beginTime;
                    para[3] = task;
                    return Convert.ToInt32(m.Invoke(this, para));
                }
            }
            return 0;
        }

        #endregion 积分任务//判断方法

        #endregion OtherMethod
    }
}