﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

namespace Tunynet.Common
{
    /// <summary>
    /// 热词业务逻辑
    /// </summary>
    public class SearchWordService : ISearchWordService
    {
        private ISearchWordRepository searchWordRepositry;
        private CountService countService = new CountService(TenantTypeIds.Instance().SearchWord());

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="searchWordRepositry"></param>
        public SearchWordService(ISearchWordRepository searchWordRepositry)
        {
            this.searchWordRepositry = searchWordRepositry;
        }

        /// <summary>
        /// 根据Id查询搜索词
        /// </summary>
        /// <param name="id">搜索词Id</param>
        /// <returns></returns>
        public SearchWord Gets(long id)
        {
            return searchWordRepositry.Get(id);
        }

        /// <summary>
        /// 获取搜索词分页集合
        /// </summary>
        /// <param name="word">搜索词</param>
        /// <param name="searchTypeCode">搜索类型</param>
        /// <param name="pageSize">分页大小</param>
        /// <param name="pageIndex">当前页码</param>
        /// <returns></returns>
        public PagingDataSet<SearchWord> GetsForWord(string word, string searchTypeCode, int pageSize = 20, int pageIndex = 1)
        {
            return searchWordRepositry.GetsForWord(word, searchTypeCode, pageSize, pageIndex);
        }

        /// <summary>
        /// 增加实体
        /// </summary>
        /// <param name="entity">实体数据</param>
        /// <param name="changeCount">变化计数</param>
        public void Set(SearchWord entity, int changeCount = 1)
        {
            SearchWord searchWord = searchWordRepositry.GetSearchWord(entity.Word, entity.SearchTypeCode);
            if (searchWord == null)
            {
                entity.DateCreated = DateTime.Now;
                entity.LastModified = DateTime.Now;
                searchWordRepositry.Insert(entity);
                countService.ChangeCount(CountTypes.Instance().SearchWordCounts(), entity.Id, 0, changeCount, true);
            }
            else
            {
                countService.ChangeCount(CountTypes.Instance().SearchWordCounts(), searchWord.Id, 0, changeCount, true);
            }
        }

        /// <summary>
        /// 根据Id删除实体
        /// </summary>
        /// <param name="id">搜索词id</param>
        public void Delete(long id)
        {
            searchWordRepositry.DeleteByEntityId(id);
            countService.Delete(id);
        }

        /// <summary>
        /// 更新实体
        /// </summary>
        /// <param name="entity">实体数据</param>
        /// <param name="changeCount">变化计数</param>
        public void Update(SearchWord entity, int changeCount = 1)
        {
            searchWordRepositry.Update(entity);
            countService.ChangeCount(CountTypes.Instance().SearchWordCounts(), entity.Id, 0, changeCount, true);
        }

        /// <summary>
        /// 获取前六条数据
        /// </summary>
        /// <param name="searchTypeCode">热词类型</param>
        /// <returns></returns>
        public IEnumerable<SearchWord> GetTopWord(string searchTypeCode)
        {
            return searchWordRepositry.GetTopWord(searchTypeCode);
        }
    }
}