﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using Tunynet.Events;

namespace Tunynet.Common
{
    /// <summary>
    /// 用户签到业务逻辑
    /// </summary>
    public class UserSignInService
    {
        private IUserSignInDetailRepository userSignInDetailRepository;
        private IUserSignInRepository userSignInRepository;

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="userSignInDetailRepository"></param>
        /// <param name="userSignInRepository"></param>
        public UserSignInService(IUserSignInDetailRepository userSignInDetailRepository, IUserSignInRepository userSignInRepository)
        {
            this.userSignInDetailRepository = userSignInDetailRepository;
            this.userSignInRepository = userSignInRepository;
        }

        private static readonly object lockObject = new object();

        /// <summary>
        /// 用户签到保存
        /// </summary>
        /// <param name="userId">用户Id</param>
        public bool Set(long userId)
        {
            lock (lockObject)
            {
                var result = userSignInDetailRepository.IsSignIn(userId);
                if (!result)
                {
                    UserSignInDetail userSignInDetail = UserSignInDetail.New(userId);
                    userSignInDetailRepository.Insert(userSignInDetail);
                    EventBus<UserSignInDetail>.Instance().OnAfter(userSignInDetail, new CommonEventArgs(EventOperationType.Instance().Create()));
                }
                return !result;
            }
        }

        /// <summary>
        /// 根据用户ID查询用户的签到统计
        /// </summary>
        /// <param name="userId">用户id</param>
        /// <returns>用户签到统计实体</returns>
        public UserSignIn GetByUserId(long userId)
        {
            return userSignInRepository.GetByUserId(userId);
        }

        /// <summary>
        /// 获取用户签到数据
        /// </summary>
        /// <param name="keyword">关键字（支持昵称，姓名，手机号，邮箱）</param>
        /// <param name="userSignInOrder">排序：总累计签到、月累计</param>
        /// <param name="pageIndex">页码</param>
        /// <param name="pageSize">数量</param>
        /// <returns></returns>
        public PagingDataSet<UserSignIn> Gets(string keyword, UserSignInOrder userSignInOrder = UserSignInOrder.SignCount_Desc, int pageIndex = 1, int pageSize = 20)
        {
            return userSignInRepository.Gets(keyword, userSignInOrder, pageIndex, pageSize);
        }

        /// <summary>
        /// 获取今天签到总人数
        /// </summary>
        /// <returns></returns>
        public int GetSignInTodayCount()
        {
            return userSignInDetailRepository.GetSignInTodayCount();
        }

        /// <summary>
        /// 获取用户签到历史明细
        /// </summary>
        /// <param name="userId"></param>
        /// <param name="recentMonths">历史前几个月的(不填为当前月)</param>
        /// <param name="isMobile">是否手机端</param>
        /// <returns></returns>
        public IEnumerable<UserSignInDetail> GetUserHistorDetails(long userId, int recentMonths = 0, bool isMobile = false)
        {
            if (recentMonths < 0)
                return new List<UserSignInDetail>();
            return userSignInDetailRepository.GetUserHistorDetails(userId, recentMonths, isMobile);
        }

        /// <summary>
        /// 删除用户时候删除签到记录
        /// </summary>
        public void DeleteTrashDatas()
        {
            userSignInRepository.DeleteTrashDatas();
        }

        /// <summary>
        /// 获取用户某天的签到时间
        /// </summary>
        /// <param name="userId">用户Id</param>
        /// <param name="taskTime">任务时间</param>
        /// <returns></returns>
        public DateTime GetSignInTime(long userId,DateTime taskTime) {
            return userSignInDetailRepository.GetSignInTime(userId,taskTime);
        }
    }
}