﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using Tunynet.CMS;
using Tunynet.Common.Repositories;
using Tunynet.Events;
using Tunynet.Logging;
using Tunynet.Post;
using Tunynet.Settings;
using Tunynet.Utilities;

namespace Tunynet.Common
{
    /// <summary>
    /// 评论创建和删除的事件
    /// </summary>
    public class CommentEventModule : IEventMoudle
    {
        private PointService pointService;
        private AuditService auditService;
        private TenantTypeService tenantTypeService;
        private ContentItemService contentItemService;
        private ThreadRepository threadRepository;
        private ThreadService threadService;
        private OperationLogService operationLogService;
        private UserService userService;
        private RoleService roleService;
        private CommentService commentService;
        private IKvStore kvStore;
        private NoticeService noticeService;
        private INoticeSender noticeSender;
        private ImpeachReportService impeachReportService;
        private ISettingsManager<SiteSettings> siteSettings = DIContainer.Resolve<ISettingsManager<SiteSettings>>();

        //构造
        public CommentEventModule(PointService pointService, AuditService auditService, TenantTypeService tenantTypeService,
            ContentItemService contentItemService, ThreadService threadService, OperationLogService operationLogService, UserService userService, RoleService roleService, CommentService commentService, ThreadRepository threadRepository, IKvStore kvStore, NoticeService noticeService, INoticeSender noticeSender, ImpeachReportService impeachReportService)
        {
            this.pointService = pointService;
            this.auditService = auditService;
            this.tenantTypeService = tenantTypeService;
            this.contentItemService = contentItemService;
            this.threadService = threadService;
            this.operationLogService = operationLogService;
            this.userService = userService;
            this.commentService = commentService;
            this.roleService = roleService;
            this.threadRepository = threadRepository;
            this.kvStore = kvStore;
            this.noticeService = noticeService;
            this.noticeSender = noticeSender;
            this.impeachReportService = impeachReportService;
        }

        /// <summary>
        /// 注册事件处理程序
        /// </summary>
        public void RegisterEventHandler()
        {
            EventBus<Comment>.Instance().After += new CommonEventHandler<Comment, CommonEventArgs>(CommentModuleForManagerOperation_After);
            EventBus<Comment>.Instance().After += new CommonEventHandler<Comment, CommonEventArgs>(CommentIndex_After);
            EventBus<Comment, AuditEventArgs>.Instance().After += new CommonEventHandler<Comment, AuditEventArgs>(CommentPointEventModule_After);
            EventBus<Comment, AuditEventArgs>.Instance().After += new CommonEventHandler<Comment, AuditEventArgs>(CommentCountEventModule_After);
            EventBus<Comment, CommonEventArgs>.Instance().After += new CommonEventHandler<Comment, CommonEventArgs>(CommentModuleForOperationLog_After);
            //用户修改对外显示名称事件
            EventBus<User>.Instance().AfterWithHistory += new EventHandlerWithHistory<User, CommonEventArgs>(UserEventModule_AfterWithHistory);
        }

        /// <summary>
        /// 用户修改对外显示名称事件
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="eventArgs"></param>
        /// <param name="historyData"></param>
        private void UserEventModule_AfterWithHistory(User sender, CommonEventArgs eventArgs, User historyData)
        {
            if (historyData != null && eventArgs.EventOperationType == EventOperationType.Instance().Update())
            {
                if (userService.CanEditDisplayName(sender, historyData))
                {
                    commentService.UpdateCommentAuthor(sender.UserId, sender.DisplayName);
                }
            }
        }

        /// <summary>
        /// 评论创建处理数量事件
        /// </summary>
        /// <param name="comment"></param>
        /// <param name="eventArgs"></param>
        private void CommentModuleForManagerOperation_After(Comment comment, CommonEventArgs eventArgs)
        {
            CountService countService = new CountService(comment.TenantTypeId);

            if (eventArgs.EventOperationType == EventOperationType.Instance().Create())
            {
                if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                {
                    //评论贴子则更新回复时间 （调用Repository不触发service中update事件
                    var thread = threadService.Get(comment.CommentedObjectId);
                    thread.LastModified = DateTime.Now;
                    threadRepository.Update(thread);
                }
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Delete())
            {
                //删除举报
                impeachReportService.DeleteTrashDatas();
            }
        }

        /// <summary>
        /// *评论通过审核事件处理
        /// *积分操作事件
        /// *发送评论审核通过通知
        /// </summary>
        /// <param name="comment">评论实体</param>
        /// <param name="eventArgs">审核状态</param>
        private void CommentPointEventModule_After(Comment comment, AuditEventArgs eventArgs)
        {
            //因为扩展模块都单独定义了评论的事件 所以这里增加了租户id的限制 目前没有区分扩展模块的方法 暂时先这样
            if (!tenantTypeService.Get(comment.TenantTypeId)?.ClassType.Contains("Spacebuilder") ?? false)
            {
                string pointItemKey = string.Empty;
                string eventOperationType = string.Empty;
                string description = string.Empty;
                string tenantTypeId = comment.TenantTypeId;
                CountService countService = new CountService(comment.TenantTypeId);
                if (comment.TenantTypeId == TenantTypeIds.Instance().ContentItem())
                {
                    var contentItem = contentItemService.Get(comment.CommentedObjectId);

                    if (contentItem.ContentModel.ModelKey == ContentModelKeys.Instance().Article() || contentItem.ContentModel.ModelKey == ContentModelKeys.Instance().Contribution())
                    {
                        tenantTypeId = TenantTypeIds.Instance().CMS_Article();
                    }
                    else if (contentItem.ContentModel.ModelKey == ContentModelKeys.Instance().Image())
                    {
                        tenantTypeId = TenantTypeIds.Instance().CMS_Image();
                    }
                    else if (contentItem.ContentModel.ModelKey == ContentModelKeys.Instance().Video())
                    {
                        tenantTypeId = TenantTypeIds.Instance().CMS_Video();
                    }
                }

                if (eventArgs.EventOperationType == EventOperationType.Instance().Create() || eventArgs.EventOperationType == EventOperationType.Instance().Approved())
                {
                    //评论审核通过则发送通知给相应用户
                    Notice notice = Notice.New();
                    notice.LeadingActor = comment.Author;
                    notice.LeadingActorUserId = comment.UserId;
                    notice.RelativeObjectUrl = comment.GetCommentedObjectUrl();
                    notice.LeadingActorUrl = SiteUrls.Instance().SpaceHome(comment.UserId);
                    notice.Body = HtmlUtility.TrimHtml(comment.Body, 110);
                    notice.ObjectId = comment.Id;
                    if (comment.ParentId != 0)
                    {
                        var parentComment = commentService.Get(comment.ParentId);
                        notice.ReceiverId = parentComment.UserId;
                        notice.NoticeTypeKey = NoticeTypeKeys.Instance().NewReply(string.Empty);
                        notice.RelativeObjectId = comment.CommentedObjectId;
                        notice.RelativeObjectName = HtmlUtility.TrimHtml(parentComment.Body, 110);
                    }
                    else
                    {
                        notice.ReceiverId = comment.GetCommentedObject().UserId;
                        notice.NoticeTypeKey = NoticeTypeKeys.Instance().NewReply(tenantTypeId);
                        notice.RelativeObjectId = comment.CommentedObjectId;
                        notice.RelativeObjectName = HtmlUtility.TrimHtml(comment.GetCommentedObject().Name, 110);
                    }

                    if (notice.ReceiverId != eventArgs.OperatorInfo.OperationUserId)
                    {
                        //发送通知
                        noticeService.Create(notice);
                        noticeSender.Send(notice);
                    }
                }

                bool? auditDirection = auditService.ResolveAuditDirection(eventArgs.OldAuditStatus, eventArgs.NewAuditStatus);
                if (auditDirection == true) //加积分
                {
                    pointItemKey = PointItemKeys.Instance().CreateComment();
                    if (eventArgs.OldAuditStatus == null)
                        eventOperationType = EventOperationType.Instance().Create();
                    else
                        eventOperationType = EventOperationType.Instance().Approved();
                }
                else if (auditDirection == false) //减积分
                {
                    pointItemKey = PointItemKeys.Instance().DeleteComment();
                    if (eventArgs.NewAuditStatus == null)
                        eventOperationType = EventOperationType.Instance().Delete();
                    else
                        eventOperationType = EventOperationType.Instance().Disapproved();
                }
                if (!string.IsNullOrEmpty(pointItemKey))
                {
                    //描述
                    if (eventOperationType == EventOperationType.Instance().Create())
                    {
                        description = string.Format("发布评论");
                    }
                    else if (eventOperationType == EventOperationType.Instance().Delete())
                    {
                        description = string.Format("删除评论");
                    }
                    else if (eventOperationType == EventOperationType.Instance().Approved())
                    {
                        description = string.Format("评论通过审核");
                    }
                    else if (eventOperationType == EventOperationType.Instance().Disapproved())
                    {
                        description = string.Format("评论不通过审核");
                    }
                    pointService.GenerateByRoles(comment.UserId, eventArgs.OperatorInfo.OperationUserId, pointItemKey, description, comment.TenantTypeId, comment.Id, comment.Subject, false, true);
                }
            }
        }

        /// <summary>
        /// 评论操作日志事件
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="eventArgs"></param>
        private void CommentModuleForOperationLog_After(Comment sender, CommonEventArgs eventArgs)
        {
            var commentedObject = commentService.Get(sender.Id).GetCommentedObject();

            OperationLog newLog = new OperationLog(eventArgs.OperatorInfo);
            newLog.OperationObjectId = sender.Id;
            newLog.OperationObjectName = "<a class=\"a\" target=\"_blank\" href=" + commentedObject.DetailUrl + ">" + commentedObject.Name + "</a>";
            newLog.OperationType = eventArgs.EventOperationType;
            newLog.TenantTypeId = TenantTypeIds.Instance().Comment();
            newLog.OperationUserRole = string.Join(",", roleService.GetRoleNamesOfUser(eventArgs.OperatorInfo.OperationUserId));

            if (eventArgs.EventOperationType == EventOperationType.Instance().Create())
            {
                newLog.Description = "发布评论：" + sender.Body;
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Approved())
            {
                newLog.Description = "评论通过审核：" + sender.Body;
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Disapproved())
            {
                newLog.Description = "评论不通过审核：" + sender.Body;
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Delete())
            {
                newLog.Description = string.Format("删除评论: {0}", sender.Body);
            }

            operationLogService.Create(newLog);
        }

        /// <summary>
        /// 处理评论计数事件
        /// </summary>
        /// <remarks>
        /// 使用count计数的全部修改为根据站点设置进行处理，用kvstore对个人空间的计数不做修改
        /// </remarks>
        /// <param name="comment">评论实体</param>
        /// <param name="eventArgs">审核状态</param>
        private void CommentCountEventModule_After(Comment comment, AuditEventArgs eventArgs)
        {
            Comment commentParent = commentService.Get(comment.ParentId);
            //计数需要根据站点设置进行处理
            var setting = siteSettings.Get();
            CountService countService = new CountService(comment.TenantTypeId);

            if (eventArgs.EventOperationType == EventOperationType.Instance().Create())
            {
                //新审核状态对外显示则增加计数
                if ((int)eventArgs.NewAuditStatus >= (int)setting.AuditStatus)
                {
                    //对象评论计数
                    countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, 1, true);
                    if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                    {
                        var thread = threadService.Get(comment.CommentedObjectId);
                        if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                        {
                            //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                            new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, 1, true);
                        }
                    }
                }

                if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                {
                    //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread(), comment.ApprovalStatus));
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread()));
                }

                //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null, comment.ApprovalStatus));
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null));

                if (commentParent != null)
                {
                    //维护 子集个数和用户评论计数
                    commentParent.ChildrenCount++;
                    commentService.Update(commentParent);
                }
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Approved())
            {
                //旧的审核状态不对外显示 则增加计数
                if ((int)eventArgs.OldAuditStatus < (int)setting.AuditStatus)
                {
                    //对象评论计数 如果是父级评论 需要变更的计数为对外显示的子评论数+1
                    if (comment.ChildrenCount > 0)
                    {
                        //对外显示的子评论数量
                        var auditChildrenCount = commentService.GetDescendantChildrenCount(comment.Id);
                        countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, auditChildrenCount + 1, true);
                        if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                        {
                            var thread = threadService.Get(comment.CommentedObjectId);
                            if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                            {
                                //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                                new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, auditChildrenCount + 1, true);
                            }
                        }
                    }
                    else
                    {
                        countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, 1, true);
                        if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                        {
                            var thread = threadService.Get(comment.CommentedObjectId);
                            if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                            {
                                //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                                new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, 1, true);
                            }
                        }
                    }
                    if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                    {
                        var thread = threadService.Get(comment.CommentedObjectId);
                        if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                        {
                            //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                            new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, 1, true);
                        }
                    }
                }

                if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                {
                    //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread(), eventArgs.OldAuditStatus), -1);
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread(), eventArgs.NewAuditStatus));
                }
                //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null, eventArgs.OldAuditStatus), -1);
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null, eventArgs.NewAuditStatus));
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Disapproved())
            {
                //旧的审核状态对外显示 则减少计数
                if ((int)eventArgs.OldAuditStatus >= (int)setting.AuditStatus)
                {
                    //对象评论计数 如果是父级评论 需要变更的计数为对外显示的子评论数+1
                    if (comment.ChildrenCount > 0)
                    {
                        //对外显示的子评论数量
                        var auditChildrenCount = commentService.GetDescendantChildrenCount(comment.Id);
                        countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, -auditChildrenCount - 1, true);
                        if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                        {
                            var thread = threadService.Get(comment.CommentedObjectId);
                            if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                            {
                                //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                                new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, -auditChildrenCount - 1, true);
                            }
                        }
                    }
                    else
                    {
                        countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, -1, true);
                        if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                        {
                            var thread = threadService.Get(comment.CommentedObjectId);
                            if ((int)thread.ApprovalStatus >= (int)setting.AuditStatus)
                            {
                                //如果是回帖 且帖子是对外显示的 更新贴吧的评论计数
                                new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, -1, true);
                            }
                        }
                    }
                }
                //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread(), eventArgs.OldAuditStatus), -1);
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null, eventArgs.OldAuditStatus), -1);
            }
            else if (eventArgs.EventOperationType == EventOperationType.Instance().Delete())
            {
                //对象评论计数 如果是父级评论 需要变更的计数为对外显示的子评论数+1
                if (comment.ChildrenCount > 0)
                {
                    //对外显示的子评论数量
                    var auditChildrenCount = commentService.GetDescendantChildrenCount(comment.Id);
                    countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, -auditChildrenCount - 1, true);
                    if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                    {
                        var thread = threadService.Get(comment.CommentedObjectId);
                        //如果是回帖 更新贴吧的评论计数
                        new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, -auditChildrenCount - 1, true);
                    }
                }
                else
                {
                    countService.ChangeCount(CountTypes.Instance().CommentCount(), comment.CommentedObjectId, comment.OwnerId, -1, true);
                    if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                    {
                        var thread = threadService.Get(comment.CommentedObjectId);
                        //如果是回帖 更新贴吧的评论计数
                        new CountService(TenantTypeIds.Instance().Section()).ChangeCount(CountTypes.Instance().CommentCount(), thread.SectionId, thread.SectionId, -1, true);
                    }
                }

                //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                if (comment.TenantTypeId == TenantTypeIds.Instance().Thread())
                {
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread(), comment.ApprovalStatus), -1);
                    kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, TenantTypeIds.Instance().Thread()), -1);
                }
                //计数分为两个一个为个人空间计数,一个为发布时候的每个审核状态 计数
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null, comment.ApprovalStatus), -1);
                kvStore.Increase(KvKeys.Instance().UserCommentCount(comment.UserId, null), -1);

                if (commentParent != null)
                {
                    //维护 子集个数和用户评论计数
                    commentParent.ChildrenCount--;
                    commentService.Update(commentParent);
                }
                new CommentRepository().Delete(comment.Id);
            }
        }

        #region 评论增量索引

        /// <summary>
        /// 评论增量索引
        /// </summary>
        /// <param name="comment"></param>
        /// <param name="eventArgs"></param>
        private void CommentIndex_After(Comment comment, CommonEventArgs eventArgs)
        {
            if (comment == null)
            {
                return;
            }

            //添加索引
            if (eventArgs.EventOperationType == EventOperationType.Instance().Create())
            {
                kvStore.Append(KvKeys.Instance().CommentSearch(), comment.Id);
            }

            //删除索引
            if (eventArgs.EventOperationType == EventOperationType.Instance().Delete())
            {
                kvStore.Append(KvKeys.Instance().CommentDeleteSerach(), comment.Id);
            }

            //更新索引
            if (eventArgs.EventOperationType == EventOperationType.Instance().Update())
            {
                kvStore.Append(KvKeys.Instance().CommentUpdateSearch(), comment.Id);
            }
        }

        #endregion 评论增量索引
    }
}