﻿//------------------------------------------------------------------------------
// <copyright company="Tunynet">
//     Copyright (c) Tunynet Inc.  All rights reserved.
// </copyright>
//------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using Tunynet.Common;

namespace Tunynet.Spacebuilder
{
    public class ManageTenantTypesService
    {
        #region 构造器

        private IManageTenantTypesRepository tenantTypeRepository;

        /// <summary>
        /// 构造器
        /// </summary>
        /// <param name="userDataRepository">用户数据仓储</param>
        public ManageTenantTypesService(IManageTenantTypesRepository tenantTypeRepository)
        {
            this.tenantTypeRepository = tenantTypeRepository;
        }

        #endregion 构造器  

        /// <summary>
        /// 更新租户信息
        /// </summary>
        /// <param name="model"></param>
        public int Update(TenantTypeEditModel model)
        {
            var tenantTypeInfo = GetTenantTypeInfo(model.TenantTypeId);

            var tenantType = new TenantType()
            {
                TenantTypeId = model.TenantTypeId,
                Name = model.Name,
                ClassType = model.ClassType ?? "",
            };

            if (model.IsInDatabase)
            {
                tenantTypeRepository.Update(tenantType);
            }
            else
            {
                tenantTypeRepository.Insert(tenantType);
            }

            var lineCount = tenantTypeRepository.UpdateTenantTypeInService(model.TenantTypeId, model.ServiceKeys);
            return lineCount + 1;
        }

        /// <summary>
        /// 删除租户信息
        /// </summary>
        /// <param name="tenantTypeId"></param>
        /// <returns></returns>
        public int Delete(string tenantTypeId)
        {
            var count = tenantTypeRepository.DeleteTenantType(tenantTypeId);
            return count;
        }

        /// <summary>
        /// 根据租户Id租户信息
        /// </summary>
        /// <param name="tenantTypeId"></param>
        /// <returns></returns>
        public TenantTypeEditModel GetTenantTypeInfo(string tenantTypeId)
        {
            var tenantTypes = GetTenantTypesInDatabase(tenantTypeId);
            var tenantTypesInCode = GetTenantTypesInCode(tenantTypeId).FirstOrDefault();

            var tenantTypeEditModel = new TenantTypeEditModel();
            tenantTypeEditModel.TenantTypeId = tenantTypeId;
            tenantTypeEditModel.IsInDatabase = tenantTypes != null && tenantTypes.Any();
            if (tenantTypes.FirstOrDefault() != null)
            {
                tenantTypeEditModel.Name = tenantTypes.FirstOrDefault().Name;
            }
            else
            {
                tenantTypeEditModel.Name = tenantTypesInCode.Key != null ? tenantTypesInCode.Value : "";
            }
            tenantTypeEditModel.ClassType = tenantTypes.FirstOrDefault()?.ClassType;
            tenantTypeEditModel.ServiceKeys = tenantTypes?.Select(n => n.ServiceKey);
            tenantTypeEditModel.ServiceKey = "请使用ServiceKeys属性";
            tenantTypeEditModel.IsInCode = tenantTypesInCode.Key != null;
            return tenantTypeEditModel;
        }

        /// <summary>
        /// 获取所有的租户
        /// </summary>
        /// <returns></returns>
        public IEnumerable<TenantType> Gets()
        {
            return tenantTypeRepository.GetAll();
        }

        /// <summary>
        /// 获取数据库里面的租户及租户所关联到的服务
        /// </summary>
        /// <param name="tenantTypeId">租户Id</param>
        /// <param name="serviceKey">服务</param>
        /// <returns></returns>
        public IEnumerable<TenantTypeEditModel> GetTenantTypesInDatabase(string tenantTypeId = "", string serviceKey = "")
        {
            return tenantTypeRepository.Gets(tenantTypeId, serviceKey);
        }

        /// <summary>
        /// 获取代码中存在的所有租户
        /// </summary>
        /// <returns>TenantTypeId,Name字典</returns>
        public Dictionary<string, string> GetTenantTypesInCode(string tenantTypeId = "")
        {
            Dictionary<string, string> tenantTypesInCode = new Dictionary<string, string>();
            List<MethodInfo> methods = new List<MethodInfo>();

            Type type = typeof(TenantTypeIds);
            //获取构造函数
            ConstructorInfo constructor = type.GetConstructor(BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic, null, Type.EmptyTypes, null);
            //执行构造函数获得类的实例
            object classObject = constructor.Invoke(null);
            MethodInfo[] tenantTypeMethods = type.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly);

            //获取所有的方法
            if (tenantTypeMethods.Count() > 0)
            {
                //遍历方法并执行，获取返回结果
                foreach (var method in tenantTypeMethods)
                {
                    object value = method.Invoke(classObject, null);

                    var attribute = method.GetCustomAttributes(typeof(System.ComponentModel.DataAnnotations.DisplayAttribute), false)
                        .Cast<System.ComponentModel.DataAnnotations.DisplayAttribute>()
                        .FirstOrDefault();

                    tenantTypesInCode.Add(value as string, attribute?.Name ?? "没写Display");
                }
            }

            Type typeSpacebuilder = typeof(global::Tunynet.Spacebuilder.TenantTypeIdsExtension);
            methods.AddRange(typeSpacebuilder.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));

            //Type typeAsk = typeof(global::Spacebuilder.Ask.TenantTypeIdsExtension);
            Type typeAsk = Type.GetType("Spacebuilder.Ask.TenantTypeIdsExtension, Spacebuilder.Ask");
            if (typeAsk != null)
            {
                methods.AddRange(typeAsk.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));
            }
            //Type typeDoc = typeof(global::Spacebuilder.Doc.TenantTypeIdsExtension);
            Type typeDoc = Type.GetType("Spacebuilder.Doc.TenantTypeIdsExtension, Spacebuilder.Doc");
            if (typeDoc != null)
            {
                methods.AddRange(typeDoc.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));
            }

            //Type typeEvent = typeof(global::Spacebuilder.Event.TenantTypeIdsExtension);
            Type typeEvent = Type.GetType("Spacebuilder.Event.TenantTypeIdsExtension, Spacebuilder.Event");
            if (typeEvent != null)
            {
                methods.AddRange(typeEvent.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));
            }

            //Type typePointMall = typeof(global::Spacebuilder.PointMall.TenantTypeIdsExtension);
            Type typePointMall = Type.GetType("Spacebuilder.PointMall.TenantTypeIdsExtension, Spacebuilder.PointMall");
            if (typePointMall != null)
            {
                methods.AddRange(typePointMall.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));
            }

            //Type typeVote = typeof(global::Spacebuilder.Vote.TenantTypeIdsExtension);
            Type typeVote = Type.GetType("Spacebuilder.Vote.TenantTypeIdsExtension, Spacebuilder.Vote");
            if (typeVote != null)
            {
                methods.AddRange(typeVote.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly | BindingFlags.Static));
            }

            //获取所有的方法
            if (methods.Count() > 0)
            {
                //遍历方法并执行，获取返回结果
                foreach (var method in methods)
                {
                    object value = method.Invoke(null, new object[] { classObject });
                    var attribute = method.GetCustomAttributes(typeof(System.ComponentModel.DataAnnotations.DisplayAttribute), false)
                         .Cast<System.ComponentModel.DataAnnotations.DisplayAttribute>()
                         .FirstOrDefault();

                    tenantTypesInCode.Add(value as string, attribute?.Name ?? "没写Display");
                }
            }
            if (!string.IsNullOrEmpty(tenantTypeId))
            {
                var returnValue = new Dictionary<string, string>();
                var keyValuePair = tenantTypesInCode.Where(n => n.Key == tenantTypeId).FirstOrDefault();
                if (keyValuePair.Key != null)
                {
                    returnValue.Add(keyValuePair.Key, keyValuePair.Value);
                }
                return returnValue;
            }

            return tenantTypesInCode;
        }

        /// <summary>
        /// 获取代码中存在的所有服务
        /// </summary>
        /// <returns></returns>
        public Dictionary<string, string> GetServicesInCode()
        {
            Dictionary<string, string> servicesInCode = new Dictionary<string, string>();

            Type type = typeof(MultiTenantServiceKeys);
            GetReturnValues(type, servicesInCode);

            return servicesInCode;
        }

        /// <summary>
        /// 获取方法的返回值
        /// </summary>
        /// <param name="type"></param>
        /// <param name="returnValues"></param>
        private void GetReturnValues(Type type, Dictionary<string, string> returnValues)
        {
            //获取构造函数
            ConstructorInfo constructor = type.GetConstructor(BindingFlags.Instance | BindingFlags.Public | BindingFlags.NonPublic, null, Type.EmptyTypes, null);
            //执行构造函数获得类的实例
            object classObject = constructor.Invoke(null);
            //获取所有的方法
            MethodInfo[] methods = type.GetMethods(BindingFlags.Instance | BindingFlags.Public | BindingFlags.DeclaredOnly);
            if (methods.Count() > 0)
            {
                //遍历方法并执行，获取返回结果
                foreach (var method in methods)
                {
                    object value = method.Invoke(classObject, null);
                    var attribute = method.GetCustomAttributes(typeof(System.ComponentModel.DataAnnotations.DisplayAttribute), false)
                       .Cast<System.ComponentModel.DataAnnotations.DisplayAttribute>()
                       .FirstOrDefault();

                    returnValues.Add(value as string, attribute?.Name ?? "没写Display");
                }
            }
        }

        ///// <summary>
        ///// 根据租户Id获取租户实体
        ///// </summary>
        ///// <param name="tenantTypeId"></param>
        ///// <returns></returns>
        //public IEnumerable<TenantTypeEditModel> GetTenantTypeById(string tenantTypeId = null)
        //{
        //    return tenantTypeRepository.GetTenantTypeById(tenantTypeId);
        //}
    }
}
