﻿using System;
using System.Web;
using System.Web.SessionState;
using System.Web.UI;
using System.Text;
using System.Web.UI.HtmlControls;

namespace Bice.Refresh
{
    public class FeedbackBasePage : System.Web.UI.Page
    {
        // ***********************************************************
        // Ctor
        public FeedbackBasePage()
        {
            // Register a Load handler
            this.Load += new EventHandler(FeedbackBasePage_Load);
        }
        // ***********************************************************

        // ***********************************************************
        // Constants
        public const string BodyId = "Body";
        // ***********************************************************

        // ***********************************************************
        // Public members
        public string TargetUrl = "";
        // ***********************************************************

        // ***********************************************************
        // Inject auto-redirect code in the onload DHTML event
        public string GetAutoRedirectScript()
        {
            return String.Format("location.href='{0}';", TargetUrl);
        }
        // ***********************************************************

        // ***********************************************************
        // Initialize the page
        private void FeedbackBasePage_Load(object sender, EventArgs e)
        {
            // Capture the name of the target page 
            if (Request.QueryString[LengthyAction.TargetPageId] != null)
            {
                TargetUrl = Request.QueryString[LengthyAction.TargetPageId].ToString();
            }


            // If the <body> is runat=server and named Body, injects
            // minimum script code automatically. If you want to do more 
            // sophisticated things, just change the ID of the <body>
            // or avoid the runat=server attribute. Use the GetAutoRedirectScript
            // method to get the auto-redirect script code you anyway
            // need. 
            HtmlGenericControl body = FindControl(BodyId) as HtmlGenericControl;
            if (body != null)
            {
                body.Attributes["onload"] = GetAutoRedirectScript();
            }

        }
        // ***********************************************************
    }
    public class LengthyAction
    {
        // ***********************************************************
        // Constants
        private const string UrlFormatString = "{0}?{1}={2}";
        public const string TargetPageId = "target";
        public const string PageOperationEntry = "IsLengthyOperation";
        // ***********************************************************

        // ***********************************************************
        // Start a lengthy operation on an external page using an 
        // intermediate feedback page to keep the user informed 
        public static void Start(string feedbackPageUrl, string targetPageUrl)
        {
            // Any parameter MUST be appended to the query string of 
            // the target page--the page that contains the lengthy
            // operation. In alternative, use Session.

            // Prepare the URL for the feedback page
            string url = String.Format(UrlFormatString,
                feedbackPageUrl, TargetPageId, targetPageUrl);

            // Redirect the call to the feedback page
            HttpContext.Current.Response.Redirect(url);
        }
        // ***********************************************************
    }
    public class Page : System.Web.UI.Page
    {
        #region Constants

        // ***********************************************************
        // Constants
        public const string RefreshTicketCounter = "RefreshTicketCounter";
        private const string SetFocusFunctionName = "__setFocus";
        private const string SetFocusScriptName = "__inputFocusHandler";
        // ***********************************************************

        #endregion


        // ***********************************************************
        // Ctor
        public Page()
        {
            m_focusedControl = "";

            // Register a PreRender handler
            this.PreRender += new EventHandler(RefreshPage_PreRender);
        }
        // ***********************************************************

        // **************************************************************
        // Indicates if the page is being viewed in response to F5 hit
        public bool IsPageRefresh
        {
            get
            {
                object o = HttpContext.Current.Items[RefreshAction.PageRefreshEntry];
                if (o == null)
                    return false;
                return (bool)o;
            }
        }
        // **************************************************************

        // **************************************************************
        // Increase the internal counter used to generate refresh tickets
        public void TrackRefreshState()
        {
            InitRefreshState();
            int ticket = Convert.ToInt32(Session[RefreshTicketCounter]) + 1;
            Session[RefreshTicketCounter] = ticket;
        }
        // **************************************************************

        // **************************************************************
        // Set the control with the input focus
        public void SetFocus(string ctlId)
        {
            m_focusedControl = ctlId;
        }
        // **************************************************************


        #region Private Members

        // **************************************************************
        // Create the hidden field to store the current request ticket
        private void SaveRefreshState()
        {
            int ticket = Convert.ToInt32(Session[RefreshTicketCounter]) + 1;
            RegisterHiddenField(RefreshAction.CurrentRefreshTicketEntry, ticket.ToString());
        }
        // **************************************************************

        // **************************************************************
        // Ensure that the ticket counter is initialized
        private void InitRefreshState()
        {
            if (Session[RefreshTicketCounter] == null)
                Session[RefreshTicketCounter] = 0;
        }
        // **************************************************************

        // **************************************************************
        // Handle the PreRender event
        private void RefreshPage_PreRender(object sender, EventArgs e)
        {
            SaveRefreshState();
            AddSetFocusScript();
        }
        // **************************************************************

        // **************************************************************
        // Add any script code required for the SetFocus feature
        private void AddSetFocusScript()
        {
            if (m_focusedControl == "")
                return;

            // Add the script to declare the function
            // (Only one form in ASP.NET pages)
            StringBuilder sb = new StringBuilder("");
            sb.Append("<script language=javascript>");
            sb.Append("function ");
            sb.Append(SetFocusFunctionName);
            sb.Append("(ctl) {");
            sb.Append("  if (document.forms[0][ctl] != null)");
            sb.Append("  {document.forms[0][ctl].focus();}");
            sb.Append("}");

            // Add the script to call the function
            sb.Append(SetFocusFunctionName);
            sb.Append("('");
            sb.Append(m_focusedControl);
            sb.Append("');<");
            sb.Append("/");   // break like this to avoid misunderstandings...
            sb.Append("script>");

            // Register the script (names are CASE-SENSITIVE)
            if (!IsStartupScriptRegistered(SetFocusScriptName))
                RegisterStartupScript(SetFocusScriptName, sb.ToString());
        }
        // **************************************************************

        #endregion


        #region Private Properties
        // ***********************************************************
        // Private properties
        private string m_focusedControl;
        // ***********************************************************
        #endregion
    }
    public class RefreshAction
    {
        // ***********************************************************
        // Constants
        public const string LastRefreshTicketEntry = "__LASTREFRESHTICKET";
        public const string CurrentRefreshTicketEntry = "__CURRENTREFRESHTICKET";
        public const string PageRefreshEntry = "IsPageRefresh";
        // ***********************************************************

        // ***********************************************************
        // Manage to check if the F5 button has been pressed
        public static void Check(HttpContext ctx)
        {
            // Initialize the ticket slot
            EnsureRefreshTicket(ctx);

            // Read the last ticket served in the session (from Session)
            int lastTicket = GetLastRefreshTicket(ctx);

            // Read the ticket of the current request (from a hidden field)
            int thisTicket = GetCurrentRefreshTicket(ctx);

            // Compare tickets
            if (thisTicket > lastTicket ||
                (thisTicket == lastTicket && thisTicket == 0))
            {
                UpdateLastRefreshTicket(ctx, thisTicket);
                ctx.Items[PageRefreshEntry] = false;
            }
            else
            {
                ctx.Items[PageRefreshEntry] = true;
            }
        }
        // ***********************************************************

        // ***********************************************************
        // Ensure that the slot for the last ticket served isn't null
        private static void EnsureRefreshTicket(HttpContext ctx)
        {
            // Initialize the session slots for the page (Ticket) and the module (LastTicketServed)
            if (ctx.Session[LastRefreshTicketEntry] == null)
                ctx.Session[LastRefreshTicketEntry] = 0;
        }
        // ***********************************************************

        // ***********************************************************
        // Return the ticket of the last request in the session
        private static int GetLastRefreshTicket(HttpContext ctx)
        {
            // Extract and return the last ticket
            return Convert.ToInt32(ctx.Session[LastRefreshTicketEntry]);
        }
        // ***********************************************************

        // ***********************************************************
        // Return the ticket of the last request in the session
        private static int GetCurrentRefreshTicket(HttpContext ctx)
        {
            return Convert.ToInt32(ctx.Request[CurrentRefreshTicketEntry]);
        }
        // ***********************************************************

        // ***********************************************************
        // Return the ticket of the last request in the session
        private static void UpdateLastRefreshTicket(HttpContext ctx, int ticket)
        {
            ctx.Session[LastRefreshTicketEntry] = ticket;
        }
        // ***********************************************************

    }
    public class RefreshModule : IHttpModule
    {
        // ***********************************************************
        // IHttpModule::Init
        public void Init(HttpApplication app)
        {
            // Register for pipeline events
            app.AcquireRequestState += new EventHandler(this.OnAcquireRequestState);
        }
        // ***********************************************************

        // ***********************************************************
        // IHttpModule::Dispose
        public void Dispose()
        {
        }
        // ***********************************************************


        // ***********************************************************
        // Determine if a F5 or back/fwd action is in course
        private void OnAcquireRequestState(object sender, EventArgs e)
        {
            // Get access to the HTTP context 
            HttpApplication app = (HttpApplication)sender;
            HttpContext ctx = app.Context;

            // Check F5 action
            RefreshAction.Check(ctx);

            return;
        }
        // ***********************************************************
    }
}
