package org.ssssssss.magicapi.adapter.resource;

import org.ssssssss.magicapi.adapter.Resource;
import org.ssssssss.magicapi.adapter.ResourceAdapter;
import org.ssssssss.magicapi.utils.IoUtils;
import org.ssssssss.magicapi.utils.PathUtils;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;

public class JarResource implements Resource {

	private final JarFile jarFile;

	private final ZipEntry entry;

	private final List<JarEntry> entries;

	private final String entryName;

	private JarResource parent = this;

	private final boolean inSpringBoot;

	public JarResource(JarFile jarFile, String entryName, List<JarEntry> entries, boolean inSpringBoot) {
		this.jarFile = jarFile;
		this.entryName = entryName;
		this.inSpringBoot = inSpringBoot;
		this.entry = getEntry(this.entryName);
		this.entries = entries;
	}

	public JarResource(JarFile jarFile, String entryName, List<JarEntry> entries, JarResource parent, boolean inSpringBoot) {
		this(jarFile, entryName, entries, inSpringBoot);
		this.parent = parent;
	}

	@Override
	public boolean readonly() {
		return true;
	}

	@Override
	public byte[] read() {
		try {
			return IoUtils.bytes(this.jarFile.getInputStream(entry));
		} catch (IOException e) {
			return new byte[0];
		}
	}

	@Override
	public boolean isDirectory() {
		return this.entry.isDirectory();
	}

	@Override
	public boolean exists() {
		return this.entry != null;
	}

	protected ZipEntry getEntry(String name) {
		if (inSpringBoot && name.startsWith(ResourceAdapter.SPRING_BOOT_CLASS_PATH)) {
			name = name.substring(ResourceAdapter.SPRING_BOOT_CLASS_PATH.length());
		}
		return this.jarFile.getEntry(name);
	}

	@Override
	public Resource getResource(String name) {
		String entryName = PathUtils.replaceSlash(this.entryName + "/" + name);
		String prefix = PathUtils.replaceSlash(entryName + "/");
		return new JarResource(this.jarFile, entryName, entries.stream()
				.filter(it -> it.getName().startsWith(prefix))
				.collect(Collectors.toList()), this, this.inSpringBoot);
	}

	@Override
	public String name() {
		int index = this.entryName.lastIndexOf("/");
		return index > -1 ? this.entryName.substring(index) : this.entryName;
	}

	@Override
	public Resource parent() {
		return this.parent;
	}

	@Override
	public List<Resource> dirs() {
		return resources().stream().filter(Resource::isDirectory).collect(Collectors.toList());
	}

	@Override
	public List<Resource> files(String suffix) {
		return this.entries.stream().filter(it -> it.getName().endsWith(suffix))
				.map(entry -> new JarResource(jarFile, entry.getName(), Collections.emptyList(), this.inSpringBoot))
				.collect(Collectors.toList());
	}

	@Override
	public List<Resource> resources() {
		String prefix = PathUtils.replaceSlash(this.entryName + "/");
		return entries.stream()
				.filter(it -> it.getName().startsWith(prefix))
				.map(entry -> new JarResource(jarFile, entry.getName(), entries.stream()
						.filter(item -> item.getName().startsWith(PathUtils.replaceSlash(entry.getName() + "/")))
						.collect(Collectors.toList()), this.inSpringBoot)
				)
				.collect(Collectors.toList());
	}

	@Override
	public String getAbsolutePath() {
		return this.jarFile.getName() + "/" + this.entryName;
	}

	@Override
	public String toString() {
		return String.format("jar://%s", this.entryName);
	}
}
