package config

import (
	"fmt"
	"os"
	"path/filepath"
	"time"

	"github.com/parnurzeal/gorequest"
	"golang.org/x/xerrors"
)

// ExploitConf is exploit config
type ExploitConf struct {
	// DB type for exploit dictionary (sqlite3, mysql, postgres or redis)
	Type string

	// http://exploit-dictionary.com:1324 or DB connection string
	URL string `json:"-"`

	// /path/to/exploit.sqlite3
	SQLite3Path string `json:"-"`
}

func (cnf *ExploitConf) setDefault() {
	if cnf.Type == "" {
		cnf.Type = "sqlite3"
	}
	if cnf.URL == "" && cnf.SQLite3Path == "" {
		wd, _ := os.Getwd()
		cnf.SQLite3Path = filepath.Join(wd, "go-exploitdb.sqlite3")
	}
}

const exploitDBType = "EXPLOITDB_TYPE"
const exploitDBURL = "EXPLOITDB_URL"
const exploitDBPATH = "EXPLOITDB_SQLITE3_PATH"

// Init set options with the following priority.
// 1. Environment variable
// 2. config.toml
func (cnf *ExploitConf) Init() {
	if os.Getenv(exploitDBType) != "" {
		cnf.Type = os.Getenv(exploitDBType)
	}
	if os.Getenv(exploitDBURL) != "" {
		cnf.URL = os.Getenv(exploitDBURL)
	}
	if os.Getenv(exploitDBPATH) != "" {
		cnf.SQLite3Path = os.Getenv(exploitDBPATH)
	}
	cnf.setDefault()
}

// IsFetchViaHTTP returns wether fetch via http
func (cnf *ExploitConf) IsFetchViaHTTP() bool {
	return Conf.Exploit.Type == "http"
}

// CheckHTTPHealth do health check
func (cnf *ExploitConf) CheckHTTPHealth() error {
	if !cnf.IsFetchViaHTTP() {
		return nil
	}

	url := fmt.Sprintf("%s/health", cnf.URL)
	resp, _, errs := gorequest.New().Timeout(10 * time.Second).Get(url).End()
	//  resp, _, errs = gorequest.New().SetDebug(config.Conf.Debug).Get(url).End()
	//  resp, _, errs = gorequest.New().Proxy(api.httpProxy).Get(url).End()
	if 0 < len(errs) || resp == nil || resp.StatusCode != 200 {
		return xerrors.Errorf("Failed to connect to exploit server. url: %s, errs: %s", url, errs)
	}
	return nil
}
