
#include "luat_base.h"
#include "luat_luadb.h"
#include "luat_malloc.h"

#define LUAT_LOG_TAG "luadb"
#include "luat_log.h"

//---
static int readU8(const char* ptr, int *index) {
    int val = ptr[*index];
    *index = (*index) + 1;
    return val & 0xFF;
}

static int readU16(const char* ptr, int *index) {
    return readU8(ptr,index) + (readU8(ptr,index) << 8);
}

static int readU32(const char* ptr, int *index) {
    return readU16(ptr,index) + (readU16(ptr,index) << 16);
}
//---


int luat_luadb_umount(luadb_fs_t *fs) {
    if (fs)
        luat_heap_free(fs);
    return 0;
}

int luat_luadb_remount(luadb_fs_t *fs, unsigned flags) {
    memset(fs->fds, 0, sizeof(luadb_fd_t)*LUAT_LUADB_MAX_OPENFILE);
    return 0;
}

int luat_luadb_open(luadb_fs_t *fs, const char *path, int flags, int /*mode_t*/ mode) {
    LLOGD("open luadb path = %s flags=%d", path, flags);
    // if (flags & 0x3) { // xie 
    //     return -1;
    // }
    for (size_t i = 0; i < fs->filecount; i++)
    {
        if (!strcmp(path, fs->files[i].name)) {
            for (size_t j = 0; j < LUAT_LUADB_MAX_OPENFILE; j++)
            {
                if (fs->fds[j].file == NULL) {
                    fs->fds[j].fd_pos = 0;
                    fs->fds[j].file = &(fs->files[i]);
                    return j;
                }
            }
            
        }
    }
    return -1;
}


int luat_luadb_close(luadb_fs_t *fs, int fd) {
    if (fd < 0 || fd >= LUAT_LUADB_MAX_OPENFILE)
        return -1;
    if (fs->fds[fd].file != NULL) {
        fs->fds[fd].file = NULL;
        fs->fds[fd].fd_pos = 0;
        return 0;
    }
    return -1;
}

size_t luat_luadb_read(luadb_fs_t *fs, int fd, void *dst, size_t size) {
    if (fd < 0 || fd >= LUAT_LUADB_MAX_OPENFILE || fs->fds[fd].file == NULL)
        return 0;
    luadb_fd_t *fdt = &fs->fds[fd];
    int re = size;
    if (fdt->fd_pos + size > fdt->file->size) {
        re = fdt->file->size - fdt->fd_pos;
    }
    if (re > 0) {
        memcpy(dst, fdt->file->ptr + fdt->fd_pos, re);
        fdt->fd_pos += re;
    }
    LLOGD("luadb read name %s offset %d size %d ret %d", fdt->file->name, fdt->fd_pos, size, re);
    return re;
}

long luat_luadb_lseek(luadb_fs_t *fs, int fd, long /*off_t*/ offset, int mode) {
    if (fd < 0 || fd >= LUAT_LUADB_MAX_OPENFILE || fs->fds[fd].file == NULL)
        return -1;
    if (mode == SEEK_END) {
        fs->fds[fd].fd_pos = fs->fds[fd].file->size;
    }
    else if (mode == SEEK_CUR) {
        fs->fds[fd].fd_pos += offset;
    }
    else {
        fs->fds[fd].fd_pos = offset;
    }
    return fs->fds[fd].fd_pos;
}

// int luat_luadb_fstat(void *fs, int fd, struct stat *st) {
//     luadb_fs_t *_fs = (luadb_fs_t*)fs;
//     if (fd < 0 || fd >= LUADB2_MAX_OPENFILE || _fs->fds[fd].file == NULL)
//         return -1;
//     st->st_size = _fs->fds[fd].file->size;
//     return 0;
// }

luadb_file_t * luat_luadb_stat(luadb_fs_t *fs, const char *path) {
    for (size_t i = 0; i < fs->filecount; i++)
    {
        if (!strcmp(path, fs->files[i].name)) {
            return &fs->files[i];
        }
    }
    return NULL;
}

luadb_fs_t* luat_luadb_mount(const char* ptr) {
    int index = 0;
    int headok = 0;
    int dbver = 0;
    int headsize = 0;
    int filecount = 0;

    for (size_t i = 0; i < 128; i++)
    {
        int type = readU8(ptr, &index);
        int len = readU8(ptr, &index);
        LLOGD("PTR: %d %d %d", type, len, index);
        switch (type) {
            case 1: {// Magic, 肯定是4个字节
                if (len != 4) {
                    LLOGD("Magic len != 4");
                    goto _after_head;
                }
                if (ptr[index++] != 0x5A || ptr[index++] != 0xA5 || ptr[index++] != 0x5A || ptr[index++] != 0xA5 ) {
                    LLOGD("Magic not match");
                    goto _after_head;
                }
                break;
            }
            case 2: {
                if (len != 2) {
                    LLOGD("Version len != 2");
                    goto _after_head;
                }
                dbver = readU16(ptr, &index);
                LLOGD("LuaDB version = %d", dbver);
                break;
            }
            case 3: {
                if (len != 4) {
                    LLOGD("Header full len != 4");
                    goto _after_head;
                }
                headsize = readU32(ptr, &index);
                break;
            }
            case 4 : {
                if (len != 2) {
                    LLOGD("Lua File Count len != 4");
                    goto _after_head;
                }
                filecount = readU16(ptr, &index);
                LLOGD("LuaDB file count %d", filecount);
                break;
            }
            case 0xFE : {
                if (len != 2) {
                    LLOGD("CRC len != 4");
                    goto _after_head;
                }
                index += len;
                headok = 1;
                goto _after_head;
            }
            default: {
                index += len;
                LLOGD("skip unkown type %d", type);
                break;
            }
        }
    }

_after_head:

    if (headok == 0) {
        LLOGD("Bad LuaDB");
        return NULL;
    }
    if (dbver == 0) {
        LLOGD("miss DB version");
        return NULL;
    }
    if (headsize == 0) {
        LLOGD("miss DB headsize");
        return NULL;
    }
    if (filecount == 0) {
        LLOGD("miss DB filecount");
        return NULL;
    }
    if (filecount > 256) {
        LLOGD("too many file in LuaDB");
        return NULL;
    }

    LLOGD("LuaDB head seem ok");
    size_t msize = sizeof(luadb_fs_t) + filecount*sizeof(luadb_file_t);
    LLOGD("malloc fo luadb fs size=%d", msize);
    luadb_fs_t *fs = (luadb_fs_t*)luat_heap_malloc(msize);
    if (fs == NULL) {
        LLOGD("malloc for luadb fail!!!");
        return NULL;
    }
    memset(fs, 0, msize);
    LLOGD("LuaDB check files ....");

    fs->version = dbver;
    fs->filecount = filecount;
    //fs->ptrpos = initpos;

    int fail = 0;
    // 读取每个文件的头部
    for (size_t i = 0; i < filecount; i++)
    {
        
        LLOGD("LuaDB check files .... %d", i+1);
        
        int type = ptr[index++];
        int len = ptr[index++];
        if (type != 1 || len != 4) {
            LLOGD("bad file data 1 : %d %d %d", type, len, index);
            fail = 1;
            break;
        }
        // skip magic
        index += 4;

        // 2. 然后是名字
        type = ptr[index++];
        len = ptr[index++];
        if (type != 2) {
            LLOGD("bad file data 2 : %d %d %d", type, len, index);
            fail = 1;
            break;
        }
        // 拷贝文件名
        LLOGD("LuaDB file name len = %d", len);

        memcpy(fs->files[i].name, &(ptr[index]), len);

        fs->files[i].name[len] = 0x00;

        index += len;

        LLOGD("LuaDB file name %s", fs->files[i].name);

        // 3. 文件大小
        type = ptr[index++];
        len = ptr[index++];
        if (type != 3 || len != 4) {
            LLOGD("bad file data 3 : %d %d %d", type, len, index);
            fail = 1;
            break;
        }
        fs->files[i].size = readU32(ptr, &index);

        // 0xFE校验码
        type = ptr[index++];
        len = ptr[index++];
        if (type != 0xFE || len != 2) {
            LLOGD("bad file data 4 : %d %d %d", type, len, index);
            fail = 1;
            break;
        }
        // 校验码就跳过吧
        index += len;
        
        fs->files[i].ptr = index + ptr; // 绝对地址
        index += fs->files[i].size;

        LLOGD("LuaDB: %s %d", fs->files[i].name, fs->files[i].size);
    }

    if (fail == 0) {
        LLOGD("LuaDB check files .... ok");
        return fs;
    }
    else {
        LLOGD("LuaDB check files .... fail");
        luat_heap_free(fs);
        return NULL;
    }
}
