<?php
defined ( 'PHPNOW_DIR' ) or exit ( '非法访问' );
/**
 * 自动加载类
 * PHPnow中文应用框架 PHPnow framework 1.0
 * @copyright	(C) 2011-2012 PHPnow
 * @license		http://www.phpnow.cn
 * @author		jiaodu QQ:1286522207
 */
class PHPnowLoader {
	
	/**
	 * 默认文件扩展名
	 */
	const DEFAULT_EXTENSION = '.php';
	
	/**
	 * 追加库
	 */
	const POSITION_APPEND = 'append';
	
	/**
	 * 前置库
	 */
	const POSITION_PREPEND = 'prepend';
	
	/**
	 * 删除参数常数。使用命名空间时，将记录
	 */
	const POSITION_REMOVE = 'remove';
	
	/**
	 * @var array 包含所有注册的名称空间
	 */
	protected $namespaces = array ();
	
	/**
	 * @var array 包含所有加载的类
	 */
	protected $loadedClasses = array ();
	
	public function __construct() {
		if (version_compare ( PHP_VERSION, '5.3.0' ) >= 0) {
			spl_autoload_register ( array ($this, 'autoload' ), true, true );
		} else {
			spl_autoload_register ( array ($this, 'autoload' ), true );
		}
	}
	
	/**
	 * 类加载的名字。如果尚未加载类的类会加载
	 * 方法使用LoadFile。如果类装载，它将被添加到内部
	 * loadedClasses阵列。
	 *
	 * @param   string|array $class
	 * @param   string $path
	 * @return  bool
	 */
	public function loadClass($class, $path = null) {
		if (is_array ( $class )) {
			return min ( array_map ( array ($this, __METHOD__ ), $class ) );
		}
		if (! is_string ( $class )) {
			throw new PHPnowException ( Lang ( 'CLASS_NAME_STRING' ) );
		}
		if (! $this->isLoaded ( $class )) {
			if (! $path) {
				$path = $this->getClassPath ( $class );
			}
			if ($path) {
				$this->loadFile ( $path );
			}
		}
		if (! $this->isLoaded ( $class )) {
			return false;
		}
		
		$this->loadedClasses [] = $class;
		return true;
	}
	
	/**
	 * 加载文件的方法。如果不是可读的路径，输出缓冲不能被启动或
	 * 路径没有通过安全检查，该函数抛出一个异常。
	 * 否则包括路径，并返回结果。
	 *
	 * @param   string $path
	 * @return  mixed
	 */
	public static function loadFile($path) {
		if (! self::checkFile ( $path )) {
			throw new PHPnowException ( Lang ( 'CLASS_NAME_INCORRECT' ) );
		}
		if (! self::isReadable ( $path )) {
			throw new PHPnowException ( Lang ( 'PATH' ) . ' "' . $path . '"' . Lang ( 'FAILED_TO_LOAD' ) );
		}
		if (! ob_start ()) {
			throw new PHPnowException ( Lang ( 'BUFFER_FAILED' ) );
		}
		
		$result = include $path;
		ob_end_clean ();
		
		return $result;
	}
	
	/**
	 * 检查，如果该文件是可读的。 is_readable类如果失败，将使用功能
	 * 在stream_resolve_include_path功能。的情况下，stream_resolve_include_path
	 *
	 * @param   string $path
	 * @return  string|bool
	 */
	public static function isReadable($path) {
		if (is_readable ( $path )) {
			return $path;
		}
		
		if (function_exists ( 'stream_resolve_include_path' )) {
			$path = stream_resolve_include_path ( $path );
			if ($path !== false && is_readable ( $path )) {
				return $path;
			} else {
				return false;
			}
		}
		
		if (empty ( $path ) || $path {0} === '/' || $path {0} === DIRECTORY_SEPARATOR) {
			return false;
		}
		
		foreach ( self::explodeIncludePath () as $includePath ) {
			if ($includePath == '.') {
				continue;
			}
			$file = realpath ( $includePath . '/' . $path );
			if ($file && is_readable ( $file )) {
				return $file;
			}
		}
		
		return false;
	}
	
	/**
	 * PATH_SEPARATOR常数给定的路径
	 * @param   string $path
	 * @return  array
	 */
	public static function explodeIncludePath($path = null) {
		if (null === $path) {
			$path = get_include_path ();
		}
		
		if (PATH_SEPARATOR == ':') {
			$paths = preg_split ( '#:(?!//)#', $path );
		} else {
			$paths = explode ( PATH_SEPARATOR, $path );
		}
		return $paths;
	}
	
	/**
	 * 返回给定的类名称的路径。遍历所有命名空间
	 * @param   string $class
	 * @return  string|void
	 */
	public function getClassPath($class) {
		foreach ( $this->namespaces as $namespace ) {
			if (! strstr ( $class, $namespace ['namespace'] )) {
				continue;
			}
			$path = $namespace ['path'] . $class . $namespace ['extension'];
			$path = self::isReadable ( $path );
			if ($path) {
				return $path;
			}
		}
		//尝试自动找插件
		$path = array (APP_PATH . APP_PLUGINS_DIR_NAME . DS . $class . '.php', APP_PATH . APP_PLUGINS_DIR_NAME . DS . $class . '.class.php', PHPNOW_PLUGINS_DIR . $class . '.php', PHPNOW_PLUGINS_DIR . $class . '.class.php' );
		foreach ( $path as $p ) {
			$p = self::isReadable ( $p );
			if ($p) {
				return $p;
			}
		}
		return null;
	}
	
	/**
	 * 检查类或接口是否已加载
	 * @param   string $class
	 * @return  bool
	 */
	public static function isLoaded($class) {
		return class_exists ( $class, false ) || interface_exists ( $class, false );
	}
	
	/**
	 * 注册一个命名空间
	 * @param   string $namespace
	 * @param   string $path
	 * @param   string $separator
	 * @param   string $extension
	 * @param   string $position
	 * @return  PHPnowLoader
	 */
	public function registerNamespace($namespace, $path, $extension = self::DEFAULT_EXTENSION, $position = self::POSITION_APPEND) {
		$namespace = array ('namespace' => $namespace, 'path' => $path, 'extension' => $extension );
		
		switch ($position) {
			case self::POSITION_APPEND :
				array_push ( $this->namespaces, $namespace );
				break;
			case self::POSITION_PREPEND :
				array_unshift ( $this->namespaces, $namespace );
				break;
			default :
				break;
		}
		
		return $this;
	}
	
	/**
	 * 路径添加到包含路径的数组，并返回旧的包含路径。
	 * 指定的位置，在什么时候要添加数组中的include路径。
	 * @param   string $path
	 * @param   string $position
	 * @return  string
	 */
	public static function addIncludePath($path, $position = self::POSITION_APPEND) {
		if (is_array ( $path )) {
			return ( bool ) array_map ( __METHOD__, $path );
		}
		if (! is_string ( $path ) || ! file_exists ( $path ) || ! is_dir ( $path )) {
			throw new PHPnowException ( Lang ( 'PATH' ) . ' "' . $path . '" ' . Lang ( 'FAILED_TO_LOAD' ) );
		}
		
		$paths = self::explodeIncludePath ();
		
		if (($key = array_search ( $path, $paths )) !== false) {
			unset ( $paths [$key] );
		}
		
		switch ($position) {
			case self::POSITION_APPEND :
				array_push ( $paths, $path );
				break;
			case self::POSITION_PREPEND :
				array_unshift ( $paths, $path );
				break;
			default :
				break;
		}
		
		return self::setIncludePath ( $paths );
	}
	
	/**
	 * 设置include路径
	 * @param   string|array $path
	 * @return  string
	 */
	public static function setIncludePath($path) {
		if (is_array ( $path )) {
			$path = implode ( PATH_SEPARATOR, $path );
		}
		
		$old = set_include_path ( $path );
		if ($old !== $path && (! $old || $old == get_include_path ())) {
			throw new PHPnowException ( Lang ( 'PATH' ) . ' "' . $path . '" ' . Lang ( 'FAILED_TO_LOAD' ) );
		}
		
		return $old;
	}
	
	/**
	 * 返回的内部数组loadedClasses的，它包含了所有已经加载的类
	 * @return  array
	 */
	public function getLoadedClasses() {
		return $this->loadedClasses;
	}
	
	/**
	 * 自动加载
	 * @param   string $class
	 */
	public function autoload($class) {
		try {
			$this->loadClass ( $class );
		} catch ( Exception $e ) {
			throw new PHPnowException ( Lang ( 'CLASS' ) . ' "' . $class . '" ' . Lang ( 'FAILED_TO_LOAD' ) );
		}
	}
	
	/**
	 * 检查文件
	 * @param   string $path
	 * @return  bool
	 */
	public static function checkFile($path) {
		return ! preg_match ( '/[^a-z0-9\\/\\\\_. :-]/i', $path );
	}
}