<?php

defined('IS_SYS') or die('Access Denied.');
/****************************************************
* check.class.php 表单验证类
*
* @author lijun
* @version 2012-11-13 15:08:49
*
* -- 说明 --------------------------------
is_numeric 可验证带-+符号的数字
* -- 例子 --------------------------------
check::email("admin@uesns.com");
check::phone("(0751)81209172");
check::mobile("18755588411");
check::url("http://www.uesns.com");
check::ip("255.255.255.255");
check::ipv6("2001:0DB8:0:0:0:0:1428:0000");
check::username('uesns') && check::len('uesns', 5, 10);
****************************************************/
class Check {
	/**
	 * username 
	 * 匹配由数字、26个英文字母或者下划线组成的字符串
	 * check::username('uesns') && check::len('uesns', 3, 10)
	 */
	function username($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^\w+$/";
		return @preg_match($pattern, $val);
	}

	/**
	 * is_date ($val, $format='')
	 * 作用：检验日期的合法性
	 * 说明：默认的合法日期格式是以"-"分割的"年-月-日"
	 *       当参数 $format 设置格式时则按照这个格式检验
	 *       check::is_date('2006-12-1');
	 *       check::is_date('2006-12-1', 'Y-m-d h:i:s')
	 */
	function is_date($val, $format=''){
		if($format == ''){
			$val = explode("-", $val);
			return @checkdate( $val[1], $val[2], $val[0] );
		}else{
			return ( date($format, strtotime($val)) == $val );
		}
	}

	//时间检查
	function is_time($val, $format=''){
		if(empty($format)){
			$val = explode(":", $val);
			if(count($val)!=3) return false;
			for($i=0; $i<=2; $i++){
				if(!@preg_match("/^[0-9]{1,2}$/", $val[$i])) return false;
			}
			//小时>=24
			if($val[0]>=24) return false;
			//分和秒>59
			if($val[1]>59||$val[2]>59) return false;
			return true;
		}else{
			return ( date($format, strtotime($val)) == $val );
		}
	}

	//验证是否由数字组成,is_numeric()验证包括整数和浮点数
	function number($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^\d+$/";
		return @preg_match($pattern, $val);
	}

	//email
	function email($val) {
		$pattern = "/^[_.0-9a-z]+@[0-9a-z-]+(\.[0-9a-z-]+)*\.[a-z]{2,4}$/i";
		return @preg_match($pattern, $val);
	}

	/**
	 * url
	 * 只检查http开头形式
	 */
	function url($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^http[s]?:\/\/[A-Za-z0-9]+\.[A-Za-z0-9]+[\/=\?%\-&_~`@[\]\':+!]*([^<>\"])*$/";
		return @preg_match($pattern, $val);
	}

	//邮政编码
	function zipcode($str, $pattern=''){
		if (empty($pattern)) $pattern = "/^\d{6}$/";
		return @preg_match($pattern, $val);
	}

	/**
	 * 电话号码 phone
	 * 区号限定0开头 3~4位
	 * 号码限定非0开头 7~8位
	 * example:
	 *  号码 12345678
	 *  区号-号码 0123-12345678
	 *  (区号)号码 (0123)12345678
	 */
	function phone($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^((\(\d{3}\))|(\d{3}\-))?(\(0\d{2,3}\)|0\d{2,3}-)?[1-9]\d{6,7}$/";
		return @preg_match($pattern, $val);
	}

	/**
	 * 手机号码 mobile
	 *
	 */
	function mobile($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^(13|14|15|18)\d{9}$/";
		return @preg_match($pattern, $val);
	}

	/**
	 *    字符串长度检测 min<=$val<=max
	 *
	 *    @param    int       $_min
	 *    @param    int       $_max
	 *    @return   boolean
	 */   
	function len($val, $_min, $_max){
		return (strlen($val)>=$_min && strlen($val)<=$_max);
	}
	
	/*
	* 字符长度包括中文
	*/
	function strLens($val,$_min,$_max){
		return (cnStrLen($val)>=$_min && cnStrLen($val)<=$_max);
	}
	/**
	 *    身份证
	 *
	 *    @param    string    $val
	 *    @return   boolean
	 */
	function id_card($val){
		$val = strtoupper($val);
		$pattern = "/(^\d{15}$)|(^\d{17}([0-9]|X)$)/";
		$arr_split = array();
		if (!preg_match($pattern, $val)) return false;
		if ( strlen($val)==15 ) {
			//检查15位
			$pattern = "/^(\d{6})+(\d{2})+(\d{2})+(\d{2})+(\d{3})$/";
			@preg_match($pattern, $val, $arr_split);
			//检查生日日期是否正确
			$dtm_birth = "19".$arr_split[2] . '/' . $arr_split[3]. '/' .$arr_split[4];
			if (!strtotime($dtm_birth)) return false;
			return true;
		}else{
			//检查18位
			$pattern = "/^(\d{6})+(\d{4})+(\d{2})+(\d{2})+(\d{3})([0-9]|X)$/";
			@preg_match($pattern, $val, $arr_split);
			$dtm_birth = $arr_split[2] . '/' . $arr_split[3]. '/' .$arr_split[4];
			if (!strtotime($dtm_birth)) return false; //检查生日日期是否正确

			//检验18位身份证的校验码是否正确。
			//校验位按照ISO 7064:1983.MOD 11-2的规定生成，X可以认为是数字10。
			$arr_int = array(7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2);
			$arr_ch = array('1', '0', 'X', '9', '8', '7', '6', '5', '4', '3', '2');
			$sign = 0;
			for ( $i = 0; $i < 17; $i++ ) {
				$b = (int) $val{$i};
				$w = $arr_int[$i];
				$sign += $b * $w;
			}
			$n  = $sign % 11;
			$val_num = $arr_ch[$n];
			if ($val_num != substr($val, 17, 1)) return false;
			return true;
		}
	}

	//字符串是否全部是英文
	function english($val){
		$pattern = "/^[a-z]+$/i";
		return @preg_match($pattern, $val);
	}

	//是否是简体中文
	function gb2312($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^[".chr(0xa1)."-".chr(0xff)."]+$/";
		return @preg_match($pattern, $val);
	}
	
	function chinese($val){
		return @preg_match("^[\xa0."-".\xff]+$/", $val);
	}

	//ip
	function ip($val){
		$s = explode(".", $val);
		if (count($s) != 4) return false;
		foreach($s as $v){
			if (is_int($v)) return false;
			//192.168.01.02 检测以0开头数字
			if (preg_match("/^0[0-9]+$/",$v)) return false;
			if ($v<0 || $v>255) return false;
		}
		return true;
	}

	//ipv6
	function ipv6($val) {
		$pattern = "/^((([0-9A-Fa-f]{1,4}:){7}([0-9A-Fa-f]{1,4}|:))|(([0-9A-Fa-f]{1,4}:){6}(:[0-9A-Fa-f]{1,4}|((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){5}(((:[0-9A-Fa-f]{1,4}){1,2})|:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3})|:))|(([0-9A-Fa-f]{1,4}:){4}(((:[0-9A-Fa-f]{1,4}){1,3})|((:[0-9A-Fa-f]{1,4})?:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){3}(((:[0-9A-Fa-f]{1,4}){1,4})|((:[0-9A-Fa-f]{1,4}){0,2}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){2}(((:[0-9A-Fa-f]{1,4}){1,5})|((:[0-9A-Fa-f]{1,4}){0,3}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(([0-9A-Fa-f]{1,4}:){1}(((:[0-9A-Fa-f]{1,4}){1,6})|((:[0-9A-Fa-f]{1,4}){0,4}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:))|(:(((:[0-9A-Fa-f]{1,4}){1,7})|((:[0-9A-Fa-f]{1,4}){0,5}:((25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)(\.(25[0-5]|2[0-4]\d|1\d\d|[1-9]?\d)){3}))|:)))(%.+)?\s*$/";
		return @preg_match($pattern, $val);
	} 

	//qq
	function qq($val, $pattern=''){
		if (empty($pattern)) $pattern = "/^[1-9]\d{4,9}$/";
		return @preg_match($pattern, $val);
	}

	/*
	 * 检查文件扩展名是否合法
	 * @param name 文件名
	 * @param ext  string|array 允许的扩展 小写
	 * - example:
	 * -- extension('gif,rar,jpeg,jpg', 'uephp.jpeg') return true
	 * -- extension(array('rar','bmp','gif'), 'uephp.jpeg') return false
	 */
	function extension($name, $ext){
		if (!is_array($ext)) $ext = explode(',', $ext);
		return in_array(strtolower(end(explode('.', $name))), $ext);
	}
}
?>