/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/
	
	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
	
	1. Definitions.
	
	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.
	
	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.
	
	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.
	
	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.
	
	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.
	
	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.
	
	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).
	
	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.
	
	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."
	
	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.
	
	2. Grant of Copyright License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.
	
	3. Grant of Patent License.
	
	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.
	
	4. Redistribution.
	
	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:
	
	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.
	
	5. Submission of Contributions.
	
	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.
	
	6. Trademarks.
	
	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.
	
	7. Disclaimer of Warranty.
	
	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.
	
	8. Limitation of Liability.
	
	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.
	
	9. Accepting Warranty or Additional Liability.
	
	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.
	
	END OF TERMS AND CONDITIONS
	
	APPENDIX: How to apply the Apache License to your work
	
	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.
	
	   Copyright 2018 JFinal
	
	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at
	
	     http://www.apache.org/licenses/LICENSE-2.0
	
	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.core;

import java.util.Collection;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock.ReadLock;
import java.util.concurrent.locks.ReentrantReadWriteLock.WriteLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.client.ClientChannelContext;
import org.tio.client.ClientTioConfig;
import org.tio.client.ReconnConf;
import org.tio.core.ChannelContext.CloseCode;
import org.tio.core.intf.Packet;
import org.tio.core.intf.Packet.Meta;
import org.tio.server.ServerTioConfig;
import org.tio.utils.convert.Converter;
import org.tio.utils.lock.ReadLockHandler;
import org.tio.utils.lock.SetWithLock;
import org.tio.utils.page.Page;
import org.tio.utils.page.PageUtils;

/**
 * The Class Tio. t-io用户关心的API几乎全在这
 *
 * @author tanyaowu
 */
public class Tio {
	public static class IpBlacklist {
		/**
		 * 把ip添加到黑名单，此黑名单只针对tioConfig有效，其它tioConfig不会把这个ip视为黑名单
		 * @param tioConfig
		 * @param ip
		 * @author tanyaowu
		 */
		public static boolean add(TioConfig tioConfig, String ip) {
			return tioConfig.ipBlacklist.add(ip);
		}

		/**
		 * 添加全局ip黑名单
		 * @param ip
		 * @return
		 * @author tanyaowu
		 */
		public static boolean add(String ip) {
			return org.tio.core.maintain.IpBlacklist.GLOBAL.add(ip);
		}

		/**
		 * 清空黑名单，只针对tioConfig有效
		 * @param tioConfig
		 * @author tanyaowu
		 */
		public static void clear(TioConfig tioConfig) {
			tioConfig.ipBlacklist.clear();
		}

		/**
		 * 清空全局黑名单
		 * @author tanyaowu
		 */
		public static void clear() {
			org.tio.core.maintain.IpBlacklist.GLOBAL.clear();
		}

		/**
		 * 获取ip黑名单列表
		 * @param tioConfig
		 * @return
		 * @author tanyaowu
		 */
		public static Collection<String> getAll(TioConfig tioConfig) {
			return tioConfig.ipBlacklist.getAll();
		}

		/**
		 * 获取全局黑名单
		 * @return
		 * @author tanyaowu
		 */
		public static Collection<String> getAll() {
			return org.tio.core.maintain.IpBlacklist.GLOBAL.getAll();
		}

		/**
		 * 是否在黑名单中
		 * @param tioConfig
		 * @param ip
		 * @return
		 * @author tanyaowu
		 */
		public static boolean isInBlacklist(TioConfig tioConfig, String ip) {
			return tioConfig.ipBlacklist.isInBlacklist(ip) || org.tio.core.maintain.IpBlacklist.GLOBAL.isInBlacklist(ip);
		}

		/**
		 * 把ip从黑名单中删除
		 * @param tioConfig
		 * @param ip
		 * @author tanyaowu
		 */
		public static void remove(TioConfig tioConfig, String ip) {
			tioConfig.ipBlacklist.remove(ip);
		}

		/**
		 * 删除全局黑名单
		 * @param ip
		 * @author tanyaowu
		 */
		public static void remove(String ip) {
			org.tio.core.maintain.IpBlacklist.GLOBAL.remove(ip);
		}
	}

	/** The log. */
	private static Logger log = LoggerFactory.getLogger(Tio.class);

	/**
	 * 绑定业务id
	 * @param channelContext
	 * @param bsId
	 * @author tanyaowu
	 */
	public static void bindBsId(ChannelContext channelContext, String bsId) {
		channelContext.tioConfig.bsIds.bind(channelContext, bsId);
	}

	/**
	 * 绑定群组
	 * @param channelContext
	 * @param group
	 * @author tanyaowu
	 */
	public static void bindGroup(ChannelContext channelContext, String group) {
		channelContext.tioConfig.groups.bind(group, channelContext);
	}

	/**
	 * 将用户绑定到群组
	 * @param tioConfig
	 * @param userid
	 * @param group
	 */
	public static void bindGroup(TioConfig tioConfig, String userid, String group) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByUserid(tioConfig, userid);
		if (setWithLock != null) {
			setWithLock.handle(new ReadLockHandler<Set<ChannelContext>>() {
				@Override
				public void handler(Set<ChannelContext> set) {
					for (ChannelContext channelContext : set) {
						Tio.bindGroup(channelContext, group);
					}
				}
			});
		}
	}

	/**
	 * 绑定token
	 * @param channelContext
	 * @param token
	 * @author tanyaowu
	 */
	public static void bindToken(ChannelContext channelContext, String token) {
		channelContext.tioConfig.tokens.bind(token, channelContext);
	}

	/**
	 * 绑定用户
	 * @param channelContext
	 * @param userid
	 * @author tanyaowu
	 */
	public static void bindUser(ChannelContext channelContext, String userid) {
		channelContext.tioConfig.users.bind(userid, channelContext);
	}

	/**
	 * 阻塞发送消息到指定ChannelContext
	 * @param channelContext
	 * @param packet
	 * @return
	 * @author tanyaowu
	 */
	public static Boolean bSend(ChannelContext channelContext, Packet packet) {
		if (channelContext == null) {
			return false;
		}
		CountDownLatch countDownLatch = new CountDownLatch(1);
		return send(channelContext, packet, countDownLatch, PacketSendMode.SINGLE_BLOCK);
	}

	/**
	 * 发送到指定的ip和port
	 * @param tioConfig
	 * @param ip
	 * @param port
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean bSend(TioConfig tioConfig, String ip, int port, Packet packet) {
		return send(tioConfig, ip, port, packet, true);
	}

	/**
	 * 发消息到所有连接
	 * @param tioConfig
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static Boolean bSendToAll(TioConfig tioConfig, Packet packet, ChannelContextFilter channelContextFilter) {
		return sendToAll(tioConfig, packet, channelContextFilter, true);
	}

	/**
	 * 阻塞发消息给指定业务ID
	 * @param tioConfig
	 * @param bsId
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean bSendToBsId(TioConfig tioConfig, String bsId, Packet packet) {
		return sendToBsId(tioConfig, bsId, packet, true);
	}

	/**
	 * 发消息到组
	 * @param tioConfig
	 * @param group
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean bSendToGroup(TioConfig tioConfig, String group, Packet packet) {
		return bSendToGroup(tioConfig, group, packet, null);
	}

	/**
	 * 发消息到组
	 * @param tioConfig
	 * @param group
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static Boolean bSendToGroup(TioConfig tioConfig, String group, Packet packet, ChannelContextFilter channelContextFilter) {
		return sendToGroup(tioConfig, group, packet, channelContextFilter, true);
	}

	/**
	 * 发消息给指定ChannelContext id
	 * @param channelContextId
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean bSendToId(TioConfig tioConfig, String channelContextId, Packet packet) {
		return sendToId(tioConfig, channelContextId, packet, true);
	}

	/**
	 * 阻塞发送到指定ip对应的集合
	 * @param tioConfig
	 * @param ip
	 * @param packet
	 * @author: tanyaowu
	 */
	public static Boolean bSendToIp(TioConfig tioConfig, String ip, Packet packet) {
		return bSendToIp(tioConfig, ip, packet, null);
	}

	/**
	 * 阻塞发送到指定ip对应的集合
	 * @param tioConfig
	 * @param ip
	 * @param packet
	 * @param channelContextFilter
	 * @return
	 * @author: tanyaowu
	 */
	public static Boolean bSendToIp(TioConfig tioConfig, String ip, Packet packet, ChannelContextFilter channelContextFilter) {
		return sendToIp(tioConfig, ip, packet, channelContextFilter, true);
	}

	/**
	 * 发消息到指定集合
	 * @param tioConfig
	 * @param setWithLock
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static Boolean bSendToSet(TioConfig tioConfig, SetWithLock<ChannelContext> setWithLock, Packet packet, ChannelContextFilter channelContextFilter) {
		return sendToSet(tioConfig, setWithLock, packet, channelContextFilter, true);
	}

	/**
	 * 阻塞发消息到指定token
	 * @param tioConfig
	 * @param token
	 * @param packet
	 * @return
	 * @author tanyaowu
	 */
	public static Boolean bSendToToken(TioConfig tioConfig, String token, Packet packet) {
		return sendToToken(tioConfig, token, packet, true);
	}

	/**
	 * 阻塞发消息给指定用户
	 * @param tioConfig
	 * @param userid
	 * @param packet
	 * @return
	 * @author tanyaowu
	 */
	public static Boolean bSendToUser(TioConfig tioConfig, String userid, Packet packet) {
		return sendToUser(tioConfig, userid, packet, true);
	}

	/**
	 * 关闭连接
	 * @param channelContext
	 * @param remark
	 * @author tanyaowu
	 */
	public static void close(ChannelContext channelContext, String remark) {
		close(channelContext, null, remark);
	}

	/**
	 * 
	 * @param channelContext
	 * @param remark
	 * @param closeCode
	 */
	public static void close(ChannelContext channelContext, String remark, CloseCode closeCode) {
		close(channelContext, null, remark, closeCode);
	}

	/**
	 * 关闭连接
	 * @param channelContext
	 * @param throwable
	 * @param remark
	 * @author tanyaowu
	 */
	public static void close(ChannelContext channelContext, Throwable throwable, String remark) {
		close(channelContext, throwable, remark, false);
	}

	public static void close(ChannelContext channelContext, Throwable throwable, String remark, CloseCode closeCode) {
		close(channelContext, throwable, remark, false, closeCode);
	}

	public static void close(ChannelContext channelContext, Throwable throwable, String remark, boolean isNeedRemove) {
		close(channelContext, throwable, remark, isNeedRemove, true);
	}

	public static void close(ChannelContext channelContext, Throwable throwable, String remark, boolean isNeedRemove, CloseCode closeCode) {
		close(channelContext, throwable, remark, isNeedRemove, true, closeCode);
	}

	public static void close(ChannelContext channelContext, Throwable throwable, String remark, boolean isNeedRemove, boolean needCloseLock) {
		close(channelContext, throwable, remark, isNeedRemove, needCloseLock, null);
	}

	/**
	 * 
	 * @param channelContext
	 * @param throwable
	 * @param remark
	 * @param isNeedRemove
	 * @param needCloseLock
	 */
	public static void close(ChannelContext channelContext, Throwable throwable, String remark, boolean isNeedRemove, boolean needCloseLock, CloseCode closeCode) {
		if (channelContext == null) {
			return;
		}
		if (channelContext.isWaitingClose) {
			log.debug("{} 正在等待被关闭", channelContext);
			return;
		}

		//先立即取消各项任务，这样可防止有新的任务被提交进来
		channelContext.decodeRunnable.setCanceled(true);
		channelContext.handlerRunnable.setCanceled(true);
		channelContext.sendRunnable.setCanceled(true);

		WriteLock writeLock = null;
		if (needCloseLock) {
			writeLock = channelContext.closeLock.writeLock();

			boolean tryLock = writeLock.tryLock();
			if (!tryLock) {
				return;
			}
			channelContext.isWaitingClose = true;
			writeLock.unlock();
		} else {
			channelContext.isWaitingClose = true;
		}

		if (closeCode == null) {
			if (channelContext.getCloseCode() == CloseCode.INIT_STATUS) {
				channelContext.setCloseCode(CloseCode.NO_CODE);
			}
		} else {
			channelContext.setCloseCode(closeCode);
		}

		if (channelContext.asynchronousSocketChannel != null) {
			try {
				channelContext.asynchronousSocketChannel.shutdownInput();
			} catch (Throwable e) {
				//log.error(e.toString(), e);
			}
			try {
				channelContext.asynchronousSocketChannel.shutdownOutput();
			} catch (Throwable e) {
				//log.error(e.toString(), e);
			}
			try {
				channelContext.asynchronousSocketChannel.close();
			} catch (Throwable e) {
				//log.error(e.toString(), e);
			}
		}

		channelContext.closeMeta.setRemark(remark);
		channelContext.closeMeta.setThrowable(throwable);
		if (!isNeedRemove) {
			if (channelContext.isServer()) {
				isNeedRemove = true;
			} else {
				ClientChannelContext clientChannelContext = (ClientChannelContext) channelContext;
				if (!ReconnConf.isNeedReconn(clientChannelContext, false)) { //不需要重连
					isNeedRemove = true;
				}
			}
		}
		channelContext.closeMeta.setNeedRemove(isNeedRemove);

		channelContext.tioConfig.closeRunnable.addMsg(channelContext);
		channelContext.tioConfig.closeRunnable.execute();
	}

	/**
	 * 关闭连接
	 * @param tioConfig
	 * @param clientIp
	 * @param clientPort
	 * @param throwable
	 * @param remark
	 * @author tanyaowu
	 */
	public static void close(TioConfig tioConfig, String clientIp, Integer clientPort, Throwable throwable, String remark) {
		ChannelContext channelContext = tioConfig.clientNodes.find(clientIp, clientPort);
		close(channelContext, throwable, remark);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param ip
	 * @param remark
	 * @return
	 */
	public static void closeIp(TioConfig tioConfig, String ip, String remark) {
		closeIp(tioConfig, ip, remark, null);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param ip
	 * @param remark
	 * @param closeCode
	 */
	public static void closeIp(TioConfig tioConfig, String ip, String remark, CloseCode closeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByIp(tioConfig, ip);
		closeSet(tioConfig, setWithLock, remark, closeCode);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param group
	 * @param remark
	 * @return
	 */
	public static void closeGroup(TioConfig tioConfig, String group, String remark) {
		closeGroup(tioConfig, group, remark, null);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param group
	 * @param remark
	 * @param closeCode
	 */
	public static void closeGroup(TioConfig tioConfig, String group, String remark, CloseCode closeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByGroup(tioConfig, group);
		closeSet(tioConfig, setWithLock, remark, closeCode);
	}

	/**
	 * 关闭用户的所有连接
	 * @param tioConfig
	 * @param userid
	 * @param remark
	 * @return
	 */
	public static void closeUser(TioConfig tioConfig, String userid, String remark) {
		closeUser(tioConfig, userid, remark, null);
	}

	/**
	 * 关闭某用户的所有连接
	 * @param tioConfig
	 * @param userid
	 * @param remark
	 * @param closeCode
	 */
	public static void closeUser(TioConfig tioConfig, String userid, String remark, CloseCode closeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByUserid(tioConfig, userid);
		closeSet(tioConfig, setWithLock, remark, closeCode);
	}

	/**
	 * 关闭token的所有连接
	 * @param tioConfig
	 * @param token
	 * @param remark
	 * @return
	 */
	public static void closeToken(TioConfig tioConfig, String token, String remark) {
		closeToken(tioConfig, token, remark, null);
	}

	/**
	 * 关闭某token的所有连接
	 * @param tioConfig
	 * @param token
	 * @param remark
	 * @param closeCode
	 */
	public static void closeToken(TioConfig tioConfig, String token, String remark, CloseCode closeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByToken(tioConfig, token);
		closeSet(tioConfig, setWithLock, remark, closeCode);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param ip
	 * @param remark
	 * @return
	 */
	public static void removeIp(TioConfig tioConfig, String ip, String remark) {
		removeIp(tioConfig, ip, remark, null);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param ip
	 * @param remark
	 * @param removeCode
	 */
	public static void removeIp(TioConfig tioConfig, String ip, String remark, CloseCode removeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByIp(tioConfig, ip);
		removeSet(tioConfig, setWithLock, remark, removeCode);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param group
	 * @param remark
	 * @return
	 */
	public static void removeGroup(TioConfig tioConfig, String group, String remark) {
		removeGroup(tioConfig, group, remark, null);
	}

	/**
	 * 关闭某群所有连接
	 * @param tioConfig
	 * @param group
	 * @param remark
	 * @param removeCode
	 */
	public static void removeGroup(TioConfig tioConfig, String group, String remark, CloseCode removeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByGroup(tioConfig, group);
		removeSet(tioConfig, setWithLock, remark, removeCode);
	}

	/**
	 * 关闭用户的所有连接
	 * @param tioConfig
	 * @param userid
	 * @param remark
	 * @return
	 */
	public static void removeUser(TioConfig tioConfig, String userid, String remark) {
		removeUser(tioConfig, userid, remark, null);
	}

	/**
	 * 关闭某用户的所有连接
	 * @param tioConfig
	 * @param userid
	 * @param remark
	 * @param removeCode
	 */
	public static void removeUser(TioConfig tioConfig, String userid, String remark, CloseCode removeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByUserid(tioConfig, userid);
		removeSet(tioConfig, setWithLock, remark, removeCode);
	}

	/**
	 * 关闭token的所有连接
	 * @param tioConfig
	 * @param token
	 * @param remark
	 * @return
	 */
	public static void removeToken(TioConfig tioConfig, String token, String remark) {
		removeToken(tioConfig, token, remark, null);
	}

	/**
	 * 关闭某token的所有连接
	 * @param tioConfig
	 * @param token
	 * @param remark
	 * @param removeCode
	 */
	public static void removeToken(TioConfig tioConfig, String token, String remark, CloseCode removeCode) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByToken(tioConfig, token);
		removeSet(tioConfig, setWithLock, remark, removeCode);
	}

	/**
	 * 关闭集合
	 * @param tioConfig
	 * @param setWithLock
	 * @param remark
	 * @param closeCode
	 * @author tanyaowu
	 */
	public static void closeSet(TioConfig tioConfig, SetWithLock<ChannelContext> setWithLock, String remark, CloseCode closeCode) {
		if (setWithLock != null) {
			setWithLock.handle(new ReadLockHandler<Set<ChannelContext>>() {
				@Override
				public void handler(Set<ChannelContext> set) {
					for (ChannelContext channelContext : set) {
						Tio.close(channelContext, remark, closeCode);
					}
				}
			});
		}
	}

	/**
	 * 移除集合
	 * @param tioConfig
	 * @param setWithLock
	 * @param remark
	 * @param closeCode
	 * @author tanyaowu
	 */
	public static void removeSet(TioConfig tioConfig, SetWithLock<ChannelContext> setWithLock, String remark, CloseCode closeCode) {
		if (setWithLock != null) {
			setWithLock.handle(new ReadLockHandler<Set<ChannelContext>>() {
				@Override
				public void handler(Set<ChannelContext> set) {
					for (ChannelContext channelContext : set) {
						Tio.remove(channelContext, remark, closeCode);
					}
				}
			});
		}
	}

	/**
	 * 获取所有连接，包括当前处于断开状态的
	 * @param tioConfig
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getAll(TioConfig tioConfig) {
		return tioConfig.connections;
	}

	/**
	 * 获取所有连接，包括当前处于断开状态的
	 * @param tioConfig
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getAll(TioConfig tioConfig)
	 */
	public static SetWithLock<ChannelContext> getAllChannelContexts(TioConfig tioConfig) {
		return getAll(tioConfig);
	}

	/**
	 * 此API仅供 tio client使用
	 * 获取所有处于正常连接状态的连接
	 * @param clientTioConfig
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getConnecteds(ClientTioConfig clientTioConfig) {
		return clientTioConfig.connecteds;
	}

	/**
	 * 此API仅供 tio client使用
	 * 获取所有处于正常连接状态的连接
	 * @param clientTioConfig
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getAllConnecteds(ClientTioConfig clientTioConfig)
	 */
	public static SetWithLock<ChannelContext> getAllConnectedsChannelContexts(ClientTioConfig clientTioConfig) {
		return getConnecteds(clientTioConfig);
	}

	/**
	 * 根据业务id找ChannelContext
	 * @param tioConfig
	 * @param bsId
	 * @return
	 * @author tanyaowu
	 */
	public static ChannelContext getByBsId(TioConfig tioConfig, String bsId) {
		return tioConfig.bsIds.find(tioConfig, bsId);
	}

	/**
	 * 根据业务id找ChannelContext
	 * @param tioConfig
	 * @param bsId
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getByBsId(TioConfig tioConfig, String bsId)
	 */
	public static ChannelContext getChannelContextByBsId(TioConfig tioConfig, String bsId) {
		return getByBsId(tioConfig, bsId);
	}

	/**
	 * 根据clientip和clientport获取ChannelContext
	 * @param tioConfig
	 * @param clientIp
	 * @param clientPort
	 * @return
	 * @author tanyaowu
	 */
	public static ChannelContext getByClientNode(TioConfig tioConfig, String clientIp, Integer clientPort) {
		return tioConfig.clientNodes.find(clientIp, clientPort);
	}

	/**
	 * 根据clientip和clientport获取ChannelContext
	 * @param tioConfig
	 * @param clientIp
	 * @param clientPort
	 * @return
	 * @author tanyaowu
	 * @deprecated getByClientNode(tioConfig, clientIp, clientPort)
	 */
	public static ChannelContext getChannelContextByClientNode(TioConfig tioConfig, String clientIp, Integer clientPort) {
		return getByClientNode(tioConfig, clientIp, clientPort);
	}

	/**
	 * 根据ChannelContext.id获取ChannelContext
	 * @param channelContextId
	 * @return
	 * @author tanyaowu
	 */
	public static ChannelContext getByChannelContextId(TioConfig tioConfig, String channelContextId) {
		return tioConfig.ids.find(tioConfig, channelContextId);
	}

	/**
	 * 根据ChannelContext.id获取ChannelContext
	 * @param channelContextId
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getByChannelContextId(tioConfig, channelContextId)
	 */
	public static ChannelContext getChannelContextById(TioConfig tioConfig, String channelContextId) {
		return getByChannelContextId(tioConfig, channelContextId);
	}

	/**
	 * 获取一个组的所有客户端
	 * @param tioConfig
	 * @param group
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getByGroup(TioConfig tioConfig, String group) {
		return tioConfig.groups.clients(tioConfig, group);
	}

	/**
	 * 获取一个组的所有客户端
	 * @param tioConfig
	 * @param group
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getByGroup(tioConfig, group)
	 */
	public static SetWithLock<ChannelContext> getChannelContextsByGroup(TioConfig tioConfig, String group) {
		return getByGroup(tioConfig, group);
	}

	/**
	 * 根据token获取SetWithLock<ChannelContext>
	 * @param tioConfig
	 * @param token
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getByToken(TioConfig tioConfig, String token) {
		return tioConfig.tokens.find(tioConfig, token);
	}

	/**
	 * 根据客户端ip获取SetWithLock<ChannelContext>
	 * @param tioConfig
	 * @param ip
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getByIp(TioConfig tioConfig, String ip) {
		return tioConfig.ips.clients(tioConfig, ip);
	}

	/**
	 * 根据token获取SetWithLock<ChannelContext>
	 * @param tioConfig
	 * @param token
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getByToken(tioConfig, token)
	 */
	public static SetWithLock<ChannelContext> getChannelContextsByToken(TioConfig tioConfig, String token) {
		return getByToken(tioConfig, token);
	}

	/**
	 * 根据userid获取SetWithLock<ChannelContext>
	 * @param tioConfig
	 * @param userid
	 * @return
	 * @author tanyaowu
	 */
	public static SetWithLock<ChannelContext> getByUserid(TioConfig tioConfig, String userid) {
		return tioConfig.users.find(tioConfig, userid);
	}

	/**
	 * 根据userid获取SetWithLock<ChannelContext>
	 * @param tioConfig
	 * @param userid
	 * @return
	 * @author tanyaowu
	 * @deprecated 用getByUserid(tioConfig, userid)
	 */
	public static SetWithLock<ChannelContext> getChannelContextsByUserid(TioConfig tioConfig, String userid) {
		return getByUserid(tioConfig, userid);
	}

	/**
	 *
	 * @param tioConfig
	 * @param pageIndex
	 * @param pageSize
	 * @return
	 * @author tanyaowu
	 */
	public static Page<ChannelContext> getPageOfAll(TioConfig tioConfig, Integer pageIndex, Integer pageSize) {
		return getPageOfAll(tioConfig, pageIndex, pageSize, null);
	}

	/**
	 * 
	 * @param tioConfig
	 * @param pageIndex
	 * @param pageSize
	 * @param converter
	 * @return
	 */
	public static <T> Page<T> getPageOfAll(TioConfig tioConfig, Integer pageIndex, Integer pageSize, Converter<T> converter) {
		SetWithLock<ChannelContext> setWithLock = Tio.getAllChannelContexts(tioConfig);
		return PageUtils.fromSetWithLock(setWithLock, pageIndex, pageSize, converter);
	}

	/**
	 * 这个方法是给客户器端用的
	 * @param clientTioConfig
	 * @param pageIndex
	 * @param pageSize
	 * @return
	 * @author tanyaowu
	 */
	public static Page<ChannelContext> getPageOfConnecteds(ClientTioConfig clientTioConfig, Integer pageIndex, Integer pageSize) {
		return getPageOfConnecteds(clientTioConfig, pageIndex, pageSize, null);
	}

	/**
	 * 这个方法是给客户器端用的
	 * @param clientTioConfig
	 * @param pageIndex
	 * @param pageSize
	 * @param converter
	 * @return
	 * @author tanyaowu
	 */
	public static <T> Page<T> getPageOfConnecteds(ClientTioConfig clientTioConfig, Integer pageIndex, Integer pageSize, Converter<T> converter) {
		SetWithLock<ChannelContext> setWithLock = Tio.getAllConnectedsChannelContexts(clientTioConfig);
		return PageUtils.fromSetWithLock(setWithLock, pageIndex, pageSize, converter);
	}

	/**
	 *
	 * @param tioConfig
	 * @param group
	 * @param pageIndex
	 * @param pageSize
	 * @return
	 * @author tanyaowu
	 */
	public static Page<ChannelContext> getPageOfGroup(TioConfig tioConfig, String group, Integer pageIndex, Integer pageSize) {
		return getPageOfGroup(tioConfig, group, pageIndex, pageSize, null);
	}

	/**
	 * 
	 * @param tioConfig
	 * @param group
	 * @param pageIndex
	 * @param pageSize
	 * @param converter
	 * @return
	 */
	public static <T> Page<T> getPageOfGroup(TioConfig tioConfig, String group, Integer pageIndex, Integer pageSize, Converter<T> converter) {
		SetWithLock<ChannelContext> setWithLock = Tio.getChannelContextsByGroup(tioConfig, group);
		return PageUtils.fromSetWithLock(setWithLock, pageIndex, pageSize, converter);
	}

	/**
	 * 群组有多少个连接
	 * @param tioConfig
	 * @param group
	 * @return
	 */
	public static int groupCount(TioConfig tioConfig, String group) {
		SetWithLock<ChannelContext> setWithLock = tioConfig.groups.clients(tioConfig, group);
		if (setWithLock == null) {
			return 0;
		}

		Set<ChannelContext> set = setWithLock.getObj();
		if (set == null) {
			return 0;
		}

		return set.size();
	}

	/**
	 * 某通道是否在某群组中
	 * @param group
	 * @param channelContext
	 * @return true：在该群组
	 * @author: tanyaowu
	 */
	public static boolean isInGroup(String group, ChannelContext channelContext) {
		SetWithLock<String> setWithLock = channelContext.getGroups();
		if (setWithLock == null) {
			return false;
		}

		Set<String> set = setWithLock.getObj();
		if (set == null) {
			return false;
		}

		return set.contains(group);
	}

	/**
	 * 
	 * @param channelContext
	 * @param remark
	 */
	public static void remove(ChannelContext channelContext, String remark) {
		remove(channelContext, remark, null);
	}

	/**
	 * 和close方法对应，只不过不再进行重连等维护性的操作
	 * @param channelContext
	 * @param remark
	 * @param closeCode
	 */
	public static void remove(ChannelContext channelContext, String remark, CloseCode closeCode) {
		remove(channelContext, null, remark, closeCode);
	}

	/**
	 * 和close方法对应，只不过不再进行重连等维护性的操作
	 * @param channelContext
	 * @param throwable
	 * @param remark
	 */
	public static void remove(ChannelContext channelContext, Throwable throwable, String remark) {
		remove(channelContext, throwable, remark, (CloseCode) null);
	}

	/**
	 * 和close方法对应，只不过不再进行重连等维护性的操作
	 * @param channelContext
	 * @param throwable
	 * @param remark
	 * @param closeCode
	 */
	public static void remove(ChannelContext channelContext, Throwable throwable, String remark, CloseCode closeCode) {
		close(channelContext, throwable, remark, true, closeCode);
	}

	/**
	 * 和close方法对应，只不过不再进行重连等维护性的操作
	 * @param tioConfig
	 * @param clientIp
	 * @param clientPort
	 * @param throwable
	 * @param remark
	 */
	public static void remove(TioConfig tioConfig, String clientIp, Integer clientPort, Throwable throwable, String remark) {
		remove(tioConfig, clientIp, clientPort, throwable, remark, (CloseCode) null);
	}

	/**
	 * 删除clientip和clientPort为指定值的连接
	 * @param tioConfig
	 * @param clientIp
	 * @param clientPort
	 * @param throwable
	 * @param remark
	 * @param closeCode
	 */
	public static void remove(TioConfig tioConfig, String clientIp, Integer clientPort, Throwable throwable, String remark, CloseCode closeCode) {
		ChannelContext channelContext = tioConfig.clientNodes.find(clientIp, clientPort);
		remove(channelContext, throwable, remark, closeCode);
	}

	/**
	 * 删除clientip为指定值的所有连接
	 * @param serverTioConfig
	 * @param ip
	 * @param remark
	 */
	public static void remove(ServerTioConfig serverTioConfig, String ip, String remark) {
		remove(serverTioConfig, ip, remark, (CloseCode) null);
	}

	/**
	 *  删除clientip为指定值的所有连接
	 * @param serverTioConfig
	 * @param ip
	 * @param remark
	 * @param closeCode
	 */
	public static void remove(ServerTioConfig serverTioConfig, String ip, String remark, CloseCode closeCode) {
		SetWithLock<ChannelContext> setWithLock = serverTioConfig.ips.clients(serverTioConfig, ip);
		if (setWithLock == null) {
			return;
		}

		setWithLock.handle(new ReadLockHandler<Set<ChannelContext>>() {
			@Override
			public void handler(Set<ChannelContext> set) {
				for (ChannelContext channelContext : set) {
					Tio.remove(channelContext, remark, closeCode);
				}
			}
		});
	}

	/**
	 * 发送消息到指定ChannelContext
	 * @param channelContext
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean send(ChannelContext channelContext, Packet packet) {
		return send(channelContext, packet, null, null);
	}

	/**
	 *
	 * @param channelContext
	 * @param packet
	 * @param countDownLatch
	 * @param packetSendMode
	 * @return
	 * @author tanyaowu
	 */
	private static Boolean send(final ChannelContext channelContext, Packet packet, CountDownLatch countDownLatch, PacketSendMode packetSendMode) {
		try {
			if (packet == null || channelContext == null) {
				if (countDownLatch != null) {
					countDownLatch.countDown();
				}
				return false;
			}

			if (channelContext.isVirtual) {
				if (countDownLatch != null) {
					countDownLatch.countDown();
				}
				return true;
			}

			if (channelContext.isClosed || channelContext.isRemoved) {
				if (countDownLatch != null) {
					countDownLatch.countDown();
				}
				if (channelContext != null) {
					log.info("can't send data, {}, isClosed:{}, isRemoved:{}", channelContext, channelContext.isClosed, channelContext.isRemoved);
				}
				return false;
			}

			if (channelContext.tioConfig.packetConverter != null) {
				Packet packet1 = channelContext.tioConfig.packetConverter.convert(packet, channelContext);
				if (packet1 == null) {
					if (log.isInfoEnabled()) {
						log.info("convert后为null，表示不需要发送", channelContext, packet.logstr());
					}
					return true;
				}
				packet = packet1;
			}

			boolean isSingleBlock = countDownLatch != null && packetSendMode == PacketSendMode.SINGLE_BLOCK;

			boolean isAdded = false;
			if (countDownLatch != null) {
				Meta meta = new Meta();
				meta.setCountDownLatch(countDownLatch);
				packet.setMeta(meta);
			}

			if (channelContext.tioConfig.useQueueSend) {
				isAdded = channelContext.sendRunnable.addMsg(packet);
			} else {
				isAdded = channelContext.sendRunnable.sendPacket(packet);
			}

			if (!isAdded) {
				if (countDownLatch != null) {
					countDownLatch.countDown();
				}
				return false;
			}
			if (channelContext.tioConfig.useQueueSend) {
				channelContext.sendRunnable.execute();
			}

			if (isSingleBlock) {
				long timeout = 10;
				try {
					Boolean awaitFlag = countDownLatch.await(timeout, TimeUnit.SECONDS);
					if (!awaitFlag) {
						log.error("{}, 阻塞发送超时, timeout:{}s, packet:{}", channelContext, timeout, packet.logstr());
					}
				} catch (InterruptedException e) {
					log.error(e.toString(), e);
				}

				Boolean isSentSuccess = packet.getMeta().getIsSentSuccess();
				return isSentSuccess;
			} else {
				return true;
			}
		} catch (Throwable e) {
			log.error(channelContext + ", " + e.toString(), e);
			return false;
		}

	}

	/**
	 * 发送到指定的ip和port
	 * @param tioConfig
	 * @param ip
	 * @param port
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean send(TioConfig tioConfig, String ip, int port, Packet packet) {
		return send(tioConfig, ip, port, packet, false);
	}

	/**
	 * 发送到指定的ip和port
	 * @param tioConfig
	 * @param ip
	 * @param port
	 * @param packet
	 * @param isBlock
	 * @return
	 * @author tanyaowu
	 */
	private static Boolean send(TioConfig tioConfig, String ip, int port, Packet packet, boolean isBlock) {
		ChannelContext channelContext = tioConfig.clientNodes.find(ip, port);
		if (channelContext != null) {
			if (isBlock) {
				return bSend(channelContext, packet);
			} else {
				return send(channelContext, packet);
			}
		} else {
			log.info("{}, can find channelContext by {}:{}", tioConfig.getName(), ip, port);
			return false;
		}
	}

	public static void sendToAll(TioConfig tioConfig, Packet packet) {
		sendToAll(tioConfig, packet, null);
	}

	/**
	 * 发消息到所有连接
	 * @param tioConfig
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static void sendToAll(TioConfig tioConfig, Packet packet, ChannelContextFilter channelContextFilter) {
		sendToAll(tioConfig, packet, channelContextFilter, false);
	}

	/**
	 *
	 * @param tioConfig
	 * @param packet
	 * @param channelContextFilter
	 * @param isBlock
	 * @author tanyaowu
	 */
	private static Boolean sendToAll(TioConfig tioConfig, Packet packet, ChannelContextFilter channelContextFilter, boolean isBlock) {
		try {
			SetWithLock<ChannelContext> setWithLock = tioConfig.connections;
			if (setWithLock == null) {
				log.debug("{}, 没有任何连接", tioConfig.getName());
				return false;
			}
			Boolean ret = sendToSet(tioConfig, setWithLock, packet, channelContextFilter, isBlock);
			return ret;
		} finally {
		}
	}

	/**
	 * 发消息给指定业务ID
	 * @param tioConfig
	 * @param bsId
	 * @param packet
	 * @return
	 * @author tanyaowu
	 */
	public static Boolean sendToBsId(TioConfig tioConfig, String bsId, Packet packet) {
		return sendToBsId(tioConfig, bsId, packet, false);
	}

	/**
	 * 发消息给指定业务ID
	 * @param tioConfig
	 * @param bsId
	 * @param packet
	 * @param isBlock
	 * @return
	 * @author tanyaowu
	 */
	private static Boolean sendToBsId(TioConfig tioConfig, String bsId, Packet packet, boolean isBlock) {
		ChannelContext channelContext = Tio.getByBsId(tioConfig, bsId);
		if (channelContext == null) {
			return false;
		}
		if (isBlock) {
			return bSend(channelContext, packet);
		} else {
			return send(channelContext, packet);
		}
	}

	/**
	 * 发消息到组
	 * @param tioConfig
	 * @param group
	 * @param packet
	 * @author tanyaowu
	 */
	public static void sendToGroup(TioConfig tioConfig, String group, Packet packet) {
		sendToGroup(tioConfig, group, packet, null);
	}

	/**
	 * 发消息到组
	 * @param tioConfig
	 * @param group
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static void sendToGroup(TioConfig tioConfig, String group, Packet packet, ChannelContextFilter channelContextFilter) {
		sendToGroup(tioConfig, group, packet, channelContextFilter, false);
	}

	/**
	 * 发消息到组
	 * @param tioConfig
	 * @param group
	 * @param packet
	 * @param channelContextFilter
	 * @param isBlock
	 * @return
	 */
	private static Boolean sendToGroup(TioConfig tioConfig, String group, Packet packet, ChannelContextFilter channelContextFilter, boolean isBlock) {
		try {
			SetWithLock<ChannelContext> setWithLock = tioConfig.groups.clients(tioConfig, group);
			if (setWithLock == null) {
				log.debug("{}, 组[{}]不存在", tioConfig.getName(), group);
				return false;
			}
			Boolean ret = sendToSet(tioConfig, setWithLock, packet, channelContextFilter, isBlock);
			return ret;
		} finally {
		}
	}

	/**
	 * 发消息给指定ChannelContext id
	 * @param channelContextId
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean sendToId(TioConfig tioConfig, String channelContextId, Packet packet) {
		return sendToId(tioConfig, channelContextId, packet, false);
	}

	/**
	 * 发消息给指定ChannelContext id
	 * @param channelContextId
	 * @param packet
	 * @param isBlock
	 * @return
	 * @author tanyaowu
	 */
	private static Boolean sendToId(TioConfig tioConfig, String channelContextId, Packet packet, boolean isBlock) {
		ChannelContext channelContext = Tio.getChannelContextById(tioConfig, channelContextId);
		if (channelContext == null) {
			return false;
		}
		if (isBlock) {
			return bSend(channelContext, packet);
		} else {
			return send(channelContext, packet);
		}
	}

	/**
	 * 发送到指定ip对应的集合
	 * @param tioConfig
	 * @param ip
	 * @param packet
	 * @author: tanyaowu
	 */
	public static void sendToIp(TioConfig tioConfig, String ip, Packet packet) {
		sendToIp(tioConfig, ip, packet, null);
	}

	/**
	 * 发送到指定ip对应的集合
	 * @param tioConfig
	 * @param ip
	 * @param packet
	 * @param channelContextFilter
	 * @author: tanyaowu
	 */
	public static void sendToIp(TioConfig tioConfig, String ip, Packet packet, ChannelContextFilter channelContextFilter) {
		sendToIp(tioConfig, ip, packet, channelContextFilter, false);
	}

	/**
	 * 发送到指定ip对应的集合
	 * @param tioConfig
	 * @param ip
	 * @param packet
	 * @param channelContextFilter
	 * @param isBlock
	 * @return
	 * @author: tanyaowu
	 */
	private static Boolean sendToIp(TioConfig tioConfig, String ip, Packet packet, ChannelContextFilter channelContextFilter, boolean isBlock) {
		try {
			SetWithLock<ChannelContext> setWithLock = tioConfig.ips.clients(tioConfig, ip);
			if (setWithLock == null) {
				log.info("{}, 没有ip为[{}]的对端", tioConfig.getName(), ip);
				return false;
			}
			Boolean ret = sendToSet(tioConfig, setWithLock, packet, channelContextFilter, isBlock);
			return ret;
		} finally {
		}
	}

	/**
	 * 发消息到指定集合
	 * @param tioConfig
	 * @param setWithLock
	 * @param packet
	 * @param channelContextFilter
	 * @author tanyaowu
	 */
	public static void sendToSet(TioConfig tioConfig, SetWithLock<ChannelContext> setWithLock, Packet packet, ChannelContextFilter channelContextFilter) {
		sendToSet(tioConfig, setWithLock, packet, channelContextFilter, false);
	}

	/**
	 * 发消息到指定集合
	 * @param tioConfig
	 * @param setWithLock
	 * @param packet
	 * @param channelContextFilter
	 * @param isBlock
	 * @author tanyaowu
	 */
	private static Boolean sendToSet(TioConfig tioConfig, SetWithLock<ChannelContext> setWithLock, Packet packet, ChannelContextFilter channelContextFilter, boolean isBlock) {
		boolean releasedLock = false;
		Lock lock = setWithLock.readLock();
		lock.lock();
		try {
			Set<ChannelContext> set = setWithLock.getObj();
			if (set.size() == 0) {
				log.debug("{}, 集合为空", tioConfig.getName());
				return false;
			}

			CountDownLatch countDownLatch = null;
			if (isBlock) {
				countDownLatch = new CountDownLatch(set.size());
			}
			int sendCount = 0;
			for (ChannelContext channelContext : set) {
				if (channelContextFilter != null) {
					boolean isfilter = channelContextFilter.filter(channelContext);
					if (!isfilter) {
						if (isBlock) {
							countDownLatch.countDown();
						}
						continue;
					}
				}

				sendCount++;
				if (isBlock) {
					send(channelContext, packet, countDownLatch, PacketSendMode.GROUP_BLOCK);
				} else {
					send(channelContext, packet, null, null);
				}
			}
			lock.unlock();
			releasedLock = true;

			if (sendCount == 0) {
				return false;
			}

			if (isBlock) {
				try {
					long timeout = sendCount / 5;
					timeout = Math.max(timeout, 10);//timeout < 10 ? 10 : timeout;
					boolean awaitFlag = countDownLatch.await(timeout, TimeUnit.SECONDS);
					if (!awaitFlag) {
						log.error("{}, 同步群发超时, size:{}, timeout:{}, packet:{}", tioConfig.getName(), setWithLock.getObj().size(), timeout, packet.logstr());
						return false;
					} else {
						return true;
					}
				} catch (InterruptedException e) {
					log.error(e.toString(), e);
					return false;
				} finally {

				}
			} else {
				return true;
			}
		} catch (Throwable e) {
			log.error(e.toString(), e);
			return false;
		} finally {
			if (!releasedLock) {
				lock.unlock();
			}
		}
	}

	/**
	 * 发消息到指定token
	 * @param tioConfig
	 * @param token
	 * @param packet
	 * @return
	 * @author tanyaowu
	 */
	public static Boolean sendToToken(TioConfig tioConfig, String token, Packet packet) {
		return sendToToken(tioConfig, token, packet, false);
	}

	/**
	 * 发消息给指定token
	 * @param tioConfig
	 * @param token
	 * @param packet
	 * @param isBlock
	 * @author tanyaowu
	 */
	private static Boolean sendToToken(TioConfig tioConfig, String token, Packet packet, boolean isBlock) {
		SetWithLock<ChannelContext> setWithLock = tioConfig.tokens.find(tioConfig, token);
		try {
			if (setWithLock == null) {
				return false;
			}

			ReadLock readLock = setWithLock.readLock();
			readLock.lock();
			try {
				Set<ChannelContext> set = setWithLock.getObj();
				boolean ret = false;
				for (ChannelContext channelContext : set) {
					boolean singleRet = false;
					// 不要用 a = a || b()，容易漏执行后面的函数
					if (isBlock) {
						singleRet = bSend(channelContext, packet);
					} else {
						singleRet = send(channelContext, packet);
					}
					if (singleRet) {
						ret = true;
					}
				}
				return ret;
			} catch (Throwable e) {
				log.error(e.getMessage(), e);
			} finally {
				readLock.unlock();
			}
			return false;
		} finally {
		}
	}

	/**
	 * 发消息给指定用户
	 * @param tioConfig
	 * @param userid
	 * @param packet
	 * @author tanyaowu
	 */
	public static Boolean sendToUser(TioConfig tioConfig, String userid, Packet packet) {
		return sendToUser(tioConfig, userid, packet, false);
	}

	/**
	 * 发消息给指定用户
	 * @param tioConfig
	 * @param userid
	 * @param packet
	 * @param isBlock
	 * @author tanyaowu
	 */
	private static Boolean sendToUser(TioConfig tioConfig, String userid, Packet packet, boolean isBlock) {
		SetWithLock<ChannelContext> setWithLock = tioConfig.users.find(tioConfig, userid);
		try {
			if (setWithLock == null) {
				return false;
			}

			ReadLock readLock = setWithLock.readLock();
			readLock.lock();
			try {
				Set<ChannelContext> set = setWithLock.getObj();
				boolean ret = false;
				for (ChannelContext channelContext : set) {
					boolean singleRet = false;
					// 不要用 a = a || b()，容易漏执行后面的函数
					if (isBlock) {
						singleRet = bSend(channelContext, packet);
					} else {
						singleRet = send(channelContext, packet);
					}
					if (singleRet) {
						ret = true;
					}
				}
				return ret;
			} catch (Throwable e) {
				log.error(e.getMessage(), e);
			} finally {
				readLock.unlock();
			}
			return false;
		} finally {
		}
	}

	/**
	 * 解绑业务id
	 * @param channelContext
	 * @author tanyaowu
	 */
	public static void unbindBsId(ChannelContext channelContext) {
		channelContext.tioConfig.bsIds.unbind(channelContext);
	}

	/**
	 * 与所有组解除解绑关系
	 * @param channelContext
	 * @author tanyaowu
	 */
	public static void unbindGroup(ChannelContext channelContext) {
		channelContext.tioConfig.groups.unbind(channelContext);
	}

	/**
	 * 与指定组解除绑定关系
	 * @param group
	 * @param channelContext
	 * @author tanyaowu
	 */
	public static void unbindGroup(String group, ChannelContext channelContext) {
		channelContext.tioConfig.groups.unbind(group, channelContext);
	}

	/**
	 * 将某用户从组中解除绑定
	 * @param tioConfig
	 * @param userid
	 * @param group
	 */
	public static void unbindGroup(TioConfig tioConfig, String userid, String group) {
		SetWithLock<ChannelContext> setWithLock = Tio.getByUserid(tioConfig, userid);
		if (setWithLock != null) {
			setWithLock.handle(new ReadLockHandler<Set<ChannelContext>>() {
				@Override
				public void handler(Set<ChannelContext> set) {
					for (ChannelContext channelContext : set) {
						Tio.unbindGroup(group, channelContext);
					}
				}
			});
		}
	}

	/**
	 * 解除channelContext绑定的token
	 * @param channelContext
	 * @author tanyaowu
	 */
	public static void unbindToken(ChannelContext channelContext) {
		channelContext.tioConfig.tokens.unbind(channelContext);
	}

	/**
	 * 解除token
	 * @param tioConfig
	 * @param token
	 */
	public static void unbindToken(TioConfig tioConfig, String token) {
		tioConfig.tokens.unbind(tioConfig, token);
	}

	//	org.tio.core.TioConfig.ipBlacklist

	/**
	 * 解除channelContext绑定的userid
	 * @param channelContext
	 * @author tanyaowu
	 */
	public static void unbindUser(ChannelContext channelContext) {
		channelContext.tioConfig.users.unbind(channelContext);
	}

	/**
	 * 解除userid的绑定。一般用于多地登录，踢掉前面登录的场景
	 * @param tioConfig
	 * @param userid
	 * @author: tanyaowu
	 */
	public static void unbindUser(TioConfig tioConfig, String userid) {
		tioConfig.users.unbind(tioConfig, userid);
	}

	private Tio() {
	}

}
