/*
    Copyright 2008,2009
        Matthias Ehmann,
        Michael Gerhaeuser,
        Carsten Miller,
        Bianca Valentin,
        Alfred Wassermann,
        Peter Wilfahrt

    This file is part of JSXGraph.

    JSXGraph is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    JSXGraph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with JSXGraph.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @fileoverview In this file the geometry object Ticks is defined. Ticks provides
 * methods for creation and management of ticks on an axis.
 * @author graphjs
 * @version 0.1
 */

/**
 * Creates ticks for an axis.
 * @class Ticks provides methods for creation and management
 * of ticks on an axis.
 * @param {JXG.Line} line Reference to the axis the ticks are drawn on.
 * @param {Number,Array,Function} ticks Number, array or function defining the ticks.
 * @param {int} major Every major-th tick is drawn with heightmajorHeight, the other ones are drawn with height minorHeight.
 * @param {int} majorHeight The height used to draw major ticks.
 * @param {int} minorHeight The height used to draw minor ticks.
 * @param {String} id Unique identifier for this object.  If null or an empty string is given,
 * an unique id will be generated by Board.
 * @param {String} name Not necessarily unique name, won't be visible or used by this object.
 * @see JXG.Board#addTicks
 * @constructor
 * @extends JXG.GeometryElement
 */
JXG.Ticks = function (line, ticks, minor, majorHeight, minorHeight, id, name) {
    /* Call the constructor of GeometryElement */
    this.constructor();

    /**
     * Type of GeometryElement, value is OBJECT_TYPE_ARC.
     * @final
     * @type int
     */
    this.type = JXG.OBJECT_TYPE_TICKS;

    /**
     * Class of the element, value is OBJECT_CLASS_CIRCLE.
     * @final
     * @type int
     */
    this.elementClass = JXG.OBJECT_CLASS_LINE;

    /**
     * The line the ticks belong to.
     * @type JXG.Line
     */
    this.line = line;

    /**
     * The board the ticks line is drawn on.
     * @type JXG.Board
     */
    this.board = this.line.board;

    /**
     * A function calculating ticks delta depending on the ticks number.
     * @type Function
     */
    this.ticksFunction = null;

    /**
     * Array of fixed ticks.
     * @type Array
     */
    this.fixedTicks = null;

    /**
     * Equidistant ticks. Distance is defined by ticksFunction
     * @type bool
     */
    this.equidistant = false;

    if(JXG.IsFunction(ticks))
        this.ticksFunction = ticks;
    else if(JXG.IsArray(ticks))
        this.fixedTicks = ticks;
    else {
        if(Math.abs(ticks) < JXG.Math.eps)
            ticks = this.board.options.line.ticks.defaultDistance;
        this.ticksFunction = function (i) { return ticks; };
        this.equidistant = true;
    }

    /**
     * minorTicks is the number of minor ticks between two major ticks.
     * @type int
     */
    this.minorTicks = ( (minor == null) || (minor == 0) ? this.board.options.line.ticks.minorTicks : minor);
    if(this.minorTicks < 0)
        this.minorTicks = -this.minorTicks;

    /**
     * Total height of a major tick.
     * @type int
     */
    this.majorHeight = ( (majorHeight == null) || (majorHeight == 0) ? this.board.options.line.ticks.majorHeight : majorHeight);
    if(this.majorHeight < 0)
        this.majorHeight = -this.majorHeight;

    /**
     * Total height of a minor tick.
     * @type int
     */
    this.minorHeight = ( (minorHeight == null) || (minorHeight == 0) ? this.board.options.line.ticks.minorHeight : minorHeight);
    if(this.minorHeight < 0)
        this.minorHeight = -this.minorHeight;

    /**
     * Least distance between two ticks, measured in pixels.
     * @type int
     */
    this.minTicksDistance = this.board.options.line.ticks.minTicksDistance;

    /**
     * Maximum distance between two ticks, measured in pixels. Is used only when insertTicks
     * is set to true.
     * @type int
     * @see #insertTicks
     */
    this.maxTicksDistance = this.board.options.line.ticks.maxTicksDistance;

    /**
     * If the distance between two ticks is too big we could insert new ticks. If insertTicks
     * is <tt>true</tt>, we'll do so, otherwise we leave the distance as is.
     * This option is ignored if equidistant is false.
     * @type bool
     * @see #equidistant
     * @see #maxTicksDistance
     */
    this.insertTicks = this.board.options.line.ticks.insertTicks;

    /**
     * Draw the zero tick, that lies at line.point1?
     * @type bool
     */
    this.drawZero = this.board.options.line.ticks.drawZero;

    /**
     * Draw labels yes/no
     * @type bool
     */
    this.drawLabels = this.board.options.line.ticks.drawLabels;

    /* Call init defined in GeometryElement to set board, id and name property */
    this.init(this.board, id, name);

    this.visProp['visible'] = true;

    this.visProp['fillColor'] = this.line.visProp['fillColor'];
    this.visProp['highlightFillColor'] = this.line.visProp['highlightFillColor'];
    this.visProp['strokeColor'] = this.line.visProp['strokeColor'];
    this.visProp['highlightStrokeColor'] = this.line.visProp['highlightStrokeColor'];
    this.visProp['strokeWidth'] = this.line.visProp['strokeWidth'];

    /* Register ticks at line. */
    this.id = this.line.addTicks(this);
};

JXG.Ticks.prototype = new JXG.GeometryElement;

/**
 * Always returns false.
 * @param {int} x Coordinate in x direction, screen coordinates.
 * @param {int} y Coordinate in y direction, screen coordinates.
 * @return {bool} Always returns false.
 */
JXG.Ticks.prototype.hasPoint = function (x, y) {
   return false;
};

/**
 * This function acutally calculates the tick for one direction.
 * @param {JXG.Point} start Determines the start point from where the ticks are drawn.
 * This point is equal to the zero point of the line.
 * @param {JXG.Point} end Point to which the ticks are drawn.
 * @param {int} direction This determines the labels signum.
 * @param {int} over This should be 1 if we aren't allowed to draw ticks beyond the end point (if the line's a segment)
 * and should be zero if ticks should be drawn until the end of drawing area (if the line's a straight line).
 */
JXG.Ticks.prototype.makeTicks = function(start, end, direction, over) {
    // distance between start and end points
    var dx = start.usrCoords[1]-end.usrCoords[1]; // delta x
    var dy = start.usrCoords[2]-end.usrCoords[2]; // delta y

    // the current distance between two ticks
    var ticksDelta = 0;
    // the length of the axis between c1 and p1
    var total_length = Math.sqrt(dx*dx + dy*dy);

    if (total_length<=JXG.Math.eps)
        return;

    // x and y store the coordinates of the current tick to add
    var x = start.usrCoords[1];
    var y = start.usrCoords[2];

    // i is the amount of ticks drawn until now.
    var i = direction/Math.abs(direction);
    ticksDelta = Math.abs(this.ticksFunction(i));

    // this calculates the difference between the last and the current tick in
    // x and y direction in user coordinates
    var deltaX = (ticksDelta * dx) / (total_length);
    var deltaY = (ticksDelta * dy) / (total_length);
    
    // the current distance between two ticks in SCREEN coordinates
    // this is used to omit ticks, if they are too close to each other
    // and to add new ticks, if they are too far away from each other.
    var dist = 0;

    // if we have equidistant ticks do some delta-correction before we enter the main loop
    if(this.equidistant) {
        var deltaX_original = deltaX;
        var deltaY_original = deltaY;
        var ticksDelta_original = ticksDelta;
        
        var zero = new JXG.Coords(JXG.COORDS_BY_USER, [0, 0], this.board);
        var tmpCoords = new JXG.Coords(JXG.COORDS_BY_USER, [deltaX, deltaY], this.board);
        dist = (tmpCoords.scrCoords[1]-zero.scrCoords[1])*(tmpCoords.scrCoords[1]-zero.scrCoords[1]) + (tmpCoords.scrCoords[2]-zero.scrCoords[2])*(tmpCoords.scrCoords[2]-zero.scrCoords[2]);


        while(dist < this.minTicksDistance*this.minTicksDistance) {
            deltaX += deltaX_original;
            deltaY += deltaY_original;
            ticksDelta += ticksDelta_original;

            tmpCoords = new JXG.Coords(JXG.COORDS_BY_USER, [deltaX, deltaY], this.board);
            dist = (tmpCoords.scrCoords[1]-zero.scrCoords[1])*(tmpCoords.scrCoords[1]-zero.scrCoords[1]) + (tmpCoords.scrCoords[2]-zero.scrCoords[2])*(tmpCoords.scrCoords[2]-zero.scrCoords[2]);
        }
    }

    // position is the current position on the axis
    var position = direction*ticksDelta;

    // reference to the last added tick coordinates object
    var lastTick = new JXG.Coords(JXG.COORDS_BY_USER, [x,y], this.board);

    var newTick = null;

    // minor ticks
    var minTick = null;

    var labelText = '';

    // temporary label object to create labels for ticks
    var label = null;

    // run the loop at least one time
    var first = true;

    // we also need access to the last ticks coordinates
    // store the old coordinates
    var lastX = x;
    var lastY = y;


    // loop abort criteria is:
    // our next tick is completely out of sight.
    while( first || (((this.board.sgn(deltaX)*(x-over*deltaX) >= this.board.sgn(deltaX)*end.usrCoords[1]) && (this.board.sgn(deltaY)*(y-over*deltaY) >= this.board.sgn(deltaY)*end.usrCoords[2]))) ) {
        // we're in it, so we have at least one tick and the deltaX/Y correction
        // for equidistant ticks.
        first = false;

        // calculate the new ticks coordinates
        x = x - deltaX;
        y = y - deltaY;

        // and put them into a coords object
        newTick = new JXG.Coords(JXG.COORDS_BY_USER, [x,y], this.board);

        // we need to calculate the distance. if we're in equidistant mode, we only need
        // to calculate it once, otherwise on every walk through the loop.
        if(!this.equidistant) {
            dist = (lastTick.scrCoords[1]-newTick.scrCoords[1])*(lastTick.scrCoords[1]-newTick.scrCoords[1]) +
                   (lastTick.scrCoords[2]-newTick.scrCoords[2])*(lastTick.scrCoords[2]-newTick.scrCoords[2]);
        }

        // if we're in equidistant mode and want to insert additional ticks automatically, whenever
        // the distance between two ticks is too big, we need to calculate the new deltaX and deltaY.
        if(this.insertTicks && this.equidistant && (dist > this.maxTicksDistance*this.maxTicksDistance)) {
            // dist is indeed the distance squared. repeat this, until we fall below the maxTicksDistance limit
            while (dist > this.maxTicksDistance*this.maxTicksDistance) {
                // half the distance
                deltaX = deltaX/2;
                deltaY = deltaY/2;
                // move back towards the zeropoint
                x = x + deltaX;
                y = y + deltaY;

                ticksDelta = ticksDelta/2;
                position = position - direction*ticksDelta;

                // recalculate newTick coordinates and distance
                newTick = new JXG.Coords(JXG.COORDS_BY_USER, [x,y], this.board);
                dist = (lastTick.scrCoords[1]-newTick.scrCoords[1])*(lastTick.scrCoords[1]-newTick.scrCoords[1]) +
                       (lastTick.scrCoords[2]-newTick.scrCoords[2])*(lastTick.scrCoords[2]-newTick.scrCoords[2]);
            }
        }

        if(this.equidistant) {
            for(var z=1; z<this.minorTicks+1; z++) {
                minTick = new JXG.Coords(JXG.COORDS_BY_USER, [lastX - (deltaX*z)/(this.minorTicks+1) ,lastY - (deltaY*z)/(this.minorTicks+1)], this.board);
                minTick.major=false;
                this.ticks.push(minTick);
                this.labels.push(null);
            }
        }

        // if we're not below the minimum distance between two ticks, add the tick to our list
        if(this.equidistant || (dist > this.minTicksDistance*this.minTicksDistance)) {
            newTick.major = true;

            this.ticks.push(newTick);
            labelText = position.toString();
            if(labelText.length > 5)
                labelText = position.toPrecision(3).toString();
            label = new JXG.Label(this.board, labelText, newTick, this.id+i+"Label");
            label.distanceX = 0;
            label.distanceY = -10;
            label.setCoordinates(newTick);
            if (this.drawLabels) label.show = true; else label.show = false;
            this.labels.push(label);

            // store the old coordinates
            lastX = x;
            lastY = y;

            lastTick = newTick;
        }

        i = i + direction*1;
        
        // if not equidistant, calculate the distance to the next tick
        if(!this.equidistant) {
            ticksDelta = Math.abs(this.ticksFunction(i));
        }
        
        // calculate the ticks label text data
        position = position + direction*ticksDelta;

        // recalculate new delta* only if needed
        if(!this.equidistant) {
            deltaX = (ticksDelta * dx) / (total_length);
            deltaY = (ticksDelta * dy) / (total_length);
        }
    }
}

/**
 * (Re-)calculates the ticks coordinates.
 */
JXG.Ticks.prototype.calculateTicksCoordinates = function() {
    // in which direction do we have to go?
    // DIR_PLUS is from line.p1 to line.p2, DIR_MINUS
    // is the other direction.
    var DIR_MINUS = 1;
    var DIR_PLUS = 2;
    var required = DIR_MINUS + DIR_PLUS;

    // calculate start (c1) and end (c2) points
    // copy existing lines point coordinates
    var c1 = new JXG.Coords(JXG.COORDS_BY_USER, [this.line.point1.coords.usrCoords[1], this.line.point1.coords.usrCoords[2]], this.board);
    var c2 = new JXG.Coords(JXG.COORDS_BY_USER, [this.line.point2.coords.usrCoords[1], this.line.point2.coords.usrCoords[2]], this.board);

    // now let the renderer calculate start and end point of the line on the board, i.e.
    // intersection points of line with the boards edges in the case that the line is a straight.
    this.board.renderer.calcStraight(this.line, c1, c2);
    
    // point1 is our reference point (where zero lies on the axis)
    var p1 = this.line.point1.coords;

    // first we have to look at some special cases, i.e. if our zero point on line
    // is outside the viewable area, there are some ticks we don't need to look at.
    if(this.board.renderer.isSameDirection(p1, c1, c2)) {
        if(this.board.renderer.isSameDirection(p1, this.line.point2.coords, c1)) {
            required = DIR_PLUS;
            if(p1.distance(JXG.COORDS_BY_USER, c1) > p1.distance(JXG.COORDS_BY_USER, c2))
                c2 = c1;
        } else {
            required = DIR_MINUS;
            if(p1.distance(JXG.COORDS_BY_USER, c1) < p1.distance(JXG.COORDS_BY_USER, c2))
                c1 = c2;
        }
    } else /* p1 can be seen on the drawing area */ {
        // make sure, point2 is on that part of the line with positive tick markers on it
        if(this.board.renderer.isSameDirection(p1, this.line.point2.coords, c1)) {
            var ct = c1;
            c1 = c2;
            c2 = ct;
        }
    }
    
    if(this.ticks != null) {
        for(var j=0; j<this.ticks.length; j++) {
            if(this.labels[j] != null) {
                if (this.labels[j].show) this.board.renderer.remove(this.labels[j].rendNode);
            }
        }
    }

    // initialise storage arrays
    // ticks stores the ticks coordinates
    this.ticks = new Array();
    // labels stores the text to display beside the ticks
    this.labels = new Array();

    var label = null;
    var labelText = '';

    // reference to the newly added tick coordinates object
    var newTick = null;
    if(this.ticksFunction != null) {
        // add tick at p1?
        if(this.drawZero) {
            newTick = new JXG.Coords(JXG.COORDS_BY_USER, [p1.usrCoords[1], p1.usrCoords[2]], this.board);
            this.ticks.push(newTick);
            label = new JXG.Label(this.board, "0", newTick, this.id+"0Label");
            if (this.drawLabels) label.show = true; else label.show = false;
            this.labels.push(label);

            this.ticks[0].major = true;
        }

        if(DIR_MINUS == (required & DIR_MINUS)) {
            if(this.line.visProp['straightFirst'])
                this.makeTicks(p1, c1, -1, 0);
// the negative part even doesn't exist on this line...
//            else
//                this.makeTicks(p1, c1, -1, 1);
        }
        if(DIR_PLUS == (required & DIR_PLUS)) {
            if(this.line.visProp['straightLast'])
                this.makeTicks(p1, c2, +1, 0);
            else {
                this.makeTicks(p1, this.line.point2.coords, +1, 1);
            }
        }
    } else {
        // we have an array of fixed ticks we have to draw
        if(!this.line.visProp['straightFirst'])
            c1 = p1;
        var dx_minus = p1.usrCoords[1]-c1.usrCoords[1];
        var dy_minus = p1.usrCoords[2]-c1.usrCoords[2];
        var length_minus = Math.sqrt(dx_minus*dx_minus + dy_minus*dy_minus);

        if(!this.line.visProp['straightLast'])
            c2 = this.line.point2.coords;
        var dx_plus = p1.usrCoords[1]-c2.usrCoords[1];
        var dy_plus = p1.usrCoords[2]-c2.usrCoords[2];
        var length_plus = Math.sqrt(dx_plus*dx_plus + dy_plus*dy_plus);

        // new ticks coordinates
        var nx = 0;
        var ny = 0;

        for(var i=0; i<this.fixedTicks.length; i++) {
            // is this tick visible?
            if((-length_minus <= this.fixedTicks[i]) && (this.fixedTicks[i] <= length_plus)) {
                if(this.fixedTicks[i] < 0) {
                    nx = Math.abs(dx_minus) * this.fixedTicks[i]/length_minus;
                    ny = Math.abs(dy_minus) * this.fixedTicks[i]/length_minus;
                } else {
                    nx = Math.abs(dx_plus) * this.fixedTicks[i]/length_plus;
                    ny = Math.abs(dy_plus) * this.fixedTicks[i]/length_plus;
                }

                newTick = new JXG.Coords(JXG.COORDS_BY_USER, [p1.usrCoords[1] + nx, p1.usrCoords[2] + ny], this.board);
                this.ticks.push(newTick);
                this.ticks[this.ticks.length-1].major = true;
                labelText = this.fixedTicks[i].toString();
                if(labelText.length > 5)
                    labelText = this.fixedTicks[i].toFixed(3).toString();
                label = new JXG.Label(this.board, labelText, newTick, this.id+i+"Label");
                label.distanceX = 0;
                label.distanceY = -10;
                label.setCoordinates(newTick);
                if (this.drawLabels) label.show = true; else label.show = false;
                this.labels.push(label);
            }
        }
    }

    // this piece of code was in AbstractRenderer.updateAxisTicksInnerLoop
    // and has been moved in here to clean up the code.
    var eps = JXG.Math.eps;
    var slope = -this.line.getSlope();

    var distMaj = this.majorHeight/2;
    var distMin = this.minorHeight/2;
    var dxMaj = 0; var dyMaj = 0;
    var dxMin = 0; var dyMin = 0;

    if(Math.abs(slope) < eps) {
        // if the slope of the line is (almost) 0, we can set dx and dy directly
        dxMaj = 0;
        dyMaj = distMaj;
        dxMin = 0;
        dyMin = distMin;
    } else if((Math.abs(slope) > 1/eps) || (isNaN(slope))) {
        // if the slope of the line is (theoretically) infinite, we can set dx and dy directly
        dxMaj = distMaj;
        dyMaj = 0;
        dxMin = distMin;
        dyMin = 0;
    } else {
        // here we have to calculate dx and dy depending on the slope and the length of the tick (dist)
        // if slope is the line's slope, the tick's slope is given by
        //
        //            1          dy
        //     -   -------  =   ----                 (I)
        //          slope        dx
        //
        // when dist is the length of the tick, using the pythagorean theorem we get
        //
        //     dx*dx + dy*dy = dist*dist             (II)
        //
        // dissolving (I) by dy and applying that to equation (II) we get the following formulas for dx and dy
        dxMaj = distMaj/Math.sqrt(1/(slope*slope) + 1);
        dyMaj = -dxMaj/slope;
        dxMin = distMin/Math.sqrt(1/(slope*slope) + 1);
        dyMin = -dxMin/slope;
    }
    this.board.renderer.updateTicks(this,dxMaj,dyMaj,dxMin,dyMin);
};

/**
 * Uses the boards renderer to update the arc.
 * update() is not needed for arc.
 */
 JXG.Ticks.prototype.updateRenderer = function () {
    if (this.needsUpdate) {
        this.calculateTicksCoordinates();
        this.needsUpdate = false;
    }
};

/**
 * Creates new ticks.
 * @param {JXG.Board} board The board the ticks are put on.
 * @param {Array} parents Array containing a line and an array of positions, where ticks should be put on that line or
 *   a function that calculates the distance based on the ticks number that is given as a parameter. E.g.:<br />
 *   <tt>var ticksFunc = function(i) {</tt><br />
 *   <tt>    return 2;</tt><br />
 *   <tt>}</tt><br />
 *   for ticks with distance 2 between each tick.
 * @param {Object} attributs Object containing properties for the element such as stroke-color and visibility. See @see JXG.GeometryElement#setProperty
 * @type JXG.Ticks
 * @return Reference to the created ticks object.
 */
JXG.createTicks = function(board, parents, attributes) {
    var el;
    if ( (parents[0].elementClass == JXG.OBJECT_CLASS_LINE) && (JXG.IsFunction(parents[1]) || JXG.IsArray(parents[1]) || JXG.IsNumber(parents[1]))) {
        el = new JXG.Ticks(parents[0], parents[1], attributes['minorTicks'], attributes['majHeight'], attributes['minHeight'], attributes['id'], attributes['name']);
    } else
        throw ("Can't create Ticks with parent types '" + (typeof parents[0]) + "' and '" + (typeof parents[1]) + "' and '" + (typeof parents[2]) + "'.");

    return el;
};

JXG.JSXGraph.registerElement('ticks', JXG.createTicks);