/**
 * A extension for xuanxuan integrate with zentao server
 * 禅道集成喧喧扩展，该扩展应该上传到禅道服务器通过远程扩展的形式推送给客户端安装使用，
 * 如果是通过本地扩展的形式加载的则适用于进行调试
 */

const isWindows7 = window.navigator.userAgent.includes('Windows NT 6.');

// Current extension object
// 当前扩展对象
let extension = null;

// Check the local mode for debug
// 用于判断当前扩展是否是以本地扩展的形式加载的
let isLocalMode = false;

// Application entry url for local mode
// 应用入口地址（适用于本地扩展模式）
const entryUrlForLocal = 'http://pms.zentao.net';

// Application entry url object
// 应用入口网址对象
let entryUrlObj = null;

// Application url configurations map
// 应用网址渲染卡片配置（适用于本地扩展模式）
// 对象的属性名称为网址要批评的路径开始的部分，如果值为 true，则表示渲染为卡片，
// 如果值为一个对象，则表示渲染为卡片时所用的配置，该对象目前允许包含如下属性：
// - width: 卡片的宽度
// - height: 卡片的高度
const urlMapForLocal = {
    '/task-view-': {width: '400px', height: '300px'},
    '/bug-view-': {width: '400px', height: '400px'},
    '/story-view-': true, // 使用默认宽度和高度渲染卡片
};

// Actions urls map for local mode
// 操作链接地址清单（适用于本地扩展模式）
// 属性为操作名称，属性值为一个对象，包含如下属性：
// - title: 操作标题
// - url: 用户执行此操作打开的网址
// - width: 用户执行此操作打开的对话框宽度
// - height: 用户执行此操作打开的对话框高度
// 在对应的操作页面上添加一个 window.injectWithXXC 方法来执行填充表单操作，例如：
// window.injectWithXXC = function(message) {
//    console.log('消息', message);
//    // 在此处将 message 填充到页面表单上
// }
// 以上 message 对象包含如下属性：
// - type：消息内容类型，包括 `text`（纯文本）、`markdown`（Markdown）、`url`（网址）
// - content： 消息内容
const actionUrlsForLocal = {
    createBug: {
        title: null, url: `${entryUrlForLocal}/bug-create-1.html`, width: '800px', height: '600px'
    },
    createTask: {
        title: null, url: `${entryUrlForLocal}/task-create-1.html`, width: '800px', height: '700px'
    },
    createStory: {
        title: null, url: `${entryUrlForLocal}/story-create-1.html`, width: '800px', height: '700px'
    },
    createTodo: {
        title: null, url: `${entryUrlForLocal}/todo-create-1.html`, width: '800px', height: '700px'
    },
    createDoc: {
        title: null, url: `${entryUrlForLocal}/doc-create-1.html`
    },
};

// Save the url map setting
// 存储网址渲染卡片配置
let urlMap = null;

// Actions urls map
// 操作链接地址清单
let actionUrls = null;

/**
 * Get url map setting object
 * 获取指定网址的卡片渲染配置
 * @param {string} url Url address 要处理的网址
 * @return {boolean|Object} Return map setting object 返回卡片渲染配置对象
 */
const getUrlMapSetting = url => {
    const urlObj = new URL(url);
    const {
        pathname, hostname, search, hash
    } = urlObj;
    if (!entryUrlObj || hostname !== entryUrlObj.hostname) {
        return null;
    }
    if (!urlMap) {
        return null;
    }
    const urlSuffix = [pathname, search, hash].join('');
    for (const startPath of Object.keys(urlMap)) {
        if (urlSuffix.startsWith(startPath)) {
            return Object.assign({}, urlMap[startPath]);
        }
    }
    return null;
};

/**
 * 在对话框中打开链接
 * @param {string} url 要打开的链接
 * @param {?Object} options 对话框视图选项
 * @return {void}
 */
const openUrlInDialog = (url, options) => {
    extension.getEntryUrl(url).then(visitUrl => {
        global.Xext.app.ui.openUrlInDialog(visitUrl, options);
        return visitUrl;
    }).catch(error => {
        global.Xext.app.ui.showMessger(
            `无法从远程获取访问地址。${global.DEBUG ? `${global.Xext.lang.error(error)}(${url})` : ''}`,
            {type: 'danger', autoHide: true}
        );
    });
};

/**
 * 右键菜单生成器清单
 * @type {Object[]}
 * @private
 */
const contextMenuCreators = [];

// Export extention module
// 用于导出扩展模块
module.exports = {

    // Callback for extension attach to application
    // 当扩展被加载到应用时的回调函数
    onAttach: ext => {
        // Save extension object
        // 保存扩展对象
        extension = ext;

        // 检查是否是本地扩展（非远程扩展）
        isLocalMode = !ext.isRemote;

        if (isLocalMode) {
            extension._webViewUrl = entryUrlForLocal;
            entryUrlObj = new URL(entryUrlForLocal);
            urlMap = urlMapForLocal;
            actionUrls = actionUrlsForLocal;
        } else {
            const entryUrl = extension.serverData.entryUrl || extension.entryUrl;
            extension._webViewUrl = entryUrl;
            entryUrlObj = new URL(entryUrl);
            urlMap = extension.serverData.urls;
            actionUrls = extension.serverData.actions;
        }

        if (!actionUrls || !Object.keys(actionUrls).length) {
            return;
        }
        contextMenuCreators.push({
            // 集成文本消息右键菜单
            match: 'message.text',
            create: context => {
                const {message} = context;
                const contentElement = document.getElementById(`message-content-${message.gid}`);
                const messageContent = contentElement ? contentElement.innerText : message.content;
                const messageData = {type: message.isPlainTextContent ? 'text' : 'markdown', content: message.content};
                const executeJavaScript = `window.injectXXCData=${JSON.stringify(messageData)};window.injectWithXXC && window.injectWithXXC(window.injectXXCData)`;
                const items = [];
                if (actionUrls.createBug) {
                    items.push({
                        noLabelPrefix: true,
                        icon: 'mdi-bug',
                        label: actionUrls.createBug.title || {'zh-cn': '创建 Bug', 'zh-tw': '創建 Bug', en: 'Create Bug'},
                        click: () => openUrlInDialog(actionUrls.createBug.url, {
                            width: actionUrls.createBug.width || '900px',
                            height: actionUrls.createBug.height || '700px',
                            injectForm: {'#title': messageContent, $input: '#title'},
                            executeJavaScript
                        }),
                    });
                }
                if (actionUrls.createStory) {
                    items.push({
                        noLabelPrefix: true,
                        icon: 'mdi-lightbulb-on-outline',
                        label: actionUrls.createStory.title || {'zh-cn': '创建需求', 'zh-tw': '創建需求', en: 'Create Story'},
                        click: () => openUrlInDialog(actionUrls.createStory.url, {
                            width: actionUrls.createStory.width || '900px',
                            height: actionUrls.createStory.height || '700px',
                            injectForm: {'#title': messageContent, $input: '#title'},
                            executeJavaScript
                        }),
                    });
                }
                if (actionUrls.createTask) {
                    items.push({
                        noLabelPrefix: true,
                        icon: 'mdi-calendar-check',
                        label: actionUrls.createTask.title || {'zh-cn': '创建任务', 'zh-tw': '創建任務', en: 'Create Task'},
                        click: () => openUrlInDialog(actionUrls.createTask.url, {
                            width: actionUrls.createTask.width || '900px',
                            height: actionUrls.createTask.height || '700px',
                            injectForm: {'#name': messageContent, $input: '#name'},
                            executeJavaScript
                        }),
                    });
                }
                if (actionUrls.createDoc) {
                    items.push({
                        noLabelPrefix: true,
                        icon: 'mdi-file-document',
                        label: actionUrls.createDoc.title || {'zh-cn': '创建文档', 'zh-tw': '創建文檔', en: 'Create Document'},
                        click: () => openUrlInDialog(actionUrls.createDoc.url, {
                            width: actionUrls.createDoc.width || '900px',
                            height: actionUrls.createDoc.height || '700px',
                            injectForm: {'#url': messageContent, $input: '#url'},
                            executeJavaScript
                        }),
                    });
                }
                if (actionUrls.createTodo) {
                    items.push({
                        noLabelPrefix: true,
                        icon: 'mdi-check-circle-outline',
                        label: actionUrls.createTodo.title || {'zh-cn': '创建待办', 'zh-tw': '創建待辦', en: 'Create Todo'},
                        click: () => openUrlInDialog(actionUrls.createTodo.url, {
                            width: actionUrls.createTodo.width || '900px',
                            height: actionUrls.createTodo.height || '700px',
                            injectForm: {'#name': messageContent, $input: '#name'},
                            executeJavaScript
                        }),
                    });
                }
                return items;
            }
        });

        if (actionUrls.createDoc) {
            contextMenuCreators.push({
                // 集成网址右键菜单
                match: 'link',
                create: context => {
                    const {event, options} = context;
                    const link = options && options.url ? options.url : event.target.href;
                    const messageData = {type: 'url', content: link};
                    const executeJavaScript = `window.injectXXCData=${JSON.stringify(messageData)};window.injectWithXXC && window.injectWithXXC(window.injectXXCData)`;
                    return [{
                        icon: 'mdi-file-document',
                        noLabelPrefix: true,
                        label: actionUrls.createDoc.title || {'zh-cn': '创建文档', 'zh-tw': '創建文檔', en: 'Create Document'},
                        click: () => openUrlInDialog(actionUrls.createDoc.url, {
                            width: actionUrls.createDoc.width || '900px',
                            height: actionUrls.createDoc.height || '700px',
                            injectForm: {'#name': link, '#url': link, $input: '#title'},
                            executeJavaScript
                        }),
                    }];
                }
            });
        }
    },

    // Url inspectors
    // 网址解释器，可以将消息中的网址渲染成卡片形式。`urlInspectors` 为一个对象数组，每个对象包含有 `test` 属性为正则表           达式用于匹配要解释的 url 地址，`inspector` 为回调函数（`function(url: string)`）用于生成 URL 对应的卡片参数。
    urlInspectors: [{

        // A function to test the url is it possible to render with card.
        // 使用函数来判断用户发送的网址是否可以渲染为自定义卡片
        test: url => !!getUrlMapSetting(url),

        // Get url with free-login-authentication method
        // 通过 entry/visit 接口获取免登录地址
        getUrl: url => {
            // Get free-login-authentication entry url
            if (isLocalMode) {
                return url;
            }
            return extension.getEntryUrl(url);
        },

        // Skip xuanxuan meta origin request
        // 忽略默认的获取卡片渲染信息行为
        noMeta: true,

        // Inspect url and return card config data
        // 解析网址并返回卡片配置信息
        inspect: (url, originUrl) => {
            // Card meta data
            // 卡片配置对象
            const cardMeta = {};

            // Setting card basic properties
            // 设置卡片基本属性
            cardMeta.title = null;
            cardMeta.webviewContent = true;
            cardMeta.icon = false;

            const urlMapSetting = getUrlMapSetting(originUrl) || {};

            cardMeta.content = {
                // Set webview url
                // 设置原始网址
                originSrc: url.includes('?') ? `${url}&display=app` : `${url}?display=app`,

                // Set webview card url
                // 设置卡片页面网址
                src: url.includes('?') ? `${url}&display=card` : `${url}?display=card`,

                // Set webview card size
                // 设置卡片尺寸
                style: Object.assign({width: (urlMapSetting.width || '550px'), height: (urlMapSetting.height || '400px')}, urlMapSetting.style),

                // Set webview card type
                // 设置卡片渲染形式
                type: isWindows7 ? 'iframe' : 'webview'
            };
            // Return card meta setting
            // 返回卡片配置数据
            return cardMeta;
        }
    }],

    // Contextmenu creators
    // 右键菜单集成
    contextMenuCreators
};
