<?php
// +----------------------------------------------------------------------
// | ThinkPHP [ WE CAN DO IT JUST THINK ]
// +----------------------------------------------------------------------
// | Copyright (c) 2006~2017 http://thinkphp.cn All rights reserved.
// +----------------------------------------------------------------------
// | Licensed ( http://www.apache.org/licenses/LICENSE-2.0 )
// +----------------------------------------------------------------------
// | Author: liu21st <liu21st@gmail.com>
// +----------------------------------------------------------------------

namespace think;

use think\exception\ClassNotFoundException;

class Log implements LoggerInterface
{
    const EMERGENCY = 'emergency';
    const ALERT     = 'alert';
    const CRITICAL  = 'critical';
    const ERROR     = 'error';
    const WARNING   = 'warning';
    const NOTICE    = 'notice';
    const INFO      = 'info';
    const DEBUG     = 'debug';
    const SQL       = 'sql';

    // 日志信息
    protected $log = [];
    // 配置参数
    protected $config = [];
    // 日志写入驱动
    protected $driver;
    // 当前日志授权key
    protected $key;

    protected $app;

    public function __construct(App $app)
    {
        $this->app = $app;
    }

    /**
     * 日志初始化
     * @param array $config
     * @return $this
     */
    public function init($config = [])
    {
        $type  = isset($config['type']) ? $config['type'] : 'File';
        $class = false !== strpos($type, '\\') ? $type : '\\think\\log\\driver\\' . ucwords($type);

        $this->config = $config;

        unset($config['type']);

        if (class_exists($class)) {
            $this->driver = new $class($config);
        } else {
            throw new ClassNotFoundException('class not exists:' . $class, $class);
        }

        // 记录初始化信息
        $this->app->isDebug() && $this->record('[ LOG ] INIT ' . $type);

        return $this;
    }

    /**
     * 获取日志信息
     * @param string $type 信息类型
     * @return array
     */
    public function getLog($type = '')
    {
        return $type ? $this->log[$type] : $this->log;
    }

    /**
     * 记录日志信息
     * @param mixed  $msg       日志信息
     * @param string $type      日志级别
     * @param array  $context   替换内容
     * @return $this
     */
    public function record($msg, $type = 'info', array $context = [])
    {
        if (is_string($msg)) {
            $replace = [];
            foreach ($context as $key => $val) {
                $replace['{' . $key . '}'] = $val;
            }

            $msg = strtr($msg, $replace);
        }

        $this->log[$type][] = $msg;

        if (PHP_SAPI == 'cli') {
            // 命令行日志实时写入
            $this->save();
        }

        return $this;
    }

    /**
     * 清空日志信息
     * @return $this
     */
    public function clear()
    {
        $this->log = [];

        return $this;
    }

    /**
     * 当前日志记录的授权key
     * @param string  $key  授权key
     * @return $this
     */
    public function key($key)
    {
        $this->key = $key;

        return $this;
    }

    /**
     * 检查日志写入权限
     * @param array  $config  当前日志配置参数
     * @return bool
     */
    public function check($config)
    {
        if ($this->key && !empty($config['allow_key']) && !in_array($this->key, $config['allow_key'])) {
            return false;
        }

        return true;
    }

    /**
     * 保存调试信息
     * @return bool
     */
    public function save()
    {
        if (!empty($this->log)) {
            if (is_null($this->driver)) {
                $this->init($this->app['config']->pull('log'));
            }

            if (!$this->check($this->config)) {
                // 检测日志写入权限
                return false;
            }

            if (empty($this->config['level'])) {
                // 获取全部日志
                $log = $this->log;
                if (!$this->app->isDebug() && isset($log['debug'])) {
                    unset($log['debug']);
                }
            } else {
                // 记录允许级别
                $log = [];
                foreach ($this->config['level'] as $level) {
                    if (isset($this->log[$level])) {
                        $log[$level] = $this->log[$level];
                    }
                }
            }

            $result = $this->driver->save($log);
            if ($result) {
                $this->log = [];
            }

            return $result;
        }

        return true;
    }

    /**
     * 实时写入日志信息 并支持行为
     * @param mixed  $msg   调试信息
     * @param string $type  日志级别
     * @param bool   $force 是否强制写入
     * @return bool
     */
    public function write($msg, $type = 'info', $force = false)
    {
        // 封装日志信息
        $log = $this->log;

        if (true === $force || empty($this->config['level'])) {
            $log[$type][] = $msg;
        } elseif (in_array($type, $this->config['level'])) {
            $log[$type][] = $msg;
        } else {
            return false;
        }

        // 监听log_write
        $this->app['hook']->listen('log_write', $log);

        if (is_null($this->driver)) {
            $this->init($this->app['config']->pull('log'));
        }

        // 写入日志
        $result = $this->driver->save($log);

        if ($result) {
            $this->log = [];
        }

        return $result;
    }

    /**
     * 记录日志信息
     * @param string $level     日志级别
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function log($level, $message, array $context = [])
    {
        $this->record($message, $level, $context);
    }

    /**
     * 记录emergency信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function emergency($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录警报信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function alert($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录紧急情况
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function critical($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录错误信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function error($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录warning信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function warning($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录notice信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function notice($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录一般信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function info($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录调试信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function debug($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }

    /**
     * 记录sql信息
     * @param mixed  $message   日志信息
     * @param array  $context   替换内容
     * @return void
     */
    public function sql($message, array $context = [])
    {
        $this->log(__FUNCTION__, $message, $context);
    }
}
