<?php

namespace Edwin404\Shop\Jobs;


use Edwin404\Base\Support\BaseJob;
use Edwin404\Base\Support\ModelHelper;
use Edwin404\Shop\Helpers\ShopLockerHelper;
use Edwin404\Shop\Types\OrderStatus;
use Illuminate\Support\Facades\Log;

class OrderCancelJob extends BaseJob
{
    public $orderId;
    public $param;

    public static function create($orderId, $param = [], $delay = 0)
    {
        $job = new OrderCancelJob();
        $job->orderId = $orderId;
        $job->param = $param;
        $job->onQueue('OrderCancelJob');
        if ($delay > 0) {
            $job->delay($delay);
        }
        app('Illuminate\Contracts\Bus\Dispatcher')->dispatch($job);
    }

    public function handle()
    {
        Log::info('OrderCancelJob.' . sprintf('%05d', getmypid()) . '.Execute ->' . $this->orderId.' -> '.json_encode($this->param));
        
        ModelHelper::transactionBegin();
        if (!ShopLockerHelper::checkoutLockBegin()) {
            self::create($this->orderId, 5);
            return;
        }
        $order = ModelHelper::loadWithLock('order', ['id' => $this->orderId]);
        if (empty($order)) {
            ModelHelper::transactionCommit();
            ShopLockerHelper::checkoutLockEnd();
            return;
        }
        if ($order['status'] != OrderStatus::CANCEL_QUEUE) {
            ModelHelper::transactionCommit();
            ShopLockerHelper::checkoutLockEnd();
            return;
        }
        $orderGoods = ModelHelper::model('order_goods')->where(['orderId' => $this->orderId])->lockForUpdate()->get()->toArray();
        ModelHelper::modelJoin($orderGoods, 'goodsSnapshotId', '_goodsSnapshot', 'goods_snapshot', 'id');

        foreach ($orderGoods as $orderGood) {
            if ($orderGood['_goodsSnapshot']['specSpec']) {
                // 包含规格的商品
                ModelHelper::change(
                    'goods_spec',
                    ['goodsId' => $orderGood['goodsId'], 'spec' => $orderGood['_goodsSnapshot']['specSpec'],],
                    'stock',
                    +$orderGood['amount']
                );
                Log::info('goods.' . $orderGood['goodsId'] . '.' . $orderGood['_goodsSnapshot']['specSpec'] . ' stock +' . $orderGood['amount']);
            } else {
                // 不包含规格的商品
                ModelHelper::change(
                    'goods',
                    ['id' => $orderGood['goodsId']],
                    'stock',
                    +$orderGood['amount']
                );
                Log::info('goods.' . $orderGood['goodsId'] . ' stock +' . $orderGood['amount']);
            }
        }

        if (!empty($this->param['expire'])) {
            ModelHelper::updateOne('order', ['id' => $this->orderId], ['status' => OrderStatus::CANCEL_EXPIRED]);
        } else {
            ModelHelper::updateOne('order', ['id' => $this->orderId], ['status' => OrderStatus::CANCEL]);
        }
        ModelHelper::transactionCommit();
        ShopLockerHelper::checkoutLockEnd();

        Event::fire(new ShopOrderExpireEvent($this->orderId));

    }
}