<?php

if(!defined('SCRIPT_ROOT_PATH'))
{
	trigger_error('SCRIPT_ROOT_PATH not defined in RewriteEngine class');
}
else if(!class_exists('files'))
{
	require_once (SCRIPT_ROOT_PATH.'/class/libs/lib.files.php');
}

/*
*	RewriteEngine class. Rewrites URLs and creates HTaccess on demand by loading default htaccess.
*	@class	RewriteEngine
*	@package	Kernel
*	@author	Jerome Loisel
*/
class RewriteEngine
{
	// Is url rewrite enabled ?
	var $rewrite_enabled = false;
	// current htaccess content
	var $htaccess = '';
	// Default htaccess to load into $this->htaccess
	var $default_file = '';
	
	/*
	*	Constructor
	*	@function	RewriteEngine
	*	@param	boolean	rewrite enabled ? (optional)
	*	@param	string	default htaccess to load
	*/
	function RewriteEngine($rewrite_enabled = false, $default_file)
	{
		$this->rewrite_enabled = $rewrite_enabled;
		$this->htaccess = '';
		$this->default_file = $default_file;
		$this->loadDefaultHtaccess();
	}
	
	/*
	*	is URL Rewrite enabled ?
	*	@function	isEnabled
	*	@return	boolean	TRUE if enabled
	*/
	function isRewriteEnabled()
	{
		return $this->rewrite_enabled;
	}
	
	/*
	*	Enables URL rewrite
	*	@function	enableRewrite
	*	@return	void
	*/
	function enableRewrite()
	{
		$this->rewrite_enabled = true;
	}
	
	/*
	*	Disables URL rewrite
	*	@function	disableRewrite
	*	@return	void
	*/
	function disableRewrite()
	{
		$this->rewrite_enabled = false;
	}
	
	/*
	*	Get Htaccess content; loads default htaccess if empty
	*	@function	getHtaccess
	*	@return	string	content of the htaccess
	*/
	function getHtaccess()
	{
		if(empty($this->htaccess))
		{
			if($this->loadDefaultHtaccess())
			{
				return $this->htaccess;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return $this->htaccess;
		}
	}
	
	/*
	*	Writes Htaccess in script root path
	*	@function	writeHtaccess
	*	@return	boolean	TRUE if success
	*/
	function writeHtaccess()
	{
		return files::putContent(SCRIPT_ROOT_PATH.'/.htaccess',$this->htaccess);
	}
	
	/*
	*	Clears Htaccess file
	*	@function	clearHtaccess
	*	@return	boolean	TRUE if success
	*/
	function clearHtaccess()
	{
		return files::putContent(SCRIPT_ROOT_PATH.'/.htaccess',"\n");
	}
	
	/*
	*	Loads default Htaccess file
	*	@function	loadDefaultHtaccess
	*	@return	boolean	TRUE if success
	*/
	function loadDefaultHtaccess()
	{
		if (file_exists($this->default_file) && $fh = fopen( $this->default_file, 'r' ))
		{
			$lines = file($this->default_file);
			$this->htaccess = implode('', $lines);
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/*
	*	Encodes a string to make it usable for building up a rewrited URL
	*	@param string
	*	@return string : special characters cleaned up
	*/
	function code_url($string)
	{
		$words = explode(' ',$string);
		$min_length = 3;
		$cleaned_words = array();
		foreach($words as $word)
		{
			if(strlen($word) >= $min_length)
			{
				$word = strtolower($word);
				$word = str_replace(',','', $word);
				$word = str_replace('\'','', $word);
				$word = str_replace(':','', $word);
				$word = str_replace(';','', $word);
				$word = str_replace("( |')", "-", $word);
				$accent = array('','','','','','','','','',"'");
				$sans_accent = array('a','a','e','e','e','i','o','u','c','');
				$word = str_replace($accent, $sans_accent, $word);
				$word = eregi_replace("[^a-z0-9]","-",$word);
				$word = eregi_replace("(^(_)*|(_)*$)","",$word);
				$word = eregi_replace("(-){2,3}","",$word);
				$word = eregi_replace("([0-9])*","",$word);
				
				$cleaned_words[] = $word;
			}
		}
		$cleaned_string = implode('-',$cleaned_words);
		
		return $cleaned_string;
	}
	
	/*
	*	Builds up your URLs with or without rewriting, it depends on rewrite enabled or not
	*	@function	buildUrl
	*	@param	string type		Indicates which rule to apply to build the url
	*	@param	array p	Associative array containing parameters needed to build up the url
	*	@return	string		Generated URL with or without URL rewrite
	*/
	function buildUrl($type = '', $p)
	{
		if($this->isRewriteEnabled())
		{
			return $this->buildRewrittenUrl($type, $p);
		}
		else
		{
			return $this->buildStandardUrl($type, $p);
		}
	}
	
	/*
	*	this method is private. It builds up non rewritten URLs.
	*	@function	buildStandardUrl
	*	@param	string type		indicates the rule to apply to build the URL
	*	@param	array p		parameters stored in associative array
	*	@return	string url		Standard url
	*/
	function buildStandardUrl($type, $p)
	{
		$url = '';
		
		switch($type)
		{
			case 'category':
				$url = 'index.php?dir='.$p['id'];
				break;
			
			case 'category page':
				$url = 'index.php?dir='.$p['id'].'&amp;start='.$p['page'];
				break;
			
			case 'error301':
				// No error301 should be raised (at the moment) when URL rewrite is off
				break;

			case 'boost':
				$url = 'index.php?do='.$p['name'];
				break;
				
			case 'main link':
				$url = 'index.php?do='.$p['name'];
				break;
			
			case 'single':
				$url = 'single.php?id='.$p['id'];
				break;
			
			case 'top-rank':
				$url = "index.php?do=top-rank&amp;pr=".$p['pr'];
				break;
			
			case 'feed':
				$url = 'feed.php?feedid='.$p['feedid'].'&amp;linkid='.$p['linkid'];
				break;
				
			case 'tag':
				$url = 'tag.php?id='.$p['id'];
				break;

			case 'keyword':
				$url = 'index.php?do=keywords&amp;id='.$p['id'].'&amp;words='.$this->code_url($p['word']);
				break;
			
			case 'search':
				$page = (isset($p['start']) && $p['start'] > 0) ? '&amp;start='.$p['start'] : '';
				$url = 'index.php?do=recherche&amp;seek='.$p['seek'].$page;
				break;
		}
		
		return $url;
	}

	/*
	*	this method is private. It builds up rewritten URLs.
	*	@function	buildStandardUrl
	*	@param	string type		indicates the rule to apply to build the URL
	*	@param	array p		parameters stored in associative array
	*	@return	string url		rewritten url
	*/
	function buildRewrittenUrl($type, $p)
	{
		$url = '';
		
		switch($type)
		{
			case 'category':
				$url = $this->code_url($p['name']).'-c-'.$p['id'].'.html';
				break;
			
			case 'category page':
				$url = $this->code_url($p['name']).'-c-'.$p['id'].'-p-'.$p['page'].'.html';
				break;
			
			case 'error301':
				$url = $p['site_url'].'/'.$p['name'].'-'.$p['id'].'.html';
				break;
			
			case 'boost':
				$url = $p['name'].'.html';
				break;
				
			case 'main link':
				$url = $p['name'].'.html';
				break;
				
			case 'single':
				$url = $this->code_url($p['name']).'-s-'.$p['id'].'.html';
				break;
				
			case 'top-rank':
				$url = "top-rank".$p['pr'].".html";
				break;
				
			case 'feed':
				$url = $this->code_url($p['name']).'-f-'.$p['feedid'].'-'.$p['linkid'].'.html';
				break;
			
			case 'tag':
				$url = $this->code_url($p['tag']).'-t-'.$p['id'].'.html';
				break;
			
			case 'keyword':
				$url = $this->code_url($p['word']).'-k-'.$p['id'].'.html';
				break;
				
			case 'search':
				$page = (isset($p['start']) && $p['start'] > 0) ? '-p-'.$p['start'] : '';
				$url = code_url($p['seek']).$page;
				break;
		}
		
		return $url;
	}
}

/*
*	Singleton class : just unique instance of RewriteEngine needed
*	@class	MyrewriteEngine
*	@package	Kernel
*	@author	Jerome Loisel
*/
class MyRewriteEngine
{
	/*
	*	Retrieve a static reference to an instance of RewriteEngine
	*	@function	&getInstance
	*	@return	Object {@link RewriteEngine}
	*/
	function &getInstance()
	{
		static $re;
		global $CONFIG;
		
		if(!isset($re))
		{
			$rewrite_enabled = $CONFIG['url_rewriting'] == 1 ? true : false;
			$default_htaccess = SCRIPT_ROOT_PATH.'/conf/default.htaccess';
			
			$re = new RewriteEngine($rewrite_enabled,$default_htaccess);
		}
		return $re;
	}
}

?>