<?php

/**
* Object Data Types
**/
define('XOBJ_DTYPE_TXTBOX', 1);
define('XOBJ_DTYPE_TXTAREA', 2);
define('XOBJ_DTYPE_INT', 3);
define('XOBJ_DTYPE_URL', 4);
define('XOBJ_DTYPE_EMAIL', 5);
define('XOBJ_DTYPE_ARRAY', 6);
define('XOBJ_DTYPE_OTHER', 7);
define('XOBJ_DTYPE_SOURCE', 8);
define('XOBJ_DTYPE_STIME', 9);
define('XOBJ_DTYPE_MTIME', 10);
define('XOBJ_DTYPE_LTIME', 11);

//define('OBJECT_DEBUG','1');
define('DEBUG_QUERY_OK_IMG','check_on.png');
define('DEBUG_QUERY_KO_IMG','check_off.png');

/**
* Error Messages
**/
define("_ER_OB_ISREQUIRED", "%s is required.");
define("_ER_OB_MUSTBESHORTER", "%s must be less than %u caracters.");
define("_ER_OB_INVALIDEMAIL", "Invalid Email");

/**
 * 	Base class for all objects
 *	@class	MyObject
 *	@author	Jerome Loisel
 *	@copyright	Jerome Loisel
 *	@package	Kernel
 **/
class MyObject
{
	/**
	 * holds all variables(properties) of an object
	 * 
	 * @var array
	 * @access protected
	 **/
	var $attributes = array();
	
	/**
	* variables cleaned for store in DB
	* 
	* @var array
	* @access protected
	*/
	var $cleanVars = array();
	
	/**
	* is it a newly created object?
	* 
	* @var bool
	* @access private
	*/
	var $isnew = false;
	
	/**
	* has any of the values been modified?
	* 
	* @var bool
	* @access private
	*/
	var $_isDirty = false;
	
	/**
	* errors
	* 
	* @var array
	* @access private
	*/
	var $_errors = array();
	
	/**
	* table containing the object, useful for loading it
	* 
	* @var string
	* @access private
	*/
	var $table;
	
	/**
	* instance of the dabase connection
	* 
	* @var object {@Connection}
	* @access private
	*/
	var $con;
	
	
	
	/**
	 * Constructor
	 * 
	 * @param string $table : table containing or for storing the object
	 * @param associative array : attribute => value
	 * @return void
	 */
	function MyObject($table, $vars = null)
	{
		$this->con =& DatabaseFactory::getConnection();
		$this->setTable($this->con->prefix($table));
		
		if(!empty($vars))
		{
			if(is_array($vars))
			{
				$this->assignVars($vars);
			}
			else
			{
				$id = intval($vars);
				$this->load($id);
				$this->unsetNew();
			}
		}
		else
		{
			$this->setNew();
		}
	}

	/**
	 * Checks if the object is new
	 * 
	 * @return bool isNew
	 */
	function isNew()
	{
		return $this->isnew;
	}
	

	/**
	 * add an error 
	 * 
	 * @param string $value error to add
	 * @access public
	 */
    function setErrors($err_str)
    {
        $this->_errors[] = trim($err_str);
    }

	/**
	 * return the errors for this object as an array
	 * 
	 * @return array an array of errors
	 * @access public
	 */
    function getErrors()
    {
        return $this->_errors;
    }	

	
	/**
	 * Sets object attributes
	 * 
	 * @param associative array : attribute => value
	 * @return bool : true if success
	 */
	function setAttributes($attributes)
	{
		if(is_array($attributes))
		{
			$this->attributes = $attributes;
			return true;
		}
		else
		{
			return false;
		}
	}
	
	/**
	 * Add an object attribute
	 * 
	 * @param associative array : attribute => value
	 * @return bool : true if success
	 */
	function addAttribute($attribute)
	{
		if(is_array($attribute))
		{
			$this->attributes[] = $attribute;
			return true;
		}
		else
		{
			return false;
		}
	}
	
	
	/**
	 * assign all object attribute with an associative array of values ( attribute => value)
	 * 
	 * @param associative array : attribute => value
	 * @return bool
	 */
	function assignVars($array)
	{
		if(!empty($array))
		{
			if(count($array) <= count($this->attributes))
			{
				foreach($array as $key => $value)
				{
					$this->assignVar($key,$value);
				}
				return true;
			}
			else
			{
				return false;
			}
		}
	}
	
	
	/**
	 * Assign a single object attribute
	 * 
	 * @param string $key : name of the attribute
	 * @param mixed $value : new value of the attribute
	 * @return void
	 */
	function assignVar($key,$value)
	{
		$this->attributes[$key]['value'] = $value;
		$this->setDirty();
	}
	
	
	/**
	 * Same as assignVar()
	 * 
	 * @param string $key : name of the attribute
	 * @param mixed $value : new value of the attribute
	 * @return void
	 */
	function setVar($key, $value)
	{
		$this->assignVar($key,$value);
	}
	
	function setVars($array = array())
	{
		if(count($array) > 0)
		{
			foreach($array as $key => $value)
			{
				$this->assignVar($key,$value);
			}
			return true;
		}
		return false;
	}
	
	/**
	 * Returns the specified object attribute
	 * 
	 * @param string $key : name of the attribute
	 * @return mixed attribute if success, else false
	 */
	function getVar($key)
	{
		if(isset($this->attributes[$key]))
		{
			return $this->attributes[$key]['value'];
		}
		else
		{
			return false;
		}
	}
	
	
	/**
	 * returns all the attributes of the object and their properties
	 * 
	 * @return mixed array
	 */
	function getVars()
	{
		return $this->attributes;
	}
	
	
	/**
	 * Set an object as already existing
	 * 
	 * @return void
	 */
	function unsetNew()
	{
		$this->isnew = false;
	}
	
	
	/**
	 * Sets an object as new
	 * 
	 * @return void
	 */
	function setNew()
	{
		$this->isnew = true;
	}
	
	
	/**
	 * Sets Object Database table
	 * 
	 * @param string $table : name of the table where the object will be stored
	 * @return void
	 */
	function setTable($table)
	{
		$this->table = $table;
	}
	
	
	/**
	 * returns the table which is related to this object
	 * 
	 * @return string $table
	 */
	function getTable()
	{
		return $this->table;
	}
	
	
	/**
	 * Loads a single object from the database, from object's table
	 * 
	 * @param object {@criteria}
	 * @return object {@MyObject}
	 */
	function load($criteria = null)
	{
		if(isset($criteria) && is_subclass_of($criteria, 'criteriaelement'))
		{
			$sql = "SELECT * FROM `".$this->table."` ".$criteria->renderWhere()." LIMIT 1";
			$result = $this->con->select($sql);
			$array = $result->getData();
			if(is_array($array))
			{
				$this->assignVars($array[0]);
				return true;
			}
			else
			{
				return false;
			}
		}
		else
		{
			return false;
		}
	}
	

	/**
	 * Cleans objects attributes and cleans them
	 * 
	 * @return bool : true if success
	 */
	function cleanVars()
    {
        $ts =& MyTextSanitizer::getInstance();
        foreach ($this->attributes as $k => $v) {
            $cleanv = $v['value'];
			switch ($v['data_type'])
			{
				case XOBJ_DTYPE_TXTBOX:
				if (isset($v['maxlength']) && $v['maxlength'] > 0 && strlen($cleanv) > intval($v['maxlength'])) {
					$this->setErrors(sprintf(_ER_OB_MUSTBESHORTER, $k, intval($v['maxlength'])));
					continue;
				}
				$cleanv = $ts->stripSlashesGPC($cleanv);
				break;
				
				case XOBJ_DTYPE_TXTAREA:
				if (isset($v['maxlength']) && $v['maxlength'] > 0 && strlen($cleanv) > intval($v['maxlength'])) {
					$this->setErrors(sprintf(_ER_OB_MUSTBESHORTER, $k, intval($v['maxlength'])));
					continue;
				}
				$cleanv = $ts->stripSlashesGPC($cleanv);
				break;
				
				case XOBJ_DTYPE_INT:
				$cleanv = intval($cleanv);
				break;
				
				case XOBJ_DTYPE_EMAIL:

				if ($cleanv != '' && !preg_match("/^[_a-z0-9-]+(\.[_a-z0-9-]+)*@[a-z0-9-]+([\.][a-z0-9-]+)+$/i",$cleanv)) {
					$this->setErrors(_ER_OB_INVALIDEMAIL);
					continue;
				}
				$cleanv = $ts->stripSlashesGPC($cleanv);

				break;
				case XOBJ_DTYPE_URL:
				if ($cleanv != '' && !preg_match("/^http[s]*:\/\//i", $cleanv)) {
					$cleanv = 'http://' . $cleanv;
				}
				$cleanv = $ts->stripSlashesGPC($cleanv);
				break;

				case XOBJ_DTYPE_ARRAY:
				$cleanv = array_map(array($ts, 'stripSlashesGPC'), $cleanv);
				$cleanv = is_array($cleanv) ? serialize($cleanv) : serialize(array());
				break;
				
				case XOBJ_DTYPE_STIME:
				case XOBJ_DTYPE_MTIME:
				case XOBJ_DTYPE_LTIME:
				$cleanv = !is_string($cleanv) ? intval($cleanv) : strtotime($cleanv);
				break;
				
				default:
				break;
			}
			//echo $cleanv."<br />";
            $this->cleanVars[$k] =& $cleanv;
            unset($cleanv);
        }
        if (count($this->_errors) > 0) {
            return false;
        }
        $this->unsetDirty();
        return true;
    }
	
	/**
	 * Marks an object as dirty if it's attributes hasn't been cleaned
	 * 
	 * @return void
	 */
	function setDirty()
    {
        $this->_isDirty = true;
    }
	
	/**
	 * Sets an object as clean
	 * 
	 * @return void
	 */
    function unsetDirty()
    {
        $this->_isDirty = false;
    }
	
	/**
	 * Checks if an object is dirty
	 * 
	 * @return bool isDirty
	 */
    function isDirty()
    {
        return $this->_isDirty;
    }
}


/*
*	Base class for all object managers
* 	@class	MyObjectManager
*	@package	Kernel
*	@author	Jerome Loisel
*	@copyright	Jerome Loisel
*/
class MyObjectManager
{
	/**
	* table containing the handled objects
	* 
	* @var string
	* @access private
	*/
	var $table;

	/**
	* key name of the table
	* 
	* @var string
	* @access private
	*/
	var $keyname;

	/**
	* class name of the objects created
	* 
	* @var string
	* @access private
	*/
	var $classname;
	
	/**
	* instance of the dabase connection
	* 
	* @var object {@Connection}
	* @access private
	*/
	var $con;
	
	
	function displayDebugQuery($query,$result)
	{
		if(defined("OBJECT_DEBUG"))
		{
			$smarty =& TemplateEngine::getInstance();
			$image = $result ? DEBUG_QUERY_OK_IMG : DEBUG_QUERY_KO_IMG;
			$smarty->append('queries',array('sql' => $query, 'image' => $image));
		}
	}
	
	/**
	 * Object Manager Constructor
	 * 
	 * @param array specs : this array contains table, key and class name of the manipulated Object
	 * 
	 * @return void
	 */
	function MyObjectManager($specs)
	{
		$this->con =& DatabaseFactory::getConnection();
		$this->setTable($this->con->prefix($specs['table']));
		$this->setKey($specs['keyName']);
		$this->setClass($specs['className']);
	}
	
	
	/**
	 * On demand Object creation
	 * 
	 * @return object {@MyObject}
	 */
	function create($isNew = true)
	{
		$obj =& new $this->classname();
        if ($isNew == true)
		{
            $obj->setNew();
        }
		else
		{
			$obj->unsetNew();
		}
        return $obj;
	}
	
	
	/**
	 * method for setting the table
	 * 
	 * @param string $table : name of the table containing the objects
	 * 
	 * @return void
	 */
	function setTable($table)
	{
		$this->table = $table;
	}
	
	
	/**
	 * method for retrieving the table
	 * 
	 * @return string $table
	 */
	function getTable()
	{
		return $this->table;
	}
	
	
	/**
	 * method for setting the table's keyname
	 * 
	 * @param string $keyname : key name of the table containing the objects
	 * 
	 * @return void
	 */
	function setKey($keyname)
	{
		$this->keyname = $keyname;
	}
	
	
	/**
	 * method for retrieving the table's key name
	 * 
	 * @return string keyname
	 */
	function getKey()
	{
		return $this->keyname;
	}
	
	
	/**
	 * method for setting the object class
	 * 
	 * @param string $table : name of the table containing the objects
	 * 
	 * @return void
	 */
	function setClass($classname)
	{
		$this->classname = $classname;
	}
	
	/**
	 * method for retrieving the object class
	 * 
	 * @return void
	 */
	function getClass()
	{
		return $this->classname;
	}
	
	
	/**
	 * Deletes objects from a table considering a given criteria
	 * 
	 * @param object {@criteria}
	 * 
	 * @return bool : true if success
	 */
	function deleteAll($criteria = null)
	{
		$sql = "DELETE FROM `".$this->table."`";
		if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement')) {
			$sql .= ' '.$criteria->renderWhere();
		}
		$result = $this->con->execute($sql);
		$this->displayDebugQuery($sql,$result);
		if (!$result)
		{
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns an array of objects meeting the specific criteria
	 * 
	 * @param object {@criteria}
	 * @param array cols : columns to be selected in the SQL query
	 * 
	 * @return array of Object {@MyObject}
	 */
	function &getObjects($criteria = null, $cols = array())
	{
		$ret = array();
		$limit = $start = 0;
		
		$cols = count($cols) == 0 ? "*" : "`".implode("`,`",$cols)."`";
		
		$sql = "SELECT ".$cols." FROM `".$this->table."`";
		if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement'))
		{
			$sql .= ' '.$criteria->renderWhere();
			
			$sort = $criteria->getSort();
						
			if ($criteria->getSort() != '')
			{
				if(is_array($sort))
				{
					$sql .= ' ORDER BY ';
					$count = count($sort);
					$i = 0;
					foreach($sort as $column => $order)
					{
						$sql .= "`".$column."` ".$order;
						$sql = $i < $count-1 ? $sql.", " : $sql;
						$i++;
					}
				}
				else
				{
					$sql .= ' ORDER BY `'.$sort.'` '.$criteria->getOrder();
				}
			}
			$limit = $criteria->getLimit();
			$start = $criteria->getStart();
			if( ($limit && $start) != 0)
			{
				$sql .= " LIMIT ".$start.",".$limit;
			}
			else if($limit != 0)
			{
				$sql .= " LIMIT ".$limit;
			}
		}
		$result = $this->con->select($sql);
		$this->displayDebugQuery($sql,$result);
		if ($result)
		{
			$array = $result->getData();

			foreach($array as $key => $objectattr)
			{
				$object = new $this->classname();
				$object->unsetNew();
				$object->assignVars($objectattr);
				$ret[] = $object;
				unset($object);
			}
		}
		return $ret;
	}
	

	/**
	 * Returns a single object meeting the specific criteria
	 * 
	 * @param mixed id : either a string or an array
	 * @param array cols : columns to be selected in the SQL query
	 * 
	 * @return object {@MyObject}
	 */
	function &get($id, $cols = array())
	{
        if (is_array($this->keyname))
		{
            $criteria = new CriteriaCompo();
            for ($i = 0; $i < count($this->keyname); $i++)
			{
                $criteria->add(new Criteria($this->keyname[$i], intval($id[$i])));
            }
        }
        else
		{
            $criteria = new Criteria($this->keyname, $id,'=');
        }
        $criteria->setLimit(1);
        $obj_array = $this->getObjects($criteria, $cols);
        if (count($obj_array) != 1)
		{
			$object = $this->create(true);
            return $object;
        }
        return $obj_array[0];
    }
	
	/**
	 * Returns object count meeting a specific criteria
	 * 
	 * @param object $criteria {@criteria}
	 * 
	 * @return object {@MyObject}
	 */
	function getCount($criteria = null)
    {
        $sql = 'SELECT COUNT(*) AS `count` FROM `'.$this->table.'`';
        if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement'))
		{
            $sql .= ' '.$criteria->renderWhere();
			$limit = $criteria->getLimit();
			if($limit != 0)
			{
				$sql .= ' LIMIT '.$limit;
			}
        }
        $result = $this->con->select($sql);
		$this->displayDebugQuery($sql,$result);
        if (!$result)
		{
            return 0;
        }
		else
		{
			$data = $result->getData();
			return $data[0]['count'];
		}
    }
	
	/**
	 * insert a new object in the database
	 * 
	 * @param object $obj reference to the object
	 * @param bool $force whether to force the query execution despite security settings
	 * @param bool $checkObject check if the object is dirty and clean the attributes
	 * @return bool FALSE if failed, TRUE if already present and unchanged or successful
	 */
    function insert(&$obj, $checkObject = true)
    {
		if ($checkObject != false)
		{
            if (!is_object($obj))
			{
                var_dump($obj);
                return false;
            }
            if (!is_a($obj, $this->classname))
			{
                $obj->setErrors(get_class($obj)." Differs from ".$this->className);
                return false;
            }
            if (!$obj->isDirty())
			{
                $obj->setErrors("Not dirty"); // The object has not been modified ; no update or insert needed
                return true;
            }
        }
        if (!$obj->cleanVars())
		{
		$obj->setErrors("Cannot clean object attributes");
            return false;
        }
		
        foreach ($obj->cleanVars as $k => $v) {
            if ($obj->attributes[$k]['data_type'] == XOBJ_DTYPE_INT) {
                $cleanvars[$k] = intval($v);
            } elseif ( is_array( $v ) ) {
            	$cleanvars[ $k ] = $this->con->quoteString( implode( ',', $v ) );
            } else {
                $cleanvars[$k] = $this->con->quoteString($v);
            }
        }
        if ($obj->isNew())
		{
            if (!is_array($this->keyname))
			{
                if ($cleanvars[$this->keyname] < 1 && is_integer($cleanvars[$this->keyname]))
				{
                    $cleanvars[$this->keyname] = 0;
                }
            }
            $sql = "INSERT INTO `".$this->table."` (".implode(',', array_keys($cleanvars)).") VALUES (".implode(',', array_values($cleanvars)) .")";
        }
		else
		{
            $sql = "UPDATE `".$this->table."` SET";
            foreach ($cleanvars as $key => $value)
			{
                if ((!is_array($this->keyname) && $key == $this->keyname) || (is_array($this->keyname) && in_array($key, $this->keyname)))
				{
                    continue;
                }
                if (isset($notfirst) )
				{
                    $sql .= ",";
                }
                $sql .= " `".$key."` = ".$value."";
                $notfirst = true;
            }
            if (is_array($this->keyname))
			{
                $whereclause = "";
                for ($i = 0; $i < count($this->keyname); $i++)
				{
                    if ($i > 0)
					{
                        $whereclause .= " AND ";
                    }
                    $whereclause .= $this->keyName[$i]." = ".$obj->getVar($this->keyname[$i]);
                }
            }
            else
			{
                $whereclause = $this->keyname." = ".$obj->getVar($this->keyname);
            }
            $sql .= " WHERE ".$whereclause;
        }
		
		$result = $this->con->execute($sql);
		$this->displayDebugQuery($sql,$result);
        if (!$result)
		{
            return false;
        }
        if ($obj->isNew() && !is_array($this->keyname))
		{
            $obj->assignVar($this->keyname, $this->con->getLastID());
        }
        return true;
    }

	
	/**
	 * update objects in a table with specific criteria and attribute values
	 * 
	 * @param object $criteria {@criteria}
	 * @param associative array $attributes : name of the attribute => new value after update
	 * @return bool FALSE if failed, TRUE if successful
	 */
	function update($criteria, $attributes = array())
	{
		$count = count($attributes);
		if (isset($criteria) && is_subclass_of($criteria, 'criteriaelement') && is_array($attributes) && $count > 0)
		{
			$sql = "UPDATE `".$this->table."` SET ";
			$i = 1;
			foreach($attributes as $attribute => $value)
			{
				$sql = $i < $count ? $sql."`".$attribute."`='".$value."', " : $sql."`".$attribute."`='".$value."'" ;
				$i++;
			}
			$sql .= " ".$criteria->renderWhere();
			
			$limit = $criteria->getLimit() == 0 ? 1 : $criteria->getLimit();
			$sql .= " LIMIT ".$limit;
			$result = $this->con->execute($sql);
			$this->displayDebugQuery($sql,$result);
			
			return $result;
		}
		return false;
	}
}

?>