<?php
# ***** BEGIN LICENSE BLOCK *****
# This file is part of DotClear.
#
# DotClear is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# DotClear is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with DotClear; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# ***** END LICENSE BLOCK *****
/**
 *=======================================================================
 * Name:
 *	tar Class
 *
 * Author:
 *	Josh Barger <joshb@npt.com>
 *
 * Description:
 *	This class reads and writes Tape-Archive (TAR) Files and Gzip
 *	compressed TAR files, which are mainly used on UNIX systems.
 *	This class works on both windows AND unix systems, and does
 *	NOT rely on external applications!! Woohoo!
 *
 * Usage:
 *	Copyright (C) 2002  Josh Barger
 *
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Lesser General Public License for more details at:
 *		http://www.gnu.org/copyleft/lesser.html
 *
 *	If you use this script in your application/website, please
 *	send me an e-mail letting me know about it :)
 *
 * Bugs:
 *	Please report any bugs you might find to my e-mail address
 *	at joshb@npt.com.  If you have already created a fix/patch
 *	for the bug, please do send it to me so I can incorporate it into my release.
 *
 * Version History:
 *	1.0	04/10/2002	- InitialRelease
 *
 *	2.0	04/11/2002	- Merged both tarReader and tarWriter
 *				  classes into one
 *				- Added support for gzipped tar files
 *				  Remember to name for .tar.gz or .tgz
 *				  if you use gzip compression!
 *				  :: THIS REQUIRES ZLIB EXTENSION ::
 *				- Added additional comments to
 *				  functions to help users
 *				- Added ability to remove files and
 *				  directories from archive
 *	2.1	04/12/2002	- Fixed serious bug in generating tar
 *				- Created another example file
 *				- Added check to make sure ZLIB is
 *				  installed before running GZIP
 *				  compression on TAR
 *	2.2	05/07/2002	- Added automatic detection of Gzipped
 *				  tar files (Thanks go to Jrgen Falch
 *				  for the idea)
 *				- Changed "private" functions to have
 *				  special function names beginning with
 *				  two underscores
 *=======================================================================
 */
class tar {
	// Processed Archive Information
	var $tar_file;
	var $files;
	var $directories;
	var $numFiles;
	var $numDirectories;


	/**
	 * Class Constructor -- Does nothing...
	 */
	function tar() {
		return true;
	}

	/**
	 * Computes the unsigned Checksum of a file's header
	 * to try to ensure valid file
	 * PRIVATE ACCESS FUNCTION
	 */
	function __computeUnsignedChecksum($bytestring) {
		$unsigned_chksum = '';
		for($i=0; $i<512; $i++)
			$unsigned_chksum += ord($bytestring[$i]);
		for($i=0; $i<8; $i++)
			$unsigned_chksum -= ord($bytestring[148 + $i]);
		$unsigned_chksum += ord(" ") * 8;

		return $unsigned_chksum;
	}

	/**
	 * Generates a TAR file from the processed data
	 * PRIVATE ACCESS FUNCTION
	 */
	function __generateTAR() {
		// Generate Records for each directory, if we have directories
		if($this->numDirectories > 0) {
			foreach($this->directories as $key => $information) {
				unset($header);

				// Generate tar header for this directory
				// Filename, Permissions, UID, GID, size, Time, checksum, typeflag, linkname, magic, version, user name, group name, devmajor, devminor, prefix, end
				$header = str_pad($information["name"],100,chr(0));
				$header .= str_pad(decoct($information["mode"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["user_id"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["group_id"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct(0),11,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["time"]),11,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_repeat(" ",8);
				$header .= "5";
				$header .= str_repeat(chr(0),100);
				$header .= str_pad("ustar",6,chr(32));
				$header .= chr(32) . chr(0);
				$header .= str_pad("",32,chr(0));
				$header .= str_pad("",32,chr(0));
				$header .= str_repeat(chr(0),8);
				$header .= str_repeat(chr(0),8);
				$header .= str_repeat(chr(0),155);
				$header .= str_repeat(chr(0),12);

				// Compute header checksum
				$checksum = str_pad(decoct($this->__computeUnsignedChecksum($header)),6,"0",STR_PAD_LEFT);
				for($i=0; $i<6; $i++) {
					$header[(148 + $i)] = substr($checksum,$i,1);
				}
				$header[154] = chr(0);
				$header[155] = chr(32);

				// Add new tar formatted data to tar file contents
				$this->tar_file .= $header;
			}
		}

		// Generate Records for each file, if we have files (We should...)
		if($this->numFiles > 0) {
			foreach($this->files as $key => $information) {
				unset($header);

				// Generate the TAR header for this file
				// Filename, Permissions, UID, GID, size, Time, checksum, typeflag, linkname, magic, version, user name, group name, devmajor, devminor, prefix, end
				$header = str_pad($information["name"],100,chr(0));
				$header .= str_pad(decoct($information["mode"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["user_id"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["group_id"]),7,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["size"]),11,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_pad(decoct($information["time"]),11,"0",STR_PAD_LEFT) . chr(0);
				$header .= str_repeat(" ",8);
				$header .= "0";
				$header .= str_repeat(chr(0),100);
				$header .= str_pad("ustar",6,chr(32));
				$header .= chr(32) . chr(0);
				$header .= str_pad($information["user_name"],32,chr(0));	// How do I get a file's user name from PHP?
				$header .= str_pad($information["group_name"],32,chr(0));	// How do I get a file's group name from PHP?
				$header .= str_repeat(chr(0),8);
				$header .= str_repeat(chr(0),8);
				$header .= str_repeat(chr(0),155);
				$header .= str_repeat(chr(0),12);

				// Compute header checksum
				$checksum = str_pad(decoct($this->__computeUnsignedChecksum($header)),6,"0",STR_PAD_LEFT);
				for($i=0; $i<6; $i++) {
					$header[(148 + $i)] = substr($checksum,$i,1);
				}
				$header[154] = chr(0);
				$header[155] = chr(32);

				// Pad file contents to byte count divisible by 512
				$file_contents = str_pad($information["file"],(ceil($information["size"] / 512) * 512),chr(0));

				// Add new tar formatted data to tar file contents
				$this->tar_file .= $header . $file_contents;
			}
		}

		// Add 512 bytes of NULLs to designate EOF
		$this->tar_file .= str_repeat(chr(0),512);

		return true;
	}

	/**
	 * Check if this tar archive contains a specific file
	 */
	function containsFile($filename) {
		if($this->numFiles > 0) {
			foreach($this->files as $key => $information) {
				if($information["name"] == $filename)
					return true;
			}
		}

		return false;
	}

	/**
	 * Add a directory to this tar archive
	 *
	 */
	function addDirectory($dirname) {
		if(!file_exists($dirname))
			return false;

		// Get directory information
		$file_information = stat($dirname);

		// Add directory to processed data
		$this->numDirectories++;
		$activeDir		= &$this->directories[];
		$activeDir["name"]	= $dirname;
		$activeDir["mode"]	= $file_information["mode"];
		$activeDir["time"]	= $file_information["mtime"];
		$activeDir["user_id"]	= $file_information["uid"];
		$activeDir["group_id"]	= $file_information["gid"];
		$activeDir["checksum"]	= (isset($checksum))?$checksum:'';

		return true;
	}


	/**
	 * Add a file to the tar archive
	 *
	 */
	function addFile($filename) {
		// Make sure the file we are adding exists!
		if(!file_exists($filename))
			return false;

		// Make sure there are no other files in the archive that have this same filename
		if($this->containsFile($filename))
			return false;

		// Get file information
		$file_information = stat($filename);

		// Read in the file's contents
		$fp = @fopen($filename,"rb");
		$file_contents = @fread($fp,filesize($filename));

		fclose($fp);

		// Add file to processed data
		$this->numFiles++;
		$activeFile			= &$this->files[];
		$activeFile["name"]		= $filename;
		$activeFile["mode"]		= $file_information["mode"];
		$activeFile["user_id"]		= $file_information["uid"];
		$activeFile["group_id"]		= $file_information["gid"];
		$activeFile["size"]		= $file_information["size"];
		$activeFile["time"]		= $file_information["mtime"];
		$activeFile["checksum"]		= (isset($checksum))?$checksum:'';
		$activeFile["user_name"]	= "";
		$activeFile["group_name"]	= "";
		$activeFile["file"]		= $file_contents;

		return true;
	}

	/**
	 * Saves tar archive to a different file than the current file
	 *
	 */
	function toTar($filename,$useGzip) {
		if(!$filename)
			return false;

		// Encode processed files into TAR file format
		$this->__generateTar();

		// GZ Compress the data if we need to
		if($useGzip) {
			// Make sure we have gzip support
			if(!function_exists("gzencode"))
				return false;

			$file = gzencode($this->tar_file);
		} else {
			$file = $this->tar_file;
		}

		// Write the TAR file
		$fp = fopen($filename,"wb");
		fwrite($fp,$file);
		fclose($fp);

		return true;
	}
}
?>