<?php header("Content-type:text/html;charset=utf-8");	
/**
 * 通过thrift 连接hadoop的 HDFS
 * ----------------------------------------------------------------------------
 * The hadoophdfs.php file
 * ============================================================================
 * @package		Difeye
 * @author		Difeye Team  mathcn  
 * @copyright	(c) 2010-2013 Difeye Team
 * @license		http://www.difeye.com/license
 * ============================================================================
 */ 

class model_hadoophdfs  extends model_thrift
{
	private $hdfs_client = null;
	private $transport = null;

	/*
	 * create connection method
	 */
	public function connection()
	{

		$transport = new TBufferedTransport($this->socket);
		$protocol = new TBinaryProtocol($transport);

		$client = new ThriftHadoopFileSystemClient($protocol);
		$transport->open();
		$this->hdfs_client = $client;
		$this->transport = $transport;
		return $this->hdfs_client;
	}
		

	/*
	 * close connection method
	 * just close connection 
	 * return: if success return true, else return false
	 */
	public function close()
	{
		if ($this->hdfs_client != null)
		{
			$this->transport->close();
			$this->hdfs_client = null;
		}
	}

	/*
	 * rm remove directory or file method
	 * just remove directory or file on hadoop hdfs
	 * param $path the path where you want to remove from hdfs, $recursive if true try to recursive delete,else not
	 * return: if success return true, else return false
	 */
	public function rm($path, $recursive)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$dirpathname = new Pathname(array('pathname' => $path));
		if ($this->hdfs_client->exists($dirpathname))
		{
			$this->hdfs_client->rm($dirpathname, $recursive);
			return true;
		}
	}

	/*
	 * rename rename directory or file method
	 * just rename directory or file on hadoop hdfs
	 * param $path the path where you want to rename from hdfs, $dest where you want move to 
	 * return: if success return true, else return false
	 */
	public function rename($path, $dest)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$dest = str_replace("\\", '/', $dest);
		$path = new Pathname(array('pathname' => $path));
		$dest = new Pathname(array('pathname' => $dest));

		if ($this->hdfs_client->exists($path))
		{
			$this->hdfs_client->rename($path, $dest);
			return true;
		}
	}

	/*
	 * exists try to check directory or file if exists method
	 * just check directory or file on hadoop hdfs if exists
	 * param $path the path where you want to check on the hdfs
	 * return: if find return true, else return false
	 */
	public function exists($path)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$path = new Pathname(array('pathname' => $path));

		if ($this->hdfs_client->exists($path))
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	/*
	 * liststatus try to check directory information
	 * just check directory information on hadoop hdfs
	 * param $path the path where you want to check information on the hdfs
	 * return:array，something about list information 
	 */
	public function liststatus($path)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$path = new Pathname(array('pathname' => $path));
		if ($this->hdfs_client->exists($path))
			return $this->hdfs_client->listStatus($path);
		else
			return false;
	}

	/*
	 * chmod try to change file or directory permissions 
	 * just change file or directory permissions on hadoop hdfs
	 * param $path the path where you want to change permissions on the hdfs
	 */
	public function chmod($path, $mode)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$path = new Pathname(array('pathname' => $path));
		if ($this->hdfs_client->exists($path))
			return $this->hdfs_client->chmod($path, $mode);
	}

	/*
	 * chown try to change file or directory owner and group 
	 * just change file or directory owner and group on hadoop hdfs,like linux shell
	 * param $path the path where you want to change owner and group on the hdfs
	 */
	public function chown($path, $owner, $group)
	{
		if ($this->hdfs_client == null)
		{
			throw new Exception('the hdfs client object is null, please try to use connection method first');
			return false;
		}

		$path = str_replace("\\", '/', $path);
		$path = new Pathname(array('pathname' => $path));
		if ($this->hdfs_client->exists($path))
			return $this->hdfs_client->chown($path, $owner, $group);
	}


	/*
	 * createdir  method
	 * just create directory on hadoop hdfs,
	 * param for example array('/user/hadoop/test_dir1', '/user/hadoop/test_dir2')
	 * if the parameter is array，this method will try to create all of them if not exists 
	 * return: if success return true, else return false
	 */
	public function createdir($dirArr)
	{
		try
		{
			if ($this->hdfs_client == null)
			{
				throw new Exception('the hdfs client object is null, please try to use connection method first');
				return false;
			}

			if (!is_array($dirArr) && !trim($dirArr))
			{
				throw new Exception('parameter error');
				return false;
			}

			if (is_array($dirArr))
			{
				foreach ($dirArr as $dir)
				{
					$dirpathname = new Pathname(array('pathname' => $dir));
					if ($this->hdfs_client->exists($dirpathname) != true)
					{
						$this->hdfs_client->mkdirs($dirpathname);
						unset($dirpathname);
					}
				}
			}
			else
			{
				$dirpathname = new Pathname(array('pathname' => $dirArr));
				if ($this->hdfs_client->exists($dirpathname) != true)
				{
					$this->hdfs_client->mkdirs($dirpathname);
					unset($dirpathname);
				}
			}
		}
		catch (Exception $e)
		{
			echo $e->getMessage()."\n";
			throw new Exception('create directory error');
			return false;
		}

		return true;
	}

	/*
	 * uploadfile  method
	 * just upload file to hadoop hdfs,
	 * param for example 
	 * array(
	 	0 => array(
			 'src' => '/usr/local/test_data.txt', //source file path
			 'des' => '/user/hadoop/test_data.txt', //target file path
		 ),
		 1 => array(
			 'src' => '/usr/local/test_data2.txt', //source file path
			 'des' => '/user/hadoop/test_data2.txt', //target file path
		 ),
	 * )
	 * if the parameter is array，this method will try to upload all of them 
	 * return: if success return true, else return false
	 */
	public function uploadfile($fileinfo = array())
	{
		$ret = true;
		try
		{
			if ($this->hdfs_client == null)
			{
				throw new Exception('the hdfs client object is null, please try to use connection method first');
				return false;
			}

			if (!is_array($fileinfo) || count($fileinfo) < 1)
			{
				throw new Exception('the parameter must be array and not empty');
				return false;
			}

			//try to upload all files to hdfs
			foreach ($fileinfo as $file)
			{
				if (isset($file['src']) && isset($file['des']))
				{
					$src = realpath(str_replace("\\", '/', $file['src']));
					$des = str_replace("\\", '/', $file['des']);
					$hdfs_file = new Pathname(array('pathname' => $des));

					if (file_exists($src) && !$this->hdfs_client->exists($hdfs_file))
					{
						$local_file_handle = file_get_contents($src);
						
						//utf-8 二进制传输导致文件损坏；需修改 hadoop-1.0.4/src/contrib/thriftfs/src/java/org/apache/hadoop/thriftfs/HadoopThriftServer.java
						//里的wirte方法 接收后解密；
						$local_file_handle = base64_encode($local_file_handle);  
						

						$hdfs_handle = $this->hdfs_client->create($hdfs_file);
						$this->hdfs_client->write($hdfs_handle, $local_file_handle);
						
						$this->hdfs_client->close($hdfs_handle);


					}
				}
			}
		}
		catch (Exception $e)
		{
			echo $e->getMessage()."\n";
			throw new Exception('upload file error');
			return false;
		}

		return $ret;
	}

	/*
	 * downloadfile  method
	 * just download file from hadoop hdfs to local
	 * param for example 
	 * array(
	 	0 => array(
			 'src' => '/usr/local/test_data.txt', //file in the hdfs
			 'des' => '/user/hadoop/test_data.txt', //file in the local
		 ),
		 1 => array(
			 'src' => '/usr/local/test_data2.txt', //file in the hdfs
			 'des' => '/user/hadoop/test_data2.txt', //file in the local
		 ),
	 * )
	 * if the parameter is array，this method will try to download all of them 
	 * return: if success return true, else return false
	 */
	public function downloadfile($fileinfo = array())
	{
		$ret = true;
		try
		{
			if ($this->hdfs_client == null)
			{
				throw new Exception('the hdfs client object is null, please try to use connection method first');
				return false;
			}

			if (!is_array($fileinfo) || count($fileinfo) < 1)
			{
				throw new Exception('the parameter must be array and not empty');
				return false;
			}

			//try to download all files from hdfs
			foreach ($fileinfo as $file)
			{
				if (isset($file['src']) && isset($file['des']))
				{
					$src = str_replace("\\", '/', $file['src']);
					$des = str_replace("\\", '/', $file['des']);
					$hdfs_file = new Pathname(array('pathname' => $src));

					if ($this->hdfs_client->exists($hdfs_file))
					{
						$hdfs_handle = $this->hdfs_client->open($hdfs_file);
						if (file_exists($des))
						{
							unlink($des);
						}
						

						$local_file_handle = fopen($des, "a+");
						$data = "";
						$readcount = 0;
						while (true)
						{
							$tmp = $this->hdfs_client->read($hdfs_handle, $readcount * 1024*1024*10, 1024*1024*10);
							if (strlen($tmp) == 0)
							{
								break;
							}	
							//hadoop 服务端需先加密再传输；需修改 hadoop-1.0.4/src/contrib/thriftfs/src/java/org/apache/hadoop/thriftfs/HadoopThriftServer.java
							//里的read方法 接收后解密；
							$data = base64_decode($tmp);
							
							fwrite($local_file_handle,$data);

							if (strlen($data) < 1024*1024*10)
							{
								break;
							}
							$readcount++;
						}


						$this->hdfs_client->close($hdfs_handle);
					}
				}
			}
		}
		catch (Exception $e)
		{
			echo $e->getMessage()."\n";
			throw new Exception('download file error');
			return false;
		}

		return $ret;
	}


	/*
	 * encode to UTF-8  method, only support 'ASCII', 'GBK', 'UTF-8'
	 * param file context $string 
	 * return: if success return UTF-8 encode file string, else return false
	 */
	public function encode_UTF8($string){

		$code = array('ASCII', 'GBK', 'UTF-8');
		$encode= null;
		$rs = null;

		foreach($code as $c){
			if( $string === iconv('UTF-8', $c, iconv($c, 'UTF-8', $string))){
				$encode = $c; 
			}
		}


		if($encode== 'ASCII')
		{
			$rs = iconv('ASCII', 'UTF-8', $string);
		}elseif($encode== 'GBK'){
			$rs = iconv('GBK', 'UTF-8', $string);
		}elseif($encode== 'UTF-8'){
			$rs =  $string;
		}else{
			
			throw new Exception('upload file only support ASCII,GBK,UTF-8');
			return false;
			
		}

		return $rs;
	}





}
?>