<?php if ( ! defined('BASE_PATH')) { return; }
/**
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2010-2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* @interface IDatabase
* Define the methods derived classes should implement
*/
interface IDatabase
{
	/**
	* @public
	* @static
	* Retrieve the reference to the instance of this classs
	* @param array The list of parameters to use to connect to the database
	* Valid array keys:
	*	'server'   => 'db server',
	*	'user'     => 'user name',
	*	'password' => 'password',
	*	'database' => 'the database's name',
	*	'prefix'   => 'table prefix'
	* @return IDatabase object
	*/
	public static function getInstance( array $connectionArray );
	/**
	* @public
	* @throws Exception
	* Set up the connection to database
	* @return boolean true on success otherwise throws exception
	*/
	public function connect( array $dbArray = array() );
	/**
	* @public
	* Check to see if there is an open connection to the database server
	* @return boolean
	*/
	public function isConnected();
	/**
	* @public
	* @throws exception if the connection could not be closed
	* Close the current opened connection
	* @return boolean true on success
	*/
	public function disconnect();
	/**
	* @public
	* Retrieve the last generated error.
	* @return string
	*/
	public function getError();
	/**
	* @public
	* Retrieve the currrent/last executed query.
	* @return string
	*/
	public function getQuery();
	/**
	* @public
	* Retrieve the tables' prefix.
	* @return string
	*/
	public function getPrefix();
	/**
	* @public
	* Retrieve the connection object.
	* @return object
	*/
	public function getConnectionObject();
	/**
	* @public
	* Retrieve the connection array.
	* @return array
	*/
	public function getConnectionArray();
	/**
	* @public
    * Execute a sql query
	* @param string  The query to execute
	* param bool  Whether or not to return the result of the query
    * @return resource if $returnResult true, else: boolean
    */
    public function run( $query, $returnResult = false );
    /**
	* @public
    * Retrieve the specified number of records from the query
	* @param array  The list of fields to select
	* @param $table  The name of the table
	* @param string  The sql WHERE clause
	* @param array  The sql LIMIT clause. if provided, both values are required. ex: $limit = array(0,5)
    * @return array
    */
    public function get( array $fields, $table, $where = '', array $limit = array() );
    /**
	* @public
    * Retrieve all content of a table
	* @param string The name of the table
	* @param string The sql WHERE clause
    * @return array|false
    */
    public function getAll( $table, $where = '' );
    /**
	* @public
    * Retrieve the value of the selected column name
	* @param string The name of the column to retrieve from $table
	* @param string The name of the table
	* @param string The sql WHERE clause
    * @return mixed|false
    */
    public function getOne( $field, $table, $where = '' );
    /**
	* @public
    * Retrieve the content of an entire row from the database
	* @param string The name of the table
	* @param string The sql WHERE clause
    * @return array|false
    */
    public function getRow( $table, $where = '' );
    /**
	* @public
    * Retrieve all content of a selct query as an associative array
	* @param string The select query to execute
    * @return array|false
    */
    public function getAllFromQuery( $query );
    /**
	* @public
    * Retrieve the number of rows return by a mysql query
	* @param string The name of the table
	* @param string The sql WHERE clause
    * @return integer, -1 on error
    */
    public function count( $table, $where = '' );
    /**
	* @public
    * Retrieve the number of rows return by a mysql query
	* @param string The name of the table
	* @param string The sql WHERE clause
    * @return number, -1 on error
    */
    public function countFromQuery( $query );
	/**
	* @public
	* Execute a stored procedure against the database. Note that the Stored Procedure
	* should return an int (0 or 1).
	* @param string The stored procedure name
	* @param array The list of arguments to pass to the stored procedure
	* @param var The variable that will store the value returned by the stored procedure
	* @return int 0 on error, 1 on success
	*/
	public function executeSP( $spName, array $arguments, $outResult );
    /**
	* @public
    * Execute an INSERT INTO command
    * @return boolean
    */
    public function insertInto( $table, $insert_statement = array(/* db_column_name => column_value */) );
    /**
	* @public
    * Execute an UPDATE command
    * @return boolean
    */
    public function update( $table, $update_statement = array(/* db_column_name => column_value */), $where = '' );
    /**
	* @public
    * Execute a DELETE command
    * @return boolean
    */
    public function deleteFrom( $table, $where = '' );
	/**
	* @public
	* Escape a string using the htmlspecialchars function
	* @param string The string to be escaped
	* @return string
	*/
    public function escape( $string );
	/**
	* @public
	* Unescape a string using the htmlspecialchars_decode function
	* @param string The string to be unescaped
	* @return string
	*/
    public function unescape( $string );
	/**
	* @public
	* Start a transaction
	* @param boolean Set autocommit on or off. Defaults to off.
	* @return boolean
	*/
	public function startTransaction( $setAutocomit0 = true );
	/**
	* @public
	* Commit a transaction
	* @return boolean
	*/
	public function commitTransaction();
	/**
	* @public
	* Rollback a transaction
	* @return boolean
	*/
	public function rollbackTransaction();
}
/* End of file: IDatabase.php */