<?php if ( ! defined('BASE_PATH')) { exit( 'File: '.basename(__FILE__).' was not found!' ); }
/**
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2010-2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* @class IrisLang
* Provides common methds to handle multiple language files
*/
class IrisLang
{
	/**
	 * @private
	 * @desc Holds the path to the languages directory
	 * @see self::__construct()
	 * @type string
	 */
	private $_langsDir = '';
	/**
	 * @private
	 * @desc Holds the name of the default language file
	 * @see self::__construct()
	 * @type string
	 */
	private $_defaultLanguage = '';
	/**
	 * @private
	 * @desc Holds the name of the current active (loaded) language
	 * @see self::__construct()
	 * @var string
	 */
	private $_currentLoadedLang = '';
	/**
	 * @private
	 * @desc Holds the loaded language's files content
	 * @see self::load(), self::getData()
	 * @var array
	 */
	private $_lang = array();

	/**
	 * @public
	 * @desc Constructor
	 * @param string $defaultLanguageName The name of the default language
	 * @param string $currentLanguageName The name of the current language in use
	 * @param string $languagesDir The path to the languages directory
	 * @throws Exception if the specified languages directory is not found
	 * @return void
	 */
	public function __construct( $defaultLanguageName, $currentLanguageName, $languagesDir)
	{
        $this->_defaultLanguage = $defaultLanguageName;
        $this->_currentLoadedLang = $currentLanguageName;
        $this->_langsDir = $languagesDir;
        if (!is_dir($this->_langsDir)) {
        	throw new Exception('The specified languages directory was not found!');
        }
	}

	/**
	* @public
	* @desc Retrieve the content of the specified language file and assign it to the local variable
	* @param string The name of the language file to load
	* @return IrisLang object
	*/
	public function load( $lang )
	{
		$lang_file = $this->_langsDir . $lang . '.php';

		if (is_file($lang_file))
		{
			$this->_lang = include $lang_file;
			$this->_currentLoadedLang = $lang;
		}
		return $this;
	}

	/**
	 * @public
	 * @desc Retrieve the name of the current loaded language
	 * @return string
	 */
	public function getLoadedLanguage()
	{
	    return $this->_currentLoadedLang;
	}

	/**
	* @public
	* @desc Retrieve the reference to the loaded language array
	* @return array
	*/
	public function getData()
	{
		return $this->_lang;
	}

	/**
	* @public
	* @desc Retrieve the translation of a string identified by the provided $lang_array_key
	* @param string The array key to use to search for the string translation in the loaded language file
	* @return string
	*/
	public function translate( $lang_array_key )
	{
		if ( ! is_array($this->_lang)) { return ''; }
		return (isset($this->_lang[$lang_array_key])) ? trim($this->_lang[$lang_array_key]) : '';
	}

	/**
	* @public
	* @desc Display the translation of the provided language array keys. The last key must always be the items separator as it will be removed from the list!
	* @param string The list of language array keys to translate. The number is optional.
	* @param string The separator to use to separate each item from one another. This must be the last argument as it will be removed from the list!
	* This method uses the func_get_args function
	* @return string
	*/
	public function translateList( /* $lang_array_key1, $lang_array_key2,etc...,$separator */ )
	{
		$result = '';
		$args = func_get_args();
		$num = count($args);
		if (empty($args) || $num < 1) { return $result; }

		$separator = end($args);
		unset($args[$num-1]);

		$data = array_map(array($this,'translate'), $args);
		$result = implode($separator,$data);
		return $result;
	}

	/**
	* @public
	* @desc Retrieve the list of all available language files from the current loaded theme directory
	* @return array
	*/
	public function getLanguageFiles()
	{
		$files = Util::dirToArray($this->_langsDir,0,1);
		$data = array();
		if (empty($files))
		{
			return $data;
		}
		//[[ grab language files
		foreach($files as $file) {
			array_push($data,$file);
		}
		return $data;
	}

	/**
	* @public
	* @desc Add a new language file
	* @param string  The full path to the language file
	* @param bool    Whether to delete the file or not. Defaults to true.
	* @return boolean
	*/
	public function add( $language_file_path, $delete = true )
	{
		if ( ! is_file($language_file_path)) { return false; }

		$fname = basename($language_file_path);

		//[[ WHERE TO PUT THE LANG FILE
		$dir = $this->_langsDir;
		if ( ! is_dir($dir))
		{
			return false;
		}

		//[[ move/copy file into the found langs dir
		$dest_file_path = $dir.$fname;

		if (function_exists('copy'))
		{
			if ( $delete)
			{
				@unlink($language_file_path);
			}
			return copy($language_file_path, $dest_file_path);
		}
		else
		{
			if ( $delete )
			{
				@unlink($language_file_path);
			}
			return file_put_contents($dest_file_path, file_get_contents($language_file_path));
		}
	}

	/**
	* @public
	* @desc Add a new language file
	* @param string  The temporary file name from $_FILES array
	* @param string  The file name from $_FILES array
	* @param bool    Whether to delete the file or not. Defaults to true.
	* @return boolean
	*/
	public function addUploadedFile( $lang_file_tmp_name, $lang_file_name, $delete = true )
	{
		if ( ! is_file($lang_file_tmp_name)) { return false; }

		$fname = basename($lang_file_name);

		//[[ WHERE TO PUT THE LANG FILE
		$dir = $this->_langsDir;
		if ( ! is_dir($dir))
		{
			return false;
		}

		//[[ move/copy file into the found langs dir
		return move_uploaded_file($lang_file_tmp_name, $dir.$lang_file_name);
	}


	/**
	* @public
	* @desc Update a language file
	* @param string  The full path to the language file
	* @param string  The language file data. This content should be escaped using the htmlentities($data,ENT_QUOTES,'UTF-8'); function
	* @return boolean
	*/
	public function update( $language_file_path, $data )
	{
		if (! is_file($language_file_path)) { return false; }
		if (empty($data)) { return false; }

		$data = html_entity_decode($data,ENT_QUOTES,'UTF-8');

		return file_put_contents($language_file_path,$data,LOCK_EX);
	}

	/**
	* @public
	* @desc Delete a language file
	* @param string  The full path to the language file
	* @param string  The language file data. This content should be escaped using the htmlentities($data,ENT_QUOTES,'UTF-8'); function
	* @return boolean
	*/
	public function delete( $language_file_path )
	{
		//[[ The currently loaded language file cannot be deleted while is active
		$delete_file_name = basename($language_file_path,'.php');
		if (Util::stringsEqual($delete_file_name, $this->_currentLoadedLang)) { return false; }

		@chmod($language_file_path,0777);

		return @unlink($language_file_path);
	}
}
/*[ End of file: IrisLang.php ]*/