<?php  if ( ! defined('BASE_PATH')) { exit( 'File: '.basename(__FILE__).' was not found!' ); }
/**
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2010-2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* static class IrisSession
* Facilitates passing variables throughout the application using the $_SESSION as handler.
* @uses Util
*/
class IrisSession
{
    /**
     * @public
     * @static
     * @desc Holds the session's name. Defaults to PHPSESSID.
     * @type string
     */
    public static $name = 'PHPSESSID';
    /**
     * @public
     * @static
     * @desc Holds the session's length in minutes. Defaults to 0.
     * @type int
     */
    public static $lifetime = 0;
    /**
     * @public
     * @static
     * @desc Holds the path to the directory where to save the session files.
     * Defaults to the system path from the php.ini file.
     * @type string
     */
    public static $path = '';

    /**
     * @public
     * @constant
     * @desc Holds the error message to be displayed if the session was not started
     * @type string
     */
    const SESSION_NOT_STARTED_ERROR = 'The session was not started!';


    /**
     * @private
     * @desc Constructor
     */
    private function __construct(){}

    /**
     * @public
     * @static
     * @desc Check to see whether the session has been started or not
     * @return boolean
     */
    public static function isStarted()
    {
        $id = session_id();
        return (empty($id) ? false : true);
    }

    /**
     * @public
     * @static
     * @desc Start the session
     * @param string The name of the session. Optional. Defaults to PHPSESSID.
     * @param int    In minutes, the length of the session. Defaults to 0.
     * @param string The path to the directory where to save the session files. Optional.
     * Defaults to the system path from the php.ini file.
     * @return void
     */
    public static function start( $sessionName = '', $sessionLength = 0, $sessionSavePath = '' )
    {
        if ( ! empty($sessionName))     { self::$name = $sessionName; }
        if ((int)$sessionLength > 0)    { self::$lifetime = $sessionLength; }
        if ( ! empty($sessionSavePath)) { self::$path = $sessionSavePath; }

        if (empty(self::$path) || !is_dir(self::$path))
        {
            self::$path = ini_get('session.save_path');
        }

        $expire = self::$lifetime * 60;

        session_regenerate_id(true);
        session_name(self::$name);
        session_cache_limiter('nocache');
        session_cache_expire($expire);

		/*! No session in URL !*/
		ini_set("session.use_trans_sid", 0);
        ini_set("session.use_cookies", 1);

        session_save_path(self::$path);
        ini_set('session.save_path', self::$path);

        //[[ DELETE DEFAULT PHP COOKIE IF CURRENT SESSION NAME IS DIFFERENT
        if ( ! Util::StringsEqual(self::$name, 'PHPSESSID') && ! empty($_COOKIE['PHPSESSID'])) {
            setcookie('PHPSESSID','',time()-98989845,'/');
        }

        session_start();

        // set the session cookie
		ini_set("session.cookie_lifetime", $expire);
        setcookie(self::$name, session_id(), time()+$expire, '/');
    }

    /**
     * @public
     * @static
     * @desc Retrieve the value of the specified variable from the session
     * Exits the script with an error message if the session was not started
     * @param type string The name of the variable to retrieve from session
     * @return mixed
     */
    public static function get( $var )
    {
        if ( ! self::isStarted() ) {
            exit(self::SESSION_NOT_STARTED_ERROR);
        }

        return (isset($_SESSION[$var]) ? $_SESSION[$var] : '');
    }

    /**
     * @public
     * @static
     * @desc Add a variable into the session
     * Exits the script with an error message if the session was not started
     * @param type string The name of the variable to store in the session
     * @param type mixed The variable's value
     * @return void
     */
    public static function set( $name, $value )
    {
        if ( ! self::isStarted() ) {
            exit(self::SESSION_NOT_STARTED_ERROR);
        }

        $_SESSION[$name] = $value;
    }

    /**
     * @public
     * @static
     * @desc Retrieve the content of the active session
     * Exits the script with an error message if the session was not started
     * @return array
     */
    public static function getAll()
    {
        if ( ! self::isStarted() ) {
            exit(self::SESSION_NOT_STARTED_ERROR);
        }

        return $_SESSION;
    }
    /**
    * @public
    * @desc Clear all session variables or only the provided ones. This function does not destroys the session.
    * Exits the script with an error message if the session was not started
    * @param string The list of variables to remove from session. If omitted, all variables will be deleted.
    * @return void
    */
    public static function clear( /*[ $var1, $var2, etc...]*/)
    {
        if ( ! self::isStarted() ) {
            exit(self::SESSION_NOT_STARTED_ERROR);
        }

        $args = func_get_args();
        $numArgs = count($args);
        if ($numArgs > 0)
        {
            foreach($args as $arg) {
                if (isset($_SESSION[$arg])) {
                    $_SESSION[$arg] = '';
                    unset($_SESSION[$arg]);
                }
            }
        }
        else { $_SESSION = array(); }
    }

    /**
    * @public
    * @desc Destroy the existing session.
    * Exits the script with an error message if the session not started
    * @return void
    */
    public static function destroy()
    {
        if ( ! self::isStarted() ) {
            exit(self::SESSION_NOT_STARTED_ERROR);
        }

        setcookie(self::$name, '', time()-9888776689, '/');

        unset($_SESSION);
        session_unset();
        session_destroy();
        $_SESSION = array();
    }

}
/* End of file: IrisSession.php */