<?php  if ( ! defined('BASE_PATH')) { return; }
/**
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2010-2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* static class IrisValidator
* Provides different validation methods.
*/
class IrisValidator
{
    /**
    * @public
    * @static
    * @desc Regular expression to validate an email address.
    * @type string
    */
    public static $_emailFormat = '/^[^0-9][a-zA-Z0-9_]+([.][a-zA-Z0-9_]+)*[@][a-zA-Z0-9_]+([.][a-zA-Z0-9_]+)*[.][a-zA-Z]{2,4}$/';
    /**
    * @public
    * @static
    * @desc Regular expression to validate a password format.
    *         Using this regexp, usernames must be between 3 and 25 characters in length, alpha-numeric, and allow underscores and dashes.
    * @type string
    */
    public static $_usernameFormat = '/^[a-zA-Z]+[a-zA-Z\d_]{2,16}$/i';
    /**
    * @public
    * @static
    * @desc Custom Regular expression to validate a password format. Should be passed to sprintf function before use!
    *         Using this regexp, passwords must be between $minLength and $maxLength characters in length, alpha-numeric,
     * and allow the provided special characters if any.
    * @type string
    */
    public static $_passwordFormat = '/^[a-zA-Z]+[a-zA-Z\d_%s]{%d,%d}$/i';
    /**
    * @public
    * @static
    * @desc Regular expression to validate an IP v4 address.
    * @type string
    */
    public static $_ipv4Format = '/^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$/';
    /**
    * @public
    * @static
    * @desc Regular expression to validate an URL address.
    *        Urls might point to either a domain or a subdomain . Only the http protocol is allowed.
    * @type string
    */
    public static $_urlFormat = '^(https?|ftp)\:\/\/([a-z0-9+!*(),;?&=\$_.-]+(\:[a-z0-9+!*(),;?&=\$_.-]+)?@)?[a-z0-9+\$_-]+(\.[a-z0-9+\$_-]+)*(\:[0-9]{2,5})?(\/([a-z0-9+\$_-]\.?)+)*\/?(\?[a-z+&\$_.-][a-z0-9;:@/&%=+\$_.-]*)?(#[a-z_.-][a-z0-9+\$_.-]*)?\$';


	/**
	* @private
	* @desc Constructor
	*/
	private function __construct(){}

    /**
	* @public
	* @static
    * @desc Check to see if the php filters are available
    * @return boolean
    */
    public static function filtersLoaded()
    {
        return (function_exists('filter_list') ? true : false);
    }

    /**
 	* @public
	* @static
    * @desc Validate an email address against a regular expression.
	* @param string  The email to check
	* @param string  The regular expression to use for validation.
    * @return boolean
    */
    public static function ValidateEmail( $email, $regexp = null, $validateDomain = false )
    {
        if ( empty($regexp) )
        {
            $regexp = self::$_emailFormat;
            if (self::filtersLoaded())
            {
                $__email = filter_var($email, FILTER_SANITIZE_EMAIL);
                $__email = filter_var($email, FILTER_VALIDATE_EMAIL);
                if ($__email !== false) {
                    if ($validateDomain) { return self::ValidateEmailDomain($email); }
					return true;
                }
                return false;
            }
        }
        if (preg_match($regexp, $email)) {
            if ($validateDomain) { return self::ValidateEmailDomain($email); }
            return true;
        }
        return false;
    }

    public static function ValidateEmailDomain( $email )
    {
        $domain = strtok($email, "@");
        $domain = strtok("@");
        return (getmxrr($domain,$mxrecords) ? true : false);
    }

    /**
 	* @public
	* @static
    * @desc Validate a URL address against a regular expression
    * Urls might point to either a domain . Only the http protocol is allowed.
	* @param string  The url to check
	* @param string  The regular expression to use for validation.
    * @return boolean
    */
    public static function ValidateUrl( $url, $regexp = '' )
    {
        if ( empty($regexp) )
        {
            $regexp = self::$_urlFormat;
            if (self::filtersLoaded())
            {
                $__url = filter_var($url, FILTER_SANITIZE_URL); // Only allows a-zA-Z0-9`~!@#$%^&*()-_=+[{]};:'"<,>.?/|
                $__url = filter_var($url, FILTER_VALIDATE_URL);
                return (($__url !== false) ? true : false);
            }
        }
        return preg_match($regexp, $url);
    }

    /**
 	* @public
	* @static
    * @desc Validate an IP address against a regular expression
	* @param string  The ip address to check
	* @param string  The regular expression to use for validation.
    * @return boolean
    */
    public static function ValidateIPv4( $ip_address, $regexp = null )
    {
        if ( empty($regexp) )
        {
            $regexp = self::$_ipv4Format;
            if (self::filtersLoaded())
            {
                $ip = filter_var($ip_address, FILTER_VALIDATE_IP);
                return (($ip !== false) ? true : false);
            }
        }
        return preg_match($regexp, $ip_address);
    }

    /**
 	* @public
	* @static
    * @desc Validate a username against a regular expression.
    * 		Using the default regexp, usernames must be between 3 and 25 characters in length, alpha-numeric, and allow underscores and dashes.
	* @param string  The username to check
	* @param string  The regular expression to use for validation.
    * @return boolean
    */
    public static function validateUserNameFormat( $name, $regexp = null )
    {
        if ( empty($regexp) )
        {
            $regexp = self::$_usernameFormat;
        }
        return preg_match($regexp, $name);
    }

    /**
 	* @public
	* @static
    * @desc Validate a password against a regular expression.
    * 		Using the default regexp, passwords must be between 7 and 125 characters in length, alpha-numeric, and allow the following special characters: @ # $ ^ + _
	* @param string  The password to check
	* @param string  The regular expression to use for validation.
    * @return boolean
    */
    public static function validatePasswordFormat( $password
                                                    ,$regexp = null
                                                    ,$allowSpecialCharacters = '\@\#\$\^\+\.\?\!\%\&\*'
                                                    ,$minLength = 7
                                                    ,$maxLength = 125 )
    {
        if ( empty($regexp) )
        {
            self::$_passwordFormat = sprintf(self::$_passwordFormat, $allowSpecialCharacters, $minLength, $maxLength);
            $regexp = self::$_passwordFormat;
        }
        return preg_match($regexp, $password);
    }

	/**
	 * @public
	 * @static
	 * @desc Check to see if the provided string is alphanumeric or not.
	 * @param boolean
	 */
	public function isAlphaNumeric( $string )
	{
		if (empty ( $string )) { return false; }
		return (ctype_alnum($string)) ? true : false;
	}

	/**
	 * @public
	 * @static
	 * @desc Check to see if the provided string is numeric or not.
	 * @param boolean
	 */
	public function isNumeric( $string )
	{
		if (empty ( $str )) { return false; }
		return (ctype_digit($string)) ? true : false;
	}


}
/* End of file: IrisValidator.php */