<?php
/**
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2010-2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* @class IrisLogger
* This class facilitates the error logging of your application
*/
class IrisLogger
{
	/**
	* @private
	* Holds the list of log errors retrieved from the log file
	* @type array
	*/
	private $_entries = null;
	/**
	* @private
	* Holds the directory path to the log file
	* @type string
	*/
	private $_logFilePath = null;
	/**
	* @private
	* Holds the list of error severities
	* @type array
	*/
	private $_severities = array
	(
		'High' => '#f00',
		'Medium' => '#00c',
		'Low' => '#9C8426'
	);

	/**
	* @public
	* Holds the high severy error string
	* @type const
	*/
	const HIGH_SEVERITY = 'High';
	/**
	* @public
	* Holds the medium severy error string
	* @type const
	*/
	const MEDIUM_SEVERITY = 'Medium';
	/**
	* @public
	* Holds the low severy error string
	* @type const
	*/
	const LOW_SEVERITY = 'Low';

	/**
	* @public
	* Constructor
	* @param string The path to the log file
	*/
	public function __construct( $logFilePath )
	{
		$this->_logFilePath = $logFilePath;
		$this->_open();
	}

	/**
	* @public
	* Add a new error message into the internal list
	* @param string The error message to store
	* @param string The severity of the error message
	* @return IrisLogger object
	*/
	public function add( $errorMessage, $severity = self::LOW_SEVERITY )
	{
		$date = date("M-d-Y",time());

		$data = array(
			'date' => $date,
			'hour' => date("H:i:s",time()),
			'severity' => $severity,
			'message' => $errorMessage
		);

		$this->_entries->$date = @(array)$this->_entries->$date;
		if ($this->dateExists($date))
		{
			array_push($this->_entries->$date, $data);
		}
		else
		{
			array_push($this->_entries->$date, $data);
		}
		return $this;
	}

	/**
	* @public
	* Check to see whether or not the specified date exists in the
	* internal list of error messages retrieved from the log file
	* @param string The date to check. Must be in the following format: M-d-Y
	* @return boolean
	*/
	public function dateExists( $date )
	{
		return isset($this->_entries->$date);
	}

	/**
	* @public
	* Retrieve all entries from the log file for the specified date
	* @param string The date to check. Must be in the following format: M-d-Y
	* @return array
	*/
	public function getByDate( $date )
	{
		$data = array();
		if (empty($this->_entries)) { return $data; }
		foreach( $this->_entries as $k )
		{
			if ($k == $date) {
				array_push($data, $k);
				break;
			}
		}
		return $data;
	}

	/**
	* @public
	* Retrieve all entries from the log file for the specified severity type
	* @param string The error messages' severity
	* @return array
	*/
	public function getBySeverity( $severity )
	{
		$data = array();
		if (empty($this->_entries)) { return $data; }
		foreach( $this->_entries as $k=>$v )
		{
			foreach( $v as $i )
			{
				if (is_array($i)) {
					$sev = $i['severity'];
				}
				else { $sev = $i->severity; }

				if (Util::stringsEqual($sev, $severity)) {
					array_push($data, $i);
				}
			}
		}
		return $data;
	}

	/**
	* @public
	* Retrieve all entries from the internal list of error messages
	* @return array
	*/
	public function getAll()
	{
		return $this->_entries;
	}

	/**
	* @public
	* Delete all entries from the internal list of error messages
	* @return IrisLogger object
	*/
	public function deleteAll()
	{
		$this->_entries = '';
		return $this;
	}
	
	/**
	* @public
	* Delete all entries from the internal list of error messages by date
	* @param string The date to check. Must be in the following format: M-d-Y
	* @return IrisLogger object
	*/
	public function deleteByDate( $date )
	{
		if ($this->dateExists($date)) {
			unset($this->_entries->$date);
		}
		return $this;
	}

	/**
	* @public
	* Delete all entries from the internal list of error messages by severity
	* @param string The severity. Defaults to LOW_SEVERITY
	* @return IrisLogger object
	*/
	public function deleteBySeverity( $severity = self::LOW_SEVERITY )
	{
		if ( ! empty($this->_entries))
		{
			foreach( $this->_entries as $k => &$v)
			{
				$num = count($v);
				for ($i = 0; $i < $num; $i++)
				{
					$item = $v[$i];
					if (is_array($item)) {
						$sev = $item['severity'];
					}
					else { $sev = $item->severity; }
	
					if (Util::stringsEqual($sev, $severity)) {
						unset($v[$i]);
					}
				}
			}
		}
		return $this;
	}

	/**
	* @public
	* Save all entries from the internal list of error messages into the log file
	* @return IrisLogger object
	*/
	public function save()
	{
		$data = json_encode($this->_entries);
		file_put_contents($this->_logFilePath, $data);
		return $this;
	}

	/**
	* @public
	* Display all error messages from the log file in an html formatted way
	* @return string
	*/
	public function format( $entries )
	{
		if (empty($entries)) { return ''; }

		$result = new StringBuilder();

		foreach( $entries as $date=>$entry )
		{
			$result->append('<div style="margin: 2px 2px; padding: 4px 4px;">');
			$result->append('<h3>'.$date.'</h3>');
			foreach( $entry as $item )
			{
				if (is_array($item)) {
					$time = $item['hour'];
					$error = $item['message'];
					$severity = $item['severity'];
				}
				else {
					$time = $item->hour;
					$error = $item->message;
					$severity = $item->severity;
				}
				$format = '<p><strong>@ %s</strong></p><p style="color:%s;">%s</p>';
				$error = sprintf($format, $time, $this->_severities[$severity], $error);
				$result->append($error);
			}
			$result->append('</div>');
		}
		return $result->__toString();
	}

	/**
	* @public
	* Alias of the self::__toString() method
	* @return string
	*/
	public function toString()
	{
		return $this->__toString();
	}
	/**
	* @public
	* Display all error messages from the log file as they are retrieved from the log file.
	* @return json string
	*/
	public function __toString()
	{
		return file_get_contents($this->_logFilePath);
	}

	/**
	* @private
	* Open the log file and populate the internal list of error messages with the content of the log file.
	* @return IrisLogger object
	*/
	private function _open()
	{
		$this->_entries = json_decode(file_get_contents($this->_logFilePath));
		if (is_null($this->_entries)) {
			$this->_entries = new stdClass();
		}
		return $this;
	}
}
/* End of file: Logger.php */