<?php if ( ! defined('BASE_PATH')) { return; }
/**
* @requires : constant SITE_PATH           [ @see cleanRequest() }
* @requires : constant APP_LOG_FILE_PATH   [ @see logError(), getLogErrors() }
* @requires : class Util                   [ @see cleanRequest() }
* @requires : class StringBuilder          [ @see cleanRequest() }
* @requires : file core.functions.php      [ @see cleanRequest() }
|----------------------------------------------------------------------------------------------------------------------------
|
* @package      IrisMVC
* @author       Costin Trifan
* @copyright    2011 Costin Trifan <http://irismvc.net/>
* @license      Microsoft Public License (Ms-PL)  http://irismvc.net/license.txt
* @class Sentinel
* This static class provides different validation methods in order to keep your website safe, and it should be loaded before
* using any of the information coming through the request.
|----------------------------------------------------------------------------------------------------------------------------
| ! CONFIGURATION
|----------------------------------------------------------------------------------------------------------------------------
* @optional:
|     $disallowCharsUrl  : Will hold the list of characters that are not allowed in the URL
|     $maxLengthUrl      : Will hold the number of maximum allowed characters in the URL
|     $xssRegex          : Will hold the list of regular expressions to use against XSS attacks
|     $serverWaitTime    : Will hold the number of seconds the server should wait before starting to process the request
|     $urlTooLongErrorMessage     : Will hold the error message that will be displayed if the URL is longer than the specified max length
|     $headersSentErrorMessage    : Will hold the error message that will be displayed if the headers have already been sent and the redirection
|                                   to the new clean url is not possible
|----------------------------------------------------------------------------------------------------------------------------
*/
class Sentinel
{
	private function __clone(){}
	private function __sleep(){}
	private function __wakeup(){}
	private function __construct(){}
	public function __toString(){ return ''; }

	/**
	* @public
	* @static
	* @ Set the list of characters that are not allowed in the URL
	* @ The $ will not allow any of the predefined PHP variables such as: $_SERVER, $_REQUEST, etc...
	* @ to be passed into the URL.
	* @ The ( will not allow functions to be passed into the URL.
	* @ The ' (single quote) will protect URLs against SQL injections.
	* @ The \ escaped(\\) will prevent hex characters into the URL
	* @type array
	*/
	public static $disallowCharsUrl = array('@', '*', '!', '`', "'", '(', ')', '$', '\\', ',', ';','<','>');
	/**
	* @public
	* @static
	* @ Set the maximum number of characters allowed in the URL
	* @ Defaults to 255
	* @type int
	*/
	public static $maxLengthUrl = 255;
	/**
	* @public
	* @static
	* @ Set the regular expression to combat common xss attacks
	* @type array
	*/
	public static $xssRegex = array("#<(.*)>#msi", "#&lt;(.*)&gt;#msi", "#%3C(.*)?%3E#msi");
	/**
	* @public
	* @static
	* @ Will hold the number of seconds the server should wait before starting to process the request
	* @ in case of an XSS attack
	* @type int
	*/
	public static $serverWaitTime = 0;
	/**
	* @public
	* @static
	* Holds the error message that will be displayed if the headers have already been sent and the redirection
	* to the new clean url is not possible
	* @type string
	*/
	public static $headersSentErrorMessage = 'Error: An error has occurred!';
	/**
	* @public
	* @static
	* Holds the reference to the instance of the logging class
	* @type object
	*/
	public static $logger = null;
	/**
	* @private
	* @static
	* Holds the attack severity
	* @see self::isUrlDirty
	* @type string
	*/
	private static $_severity = '';

	/**
	* @public
	* @static
	* This method will escape the specified string $source.
	* Suppose the following string is passed to this function: <span>It's okay..I'm here</span>
	* It will be converted to : &lt;span&gt;It&#039;s okay..I&#039;m here&lt;/span&gt; which will be clean enough to be send to a database or to a query
	* @param string The string to be escaped
	* @return string
	*/
	public static function escape( $source )
	{
		return htmlentities($source, ENT_QUOTES, 'UTF-8', false);
	}

	/**
	* @public
	* @static
	* This method will convert all html entities from the provided string $source back to their applicable characters.
	* @param string The string to be unescaped
	* @retrn string
	*/
	public static function unescape( $source )
	{
		return html_entity_decode($source, ENT_QUOTES, 'UTF-8');
	}

	/**
	* @public
	* @static
	* @uses prepareInput()
	* This function will strip out all tags from the HTML string and also
	* remove all attributes that are inside those tags, preventing people from messing up your page.
	* @param string The string to clean up
	* @param bool Whether or not the string is unescaped. Defaults to true.
	* @param bool Whether or not to remove all tags from the string. Defaults to true.
	* @return string The cleaned and escaped string
	*/
	public static function agressiveClean( $string, $unescaped = true, $stripTags = true )
	{
		if ( ! $unescaped ) {
			$string = self::unescape($string);
		}

		//@ strip all attributes from tags
		$string = urldecode($string);
		$string = preg_replace("/<([a-z][a-z0-9]*)[^>]*?(\/?)>/msi","<$1$2>", $string);

		if ($stripTags) {
			$string = strip_tags($string);
		}

		return self::escape($string);
	}

	/**
	* @public
	* @static
	* Sanitize the values of the provided array $request using the specified escaping methods
	* from the $methods array.
	* Ex: 
	*      $_POST = Sentinel::sanitize($_POST); // uses the default Sentinel::escape method
	*      $_POST = Sentinel::sanitize($_POST, array("strip_tags", "Sentinel::escape")); // strip tags and then escape
	*      $_POST = Sentinel::sanitize($_POST, array("Sentinel::agressiveClean")); // remove all tags and their attributes
	*      $array = Sentinel::sanitize($array);
	*
	* @return array The escaped array
	*/
	public static function sanitize( array $request, array $methods = array("Sentinel::escape") )
	{
		if ( ! empty($request))
		{
			foreach( $request as $k => &$v )
			{
				if (is_string($v))
				{
					foreach( $methods as $method ) {
						$v = call_user_func($method, $v);
					}
				}
				elseif (is_array($v))
				{
					$v = self::sanitize($v, $methods);
				}
			}
		}
		return $request;
	}

	/**
	* @public
	* @static
	* Retrieve the current URL
	* @return string
	*/
	public static function getUrl()
	{
		$scheme = (!empty($_SERVER['HTTPS']) && (strtoupper($_SERVER['HTTPS']) == 'ON')) ? 'https' : 'http';
		$url = $scheme.'://'.$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'];
		return $url;
	}

	/**
	* @public
	* @static
	* Check to see if the URL contains unwanted characters
	* @return boolean
	*/
	public static function isUrlDirty( $url )
	{
		/**
		* @ Check if the number of characters in the URL have exceeded the max allowed number
		*/
		if (isset($url[self::$maxLengthUrl+1]))
		{
			self::$_severity = IrisLogger::LOW_SEVERITY;
			return true;
		}
	
		if ( ! empty(self::$disallowCharsUrl))
		{
			foreach( self::$disallowCharsUrl as $s_char )
			{
				//! If we have a match
				if ( ($s_loc = stripos($url, $s_char)) !== false )
				{
					self::$_severity = IrisLogger::MEDIUM_SEVERITY;
					return true;
				}
			}
		}
		return false;
	}

	/**
	* @ Remove invalid characters from url
	* @return string The cleaned url
	*/
	public static function removeDisallowChars( $url )
	{
		if ( ! empty(self::$disallowCharsUrl))
		{
			foreach( self::$disallowCharsUrl as $s_char )
			{
				//! If we have a match
				if ( ($s_loc = stripos($url, $s_char)) !== false )
				{
					//! Cut the url at the given position
					$url = substr($url, 0, $s_loc);

					self::$_severity = IrisLogger::MEDIUM_SEVERITY;
					break;
				}
			}
		}
		return $url;
	}

	/**
	* @ Prevent XSS attempts
	* @ Clean the url from any XSS scripts
	*/
	public static function removeXss( $url )
	{
		if ( ! empty(self::$xssRegex))
		{
			$url = preg_replace(self::$xssRegex, '', $url);
			//! replace double slashes from the end of the url, if any found
			$url = preg_replace("#/+$#", '', $url);

			self::$_severity = IrisLogger::HIGH_SEVERITY;
		}
		return $url;
	}

	/**
	* @public
	* @static
	* Remove unwanted characters and XSS attacks from URL, and reload the page using the new cleaned URL.
	* @param string the url to clean up. If omitted then the current url will be used.
	* @return string The cleaned url
	*/
	public static function cleanRequest( $url = '' )
	{
		if (empty($url)) {
			$url = self::getUrl();
		}

		// Store a copy of the url
		$curl = $url;

		// Unescape special characters from the URL (if any)
		if (strpos($url,'%') !== false)
		{
			$url = urldecode($url);
		}

		// If URL is dirty
		if (self::isUrlDirty($url))
		{
			$url = self::removeXss($url);
			$url = self::removeDisallowChars($url);
			//@ see self::removeXss()
			$curl = preg_replace("#/+$#", '', $curl);

			// Log the malformed request
			$e = new StringBuilder();
			$e->append('Request Method: '.$_SERVER['REQUEST_METHOD'],PHP_EOL);
			$e->append('Url: '.urldecode($_SERVER['REQUEST_URI']));
			$e = self::escape($e);
			self::logError($e, self::$_severity);

			/**
			* @ Reloads the page(if necessary) using the cleaned up url
			* @ If we need to redirect means the URL was dirty so let's make 
			* @ this "bad" guy wait for a long (desired) time before he would
			* @ be able to try again.
			*/
			if ( self::$serverWaitTime > 0 ) {
				sleep( self::$serverWaitTime );
			}

			if ( ! headers_sent()) {
				header("Location: {$url}");
				exit;
			}
			else { exit(self::$headersSentErrorMessage); }
		}

		//@ return clean url
		return $url;
	}

	/**
	* @public
	* @static
	* @uses const APP_LOG_FILE_PATH
	* Log the provided error message. The log file must be writable.
	* @param string The error message to save into the log file (APP_LOG_FILE_PATH)
	* return void
	*/
	public static function logError( $errorMessage = '', $severity = IrisLogger::MEDIUM_SEVERITY )
	{
		if (self::$logger)
		{
			self::$logger->add($errorMessage, $severity)->save();
		}
	}

	/**
	* @public
	* @static
	* Retrieve the content of the log file. The log file must be readable.
	* return string
	*/
	public static function getLogErrors()
	{
		if (self::$logger)
		{
			$logs = self::$logger->getAll();
			return self::$logger->format($logs);
		}
	}

}
/* End of file: Sentinel.php */