<?php
/**
 * @author wonli <wonli@live.com>
 * ReadLogFile.php
 */

namespace app\admin\modules\log;

use Cross\Exception\CoreException;
use SplFileObject;

/**
 * LOG文件处理
 * @author wonli <wonli@live.com>
 *
 * Class ReadLogFile
 * @package app\admin\modules\log
 */
class ReadLogFile
{
    /**
     * @var SplFileObject
     */
    private $fp = null;

    private $file;
    private $seek = 0;
    private $limit = 10;
    private $searchID = '';
    private $readLimit = true;
    private $searchLogTime = '';
    private $totalLine;
    private $maxLine;
    private $minLine;
    private $useBisection = true;
    private $inEndScan = false;
    private $blockSize = 30;

    /**
     * 设置要读取的文件
     *
     * @param string $file
     * @return $this
     * @throws CoreException
     */
    function setFile($file)
    {
        if (!file_exists($file)) {
            throw new CoreException('文件不存在');
        }

        $this->file = $file;
        $this->fp = new SplFileObject($this->file);
        return $this;
    }

    /**
     * 设置seek
     *
     * @param int $seek
     * @return ReadLogFile
     */
    function setSeek($seek)
    {
        $this->seek = (int)$seek;
        return $this;
    }

    /**
     * 获取当前所在行数
     *
     * @return int
     */
    function getCurrentLine()
    {
        return $this->seek - 1;
    }

    /**
     * @param $limit
     * @return ReadLogFile
     */
    function setBlockLimit($limit)
    {
        $this->limit = max(1, $limit);
        return $this;
    }

    /**
     * ID搜索
     *
     * @param string $id
     * @return ReadLogFile
     */
    function search($id)
    {
        $this->seek = 0;
        $this->searchID = $id;
        $this->readLimit = false;

        list(, $time,) = explode('.', $id);
        $this->searchLogTime = $time;

        //10秒以内不使用二分法
        if ($time < 10) {
            $this->useBisection = false;
        } else {
            $this->fp->seek(PHP_INT_MAX);
            $this->totalLine = $this->fp->key();
            $this->maxLine = $this->totalLine;
            $this->seek = ceil($this->totalLine / 2);
            $this->minLine = $this->seek;
        }

        return $this;
    }

    /**
     * @return array
     */
    function read()
    {
        if ($this->inEndScan && $this->seek > $this->maxLine) {
            return array();
        }

        $this->fp->seek($this->seek);

        $i = 0;
        $is_find = 0;
        $logContent = array();
        $buffer_block = array();
        while (!$this->fp->eof()) {

            $this->seek++;

            $buffer = $this->fp->fgets();
            $isBlockStart = preg_match("/^-.*(\(.*-(.*\.(\d+)\..*)\)).*-/", $buffer, $match);
            if ($isBlockStart) {

                //日志块最大行数
                //在endScan时决定扫描范围
                $cbs = count($buffer_block);
                if ($cbs > $this->blockSize) {
                    $this->blockSize = $cbs;
                }

                //限制读取的日志块
                if ($this->readLimit && $i >= $this->limit) {
                    $this->seek--;
                    break;
                } else {
                    $i++;
                }

                //是否已经找到匹配的内容
                if ($is_find) {
                    $this->seek--;
                    $logContent = array();
                    break;
                }

                $currentLogID = trim($match[2]);
                $currentLogGenTime = trim($match[3]);
                if ($this->searchID) {
                    if ($currentLogID == $this->searchID) {
                        $is_find = 1;
                    } elseif ($this->useBisection) {
                        //计算偏移量
                        $offset = floor(($this->maxLine - $this->minLine) / 2);
                        if ($offset >= 0) {
                            if ($offset == 0) {
                                //当二分法偏移量为0且无结果时
                                //禁用二分法, 进入最后扫描流程
                                $this->inEndScan = true;
                                $this->useBisection = false;

                                //设置最后扫描起点
                                $scanScope = ceil($this->blockSize * 2);
                                if ($this->minLine > $scanScope) {
                                    $this->minLine -= $scanScope;
                                } else {
                                    $this->minLine = 0;
                                }

                            } else {
                                if ($currentLogGenTime < $this->searchLogTime) {
                                    $this->minLine += $offset;
                                } else {
                                    $this->maxLine = $this->minLine;
                                    $this->minLine -= $offset;
                                }
                            }

                            $this->seek = $this->minLine;
                            return $this->read();
                        }
                    }

                    //处理非二分法检索的无结果状态特殊处理
                    //当前时间大于扫描时间时候, 判定无日志内容
                    if (!$this->useBisection && $currentLogGenTime > $this->searchLogTime) {
                        $buffer_block = array();
                        $logContent = array();
                        break;
                    }
                }

                //当搜索到新的块时候, 保存当前块
                if (!empty($buffer_block)) {
                    $logContent[] = implode('', $buffer_block);
                    $buffer_block = array();
                }
            }

            $buffer_block[] = $buffer;
        }

        if (!empty($buffer_block)) {
            $logContent[] = implode('', $buffer_block);
        }

        return $logContent;
    }

}