<?php
/**
 * @author wonli <wonli@live.com>
 * ConfigModule.php
 */


namespace app\forum\modules\common;

use RecursiveArrayIterator;
use RecursiveIteratorIterator;

/**
 * 配置管理
 * @author wonli <wonli@live.com>
 *
 * Class ConfigModule
 * @package app\forum\modules\common
 */
class ConfigModule extends BaseModule
{
    /**
     * 保存配置数据(有则更新,无则插入)
     *
     * @param array $data
     * @param string $groupName
     * @throws \Cross\Exception\CoreException
     */
    function saveConfig(array $data, $groupName)
    {
        $insertData = array();
        if (!empty($data)) {
            $configKeyMap = $this->getConfigKeyByGroupName($groupName);
            $pathData = self::arrayToPath($data);

            foreach ($pathData as $key => $value) {
                $configID = &$configKeyMap[$key];
                if ($configID) {
                    $this->update($configID, $value);
                } else {
                    $insertData[] = array(
                        'key' => $key,
                        'value' => $value,
                        'group_name' => $groupName,
                    );
                }
            }

            //添加
            if (!empty($insertData)) {
                $this->link->add($this->siteConfig, $insertData, true);
            }
        }

        $this->updateFileCache();
    }

    /**
     * 获取所有配置
     *
     * @param string $group_name
     * @return mixed
     * @throws \Cross\Exception\CoreException
     */
    function getAllConfig($group_name = '')
    {
        $query = $this->link->select('`key`, value')->from($this->siteConfig);
        if ($group_name) {
            $query->where(array('group_name' => $group_name));
        }

        $data = $query->stmt()->fetchAll(\PDO::FETCH_KEY_PAIR);
        if (!empty($data)) {
            return self::pathListToArray($data);
        }

        return $data;
    }

    /**
     * 获取配置
     *
     * @param string $key
     * @return bool
     * @throws \Cross\Exception\CoreException
     */
    function get($key)
    {
        return $this->link->get($this->siteConfig, '*', array(
            '`key`' => $key,
        ));
    }

    /**
     * @param int $id
     * @param mixed $value
     * @throws \Cross\Exception\CoreException
     */
    function update($id, $value)
    {
        if (is_array($value)) {
            $value = json_encode($value);
        }

        $data['value'] = $value;
        $this->link->update($this->siteConfig, $data, array(
            'id' => (int)$id
        ));
    }

    /**
     * @throws \Cross\Exception\CoreException
     */
    protected function updateFileCache()
    {
        $configFile = $this->getFilePath('config::site.config.php');
        if (file_exists($configFile)) {
            unlink($configFile);
        }

        //类型名字和mode对应关系
        $typeNameModeMap = array_flip(BaseModule::$typeMap);

        $defaultModeName = 'question';
        $availableMode = array();
        $data = $this->getAllConfig();
        if (!empty($data)) {
            foreach ($data as $key => &$configs) {
                switch ($key) {
                    //处理内容模型缓存数据
                    case 'mode':
                        $saveConfigs = array();
                        foreach ($configs as $k => $val) {
                            if ($val['status']) {
                                $saveConfigs[$k] = $val['name'];
                                $availableMode[$k] = $typeNameModeMap[$k];
                            }
                        }

                        $configs = $saveConfigs;
                        break;

                    case 'default_mode':
                        $defaultModeName = BaseModule::$typeMap[$configs];
                        break;
                }
            }

            //处理配置缓存
            $data['availableMode'] = $availableMode;
            $data['defaultModeName'] = $defaultModeName;
            $data['typeNameModeMap'] = $typeNameModeMap;

            //文件缓存
            parent::saveArrayToFile($configFile, $data);
        }
    }

    /**
     * 获取分组下的所有key
     *
     * @param string $type
     * @return array
     * @throws \Cross\Exception\CoreException
     */
    protected function getConfigKeyByGroupName($type)
    {
        $data = $this->link->select('`key`, id')->from($this->siteConfig)
            ->where(array('group_name' => $type))->stmt()->fetchAll(\PDO::FETCH_KEY_PAIR);

        return $data;

    }

    /**
     * 配置数据转二维数组
     *
     * @param array $data
     * @return array
     */
    static function pathListToArray(array $data)
    {
        $result = array();
        if (!empty($data)) {
            foreach ($data as $key => $value) {
                self::assignArrayByPath($result, $key, $value);
            }
        }

        return $result;
    }

    /**
     * 数组转路径
     *
     * @param array $data 如果数组在外部被引用修改过, 出入会出现不可预料的BUG
     * @return array
     */
    static function arrayToPath($data)
    {
        $result = array();
        $r = new RecursiveIteratorIterator(new RecursiveArrayIterator($data));
        foreach ($r as $leafValue) {
            $keys = array();
            array_map(function ($d) use (&$keys, $r) {
                $keys[] = $r->getSubIterator($d)->key();
            }, range(0, $r->getDepth()));

            $result[implode('.', $keys)] = $leafValue;
        }

        return $result;
    }

    /**
     * 路径转数组
     *
     * @param array $result
     * @param string $path
     * @param mixed $value
     */
    static function assignArrayByPath(&$result, $path, $value)
    {
        $keys = explode('.', $path);
        foreach ($keys as $key) {
            $result = &$result[$key];
        }

        $result = $value;
    }
}