<?php

namespace app\forum\modules\content;

use app\forum\modules\common\BaseModule;
use app\forum\modules\topic\TopicModule;

/**
 * @author wonli <wonli@live.com>
 * ContentModule.php
 */
class ContentModule extends BaseModule
{
    const INVITE_NEW = 0; //未处理
    const INVITE_IGNORE = 1; //忽略
    const INVITE_FINISH = 2; //已完成

    /**
     * 相关内容
     *
     * @param string $topic_ids
     * @param int $type
     * @param int $limit
     * @return array
     * @throws \Cross\Exception\CoreException
     */
    function getCorrelationContent($topic_ids, $type, $limit = 10)
    {
        $t = $this->link->select('title, type, title_id')->from($this->title)
            ->where("topic_ids = '{$topic_ids}' AND type={$type} AND status=1")
            ->orderBy('interact_count DESC')->limit($limit)
            ->getSQL(true);

        $data = $this->link->select('t.*, q.question_id, p.posts_id, a.article_id')
            ->from("({$t}) t LEFT JOIN {$this->questions} q ON t.type=1 AND t.title_id=q.title_id
                LEFT JOIN {$this->posts} p ON t.type=2 AND t.title_id=p.title_id
                LEFT JOIN {$this->articles} a ON t.type = 3 AND t.title_id = a.title_id"
            )->stmt()->fetchAll(\PDO::FETCH_ASSOC);

        return $data;
    }

    /**
     * 基础内容的附加信息
     *
     * @param string $base_sql
     * @param $uid
     * @param bool $is_editor
     * @return mixed
     * @throws \Cross\Exception\CoreException
     */
    protected function getContentBaseExtendInfo($base_sql, $uid, &$is_editor = false)
    {
        $question_base_build = $this->link->select('q.*, u.nickname, u.account, u.avatar, u.qr, c.id collection_id, fu.id follow_user_id, fc.id follow_id, count(ffc.id) total_follow')
            ->from("({$base_sql}) q LEFT JOIN {$this->user} u ON q.uid=u.uid
                LEFT JOIN {$this->collections} c ON (c.uid={$uid} AND q.title_id=c.title_id)
                LEFT JOIN {$this->following_user} fu ON (q.uid = fu.following_uid AND fu.uid = {$uid})
                LEFT JOIN {$this->following_content} fc ON (fc.uid={$uid} AND q.title_id=fc.title_id)
                LEFT JOIN {$this->following_content} ffc ON q.title_id=ffc.title_id");

        $content_base_info = $question_base_build->stmt()->fetch(\PDO::FETCH_ASSOC);
        if (empty($content_base_info['title_id'])) {
            return array();
        }

        //所属话题
        $topics_name_string = '';
        $topics_ids = $topics_info = $topics_name = array();
        if (!empty($content_base_info['topic_ids'])) {
            $TOPIC = new TopicModule();
            $topics_info = $TOPIC->getTopicInfo4Strings($content_base_info['topic_ids'], 'topic_id, topic_name, topic_url');
            array_walk($topics_info, function (&$topic) use (&$topics_name, &$topics_ids) {
                $topics_ids[] = $topic['topic_id'];
                $topics_name[] = $topic['topic_name'];
                $topic['can_choose'] = 1;
            });
            $topics_name_string = implode(',', $topics_name);
        }

        //是否允许编辑和追加内容
        $content_base_info['can_edit'] = 0;
        $content_base_info['is_editor'] = 0;
        $content_base_info['can_append'] = 0;

        if ($uid > 0) {
            //是否是编辑
            $editors = $this->getTopicEditor($topics_ids);
            if (isset($editors[$uid])) {
                $is_editor = true;
                $content_base_info['can_edit'] = 1;
                $content_base_info['is_editor'] = 1;
            }

            if ($uid == $content_base_info['uid']) {
                $content_base_info['can_edit'] = 1;

                //除问题之外可以追加
                $content_base_info['can_append'] = (int)($content_base_info['type'] != 1);
            }
        }

        $content_base_info['topics'] = $topics_info;
        $content_base_info['topics_names'] = $topics_name_string;
        $content_base_info['content_type'] = self::$typeMap[$content_base_info['type']];
        return $content_base_info;
    }

    /**
     * 更新用户邀请状态
     *
     * @param $invite_id
     * @param $status
     * @return bool
     * @throws \Cross\Exception\CoreException
     */
    function updateInviteStatus($invite_id, $status)
    {
        return $this->link->update($this->invite, array('status' => (int)$status), array('id' => (int)$invite_id));
    }

    /**
     * 按类型和交互内容ID获取交互内容，并格式化为统一格式
     *
     * @param int $type
     * @param int $interact_id
     * @return array|string
     * @throws \Cross\Exception\CoreException
     */
    function getInteractContent($type, $interact_id)
    {
        $interact_id = (int)$interact_id;
        switch ($type) {
            case BaseModule::TYPE_ARTICLE:
                $AM = new ArticleModule();
                $data = $AM->getCommentById($interact_id, 'comment_id id, comment_content content');
                $data['content'] = strip_tags($data['content']);
                break;

            case BaseModule::TYPE_POSTS:
                $PM = new PostsModule();
                $data = $PM->getReplyInfoById($interact_id, 'reply_id id, reply_content content');
                break;

            case BaseModule::TYPE_QUESTION:
                $QM = new QuestionModule();
                $data = $QM->getAnswerById($interact_id, 'answer_id id, answer_content content');
                break;
            default:
                return $this->result(200541);
        }

        return $this->result(1, $data);
    }

    /**
     * 屏蔽交互内容
     *
     * @param int $uid
     * @param int $type
     * @param int $interact_id
     * @param string $act
     * @return array|string
     * @throws \Cross\Exception\CoreException
     */
    function blockInteractContent($uid, $type, $interact_id, $act = '')
    {
        switch ($type) {
            case BaseModule::TYPE_ARTICLE:
                $topic_ids = $this->link->select('t.topic_ids')
                    ->from("{$this->articles_comment} ac 
                        LEFT JOIN {$this->articles} a ON ac.article_id=a.article_id
                        LEFT JOIN {$this->title} t ON a.title_id=t.title_id")
                    ->where(array('ac.comment_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_COLUMN);


                $actMap = array(
                    'hide' => ArticleModule::COMMENT_HIDDEN,
                    'block' => ArticleModule::COMMENT_BLOCKED,
                    'normal' => ArticleModule::COMMENT_NORMAL
                );

                $M = new ArticleModule();
                $action = 'updateCommentStatus';
                break;

            case BaseModule::TYPE_POSTS:

                $topic_ids = $this->link->select('t.topic_ids')
                    ->from("{$this->reply} r 
                        LEFT JOIN {$this->posts} p ON r.posts_id=p.posts_id
                        LEFT JOIN {$this->title} t ON p.title_id=t.title_id")
                    ->where(array('r.reply_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_COLUMN);

                $actMap = array(
                    'hide' => PostsModule::REPLY_HIDDEN,
                    'block' => PostsModule::REPLY_BLOCKED,
                    'normal' => PostsModule::REPLY_NORMAL
                );

                $M = new PostsModule();
                $action = 'updateReplyStatus';
                break;

            case BaseModule::TYPE_QUESTION:
                $topic_ids = $this->link->select('t.topic_ids')
                    ->from("{$this->answers} a 
                        LEFT JOIN {$this->questions} q ON a.question_id=q.question_id
                        LEFT JOIN {$this->title} t ON q.title_id=t.title_id")
                    ->where(array('a.answer_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_COLUMN);

                $actMap = array(
                    'hide' => QuestionModule::ANSWER_HIDDEN,
                    'block' => QuestionModule::ANSWER_BLOCKED,
                    'normal' => QuestionModule::ANSWER_NORMAL
                );

                $M = new QuestionModule();
                $action = 'updateAnswerStatus';
                break;

            default:
                return $this->result(200541);
        }

        $topic_ids = explode(',', $topic_ids);
        $editors = $this->getTopicEditor($topic_ids);
        if (!isset($editors[$uid])) {
            return $this->result(200561);
        }

        if (!isset($actMap[$act])) {
            $status = $actMap['normal'];
        } else {
            $status = $actMap[$act];
        }

        $ret = call_user_func_array(array($M, $action), array($interact_id, $status));
        if ($ret) {
            return $this->result(1);
        }

        return $this->result(200562);
    }

    /**
     * 删除交互内容
     *
     * @param int $uid
     * @param int $type
     * @param int $interact_id
     * @return array|string
     * @throws \Cross\Exception\CoreException
     */
    function delInteractContent($uid, $type, $interact_id)
    {
        $interact_id = (int)$interact_id;
        switch ($type) {
            case BaseModule::TYPE_ARTICLE:
                $info = $this->link->select('t.topic_ids, ac.uid')
                    ->from("{$this->articles_comment} ac 
                        LEFT JOIN {$this->articles} a ON ac.article_id=a.article_id
                        LEFT JOIN {$this->title} t ON a.title_id=t.title_id")
                    ->where(array('ac.comment_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_ASSOC);

                $M = new ArticleModule();
                $action = 'delCommentById';
                break;

            case BaseModule::TYPE_POSTS:
                $info = $this->link->select('t.topic_ids, r.uid')
                    ->from("{$this->reply} r 
                        LEFT JOIN {$this->posts} p ON r.posts_id=p.posts_id
                        LEFT JOIN {$this->title} t ON p.title_id=t.title_id")
                    ->where(array('r.reply_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_ASSOC);

                $M = new PostsModule();
                $action = 'delReplyById';
                break;
            case BaseModule::TYPE_QUESTION:
                $info = $this->link->select('t.topic_ids, a.uid')
                    ->from("{$this->answers} a
                    LEFT JOIN {$this->questions} q ON a.question_id=q.question_id
                    LEFT JOIN {$this->title} t ON q.title_id=t.title_id")
                    ->where(array('a.answer_id' => $interact_id))
                    ->stmt()->fetch(\PDO::FETCH_ASSOC);

                $M = new QuestionModule();
                $action = 'delAnswerById';
                break;

            default:
                return $this->result(200571);
        }

        $topic_ids = explode(',', $info['topic_ids']);
        $editors = $this->getTopicEditor($topic_ids);
        if (!isset($editors[$uid]) && $uid != $info['uid']) {
            return $this->result(200572);
        }

        $ret = call_user_func_array(array($M, $action), array($interact_id));
        if ($ret) {
            return $this->result(1);
        }

        return $this->result(200573);
    }

    /**
     * 有权限编辑话题的人
     * <pre>
     * 当内容没有设置话题的时候, 所有编辑都有权限编辑话题
     * 当指定话题时, 只有该话题下的编辑有权限编辑内容
     * </pre>
     *
     * @param array $topic_ids
     * @return array
     * @throws \Cross\Exception\CoreException
     */
    private function getTopicEditor(array $topic_ids = array())
    {
        $condition = array();
        if (!empty($topic_ids)) {
            //topics_id = 0 所有话题管理权限
            $topic_ids[] = 0;
            $condition = array('topic_id' => array('IN', $topic_ids));
        }

        $topic_editors = array();
        $this->link->select('editor_uid')
            ->from($this->topic_editor)->where($condition)
            ->stmt()->fetchAll(\PDO::FETCH_FUNC, function ($editor_uid) use (&$topic_editors) {
                $editor = explode(',', $editor_uid);
                $topic_editors = array_merge($topic_editors, $editor);
            });

        $result = array();
        if (!empty($topic_editors)) {
            foreach ($topic_editors as $editor) {
                $editor = trim($editor);
                $result[$editor] = $editor;
            }
        }

        return $result;
    }
}
