<?php
namespace CSSTidy;

require_once __DIR__ . '/Container.php';
require_once __DIR__ . '/Template.php';
require_once __DIR__ . '/Configuration.php';
require_once __DIR__ . '/Output.php';

class CSSTidy
{
    /** @var string */
    private static $version = '1.4';

    /** @var \CSSTidy\Container */
    private $container;

    /** @var \CSSTidy\Logger */
    public $logger;

    /** @var \CSSTidy\Configuration */
    public $configuration;

    /**
     * @param Configuration|null $configuration
     */
    public function __construct(Configuration $configuration = null)
    {
        $this->container = new Container;

        if ($configuration) {
            $this->configuration = $this->container->configuration = $configuration;
        } else {
            $this->configuration = $this->container->configuration;
        }

        $this->logger = $this->container->logger;
    }

    /**
     * @param $string
     * @return Output
     * @throws \Exception
     */
    public function process($string)
    {
        $original = $string;

        // Temporarily set locale to en_US in order to handle floats properly
        $old = @setlocale(LC_ALL, 0);
        @setlocale(LC_ALL, 'C');

        $parsed = $this->container->parser->parse($string);

        if (!$parsed->isOk()) {
            @setlocale(LC_ALL, $old);
            throw new \Exception("Invalid CSS");
        }

        $this->container->optimiseSelector->process($parsed);
        $this->container->optimiseValue->process($parsed);

        switch ($this->configuration->getMergeSelectors()) {
            case Configuration::SEPARATE_SELECTORS:
                $this->container->selectorManipulate->separate($parsed);
                break;

            case Configuration::MERGE_SELECTORS:
                $this->container->selectorManipulate->mergeWithSameName($parsed);
                $this->container->selectorManipulate->mergeWithSameProperties($parsed);
                break;
        }

        if ($this->configuration->getDiscardInvalidSelectors()) {
            $this->container->selectorManipulate->discardInvalid($parsed);
        }

        if ($this->configuration->getOptimiseShorthands()){
            $this->container->optimiseShorthand->process($parsed);
        }

        foreach ($parsed->import as $import) {
            $this->container->optimiseLineAt->process($import);
        }

        foreach ($parsed->namespace as $namespace) {
            $this->container->optimiseLineAt->process($namespace);
        }

        if ($this->configuration->getSortProperties()) {
            $this->container->sorter->sortProperties($parsed);
        }

        if ($this->configuration->getSortSelectors()) {
            $this->container->sorter->sortSelectors($parsed);
        }

        /*echo '<pre>';
        var_export($parsed->properties);
        echo '</pre>';*/

        @setlocale(LC_ALL, $old); // Set locale back to original setting

        return new Output($this->configuration, $this->logger, $original, $parsed);
    }

    /**
     * @param string $string
     * @param string $fileDirectory
     * @return string
     */
    public function mergeImports($string, $fileDirectory = '')
    {
        preg_match_all('~@import[ \t\r\n\f]*(url\()?("[^\n\r\f\\"]+"|\'[^\n\r\f\\"]+\')(\))?([^;]*);~si', $string, $matches);

        $notResolvedImports = array();
        foreach ($matches[2] as $i => $fileName) {
            $importRule = $matches[0][$i];

            if (trim($matches[4][$i]) !== '') {
                $notResolvedImports[] = $importRule;
                continue; // Import is for other media
            }

            $fileName = trim($fileName, " \t\r\n\f'\"");

            $content = file_get_contents($fileDirectory . $fileName);

            $string = str_replace($importRule, $content ? $content : '', $string);

            if (!$content) {
                $notResolvedImports[] = $importRule;
                $this->logger->log("Import file {$fileDirectory}{$fileName} not found", Logger::WARNING);
            }
        }

        return implode("\n", $notResolvedImports) . $string;
    }

    /**
     * @return string
     */
    public static function getVersion()
    {
        return self::$version;
    }
}
?>