<?php

namespace CSSTidy;

/**
 * CSS Output class
 *
 * This class prints elements generated by parser.
 *
 * @package csstidy
 * @author Florian Schmitz (floele at gmail dot com) 2005-2006
 * @version 1.0.1
 */
class Output
{
    const AT_START = 1,
        AT_END = 2,
        SEL_START = 3,
        SEL_END = 4,
        PROPERTY = 5,
        VALUE = 6,
        COMMENT = 7,
        LINE_AT = 8;
    
    const INPUT = 'input',
        OUTPUT = 'output';

    /**
     * Saves the input CSS string
     * @var string
     */
    protected $inputCss;

    /**
     * Saves the formatted CSS string
     * @var string
     */
    protected $outputCss;

    /**
     * Saves the formatted CSS string (plain text)
     * @var string
     */
    protected $outputCssPlain;

    /** @var Configuration */
    protected $configuration;

    /** @var Logger */
    protected $logger;

    /** @var \CSSTidy\Element\Root */
    protected $parsed;

    /** @var array */
    protected $tokens = array();

    /**
     * @param Configuration $configuration
     * @param Logger $logger
     * @param string $inputCss
     * @param Element\Root $parsed
     */
    public function __construct(Configuration $configuration, Logger $logger, $inputCss, Element\Root $parsed)
    {
        $this->configuration = $configuration;
        $this->logger = $logger;
        $this->inputCss = $inputCss;
        $this->parsed = $parsed;
    }

    /**
     * Returns the CSS code as plain text
     * @param string $defaultMedia default @media to add to selectors without any @media
     * @return string
     * @access public
     * @version 1.0
     */
    public function plain($defaultMedia = null)
    {
        $this->generate($defaultMedia);
        return $this->outputCssPlain;
    }

    /**
     * Returns the formatted CSS code
     * @param string $defaultMedia default @media to add to selectors without any @media
     * @return string
     * @access public
     * @version 1.0
     */
    public function formatted($defaultMedia = null)
    {
        $this->generate($defaultMedia);
        return $this->outputCss;
    }

    /**
     * Returns the formatted CSS code to make a complete webpage
     * @param bool $externalCss indicates whether styles to be attached internally or as an external stylesheet
     * @param string $title title to be added in the head of the document
     * @return string
     * @access public
     * @version 1.4
     */
    public function formattedPage($externalCss = true, $title = '')
    {
        if ($externalCss) {
            $css = "\n\n<style type=\"text/css\">\n";
            $cssParsed = file_get_contents('cssparsed.css');
            $css .= $cssParsed; // Adds an invisible BOM or something, but not in css_optimised.php
            $css .= "\n\n</style>";
        } else {
            $css = "\n\n" . '<link rel="stylesheet" type="text/css" href="cssparsed.css">';
        }

        return <<<HTML
<!DOCTYPE html>
<html>
    <head>
        <title>$title</title>
        $css
    </head>
    <body>
        <code id="copytext">{$this->formatted()}</code>
    </body>
</html>
HTML;
    }

    /**
     * Get compression ratio
     * @access public
     * @return float
     * @version 1.2
     */
    public function getRatio()
    {
        $input = $this->size(self::INPUT);
        $output = $this->size(self::OUTPUT);

        return round(($input - $output) / $input, 3) * 100;
    }

    /**
     * Get difference between the old and new code in bytes and prints the code if necessary.
     * @access public
     * @return string
     * @version 1.1
     */
    public function getDiff()
    {
        if (!$this->outputCssPlain) {
            $this->formatted();
        }

        $diff = strlen($this->outputCssPlain) - strlen($this->inputCss);

        if ($diff > 0) {
            return '+' . $diff;
        } else if ($diff == 0) {
            return '+-' . $diff;
        }

        return $diff;
    }

    /**
     * Get the size of either input or output CSS in KB
     * @param string $loc Output::INPUT or Output::OUTPUT
     * @return float
     * @version 1.0
     */
    public function size($loc = self::OUTPUT)
    {
        if ($loc === self::OUTPUT && !$this->outputCss) {
            $this->formatted();
        }

        if ($loc === self::INPUT) {
            return (strlen($this->inputCss) / 1000);
        } else if ($loc === self::OUTPUT) {
            return (strlen($this->outputCssPlain) / 1000);
        } else {
            throw new \InvalidArgumentException("Loc must be Output::INPUT or Output::OUTPUT constant, '$loc' given");
        }
    }

    /**
     * Get the gzipped size of either input or output CSS in KB
     * @param string $loc
     * @param int $level gzip level
     * @return float
     */
    public function gzippedSize($loc = self::OUTPUT, $level = -1)
    {
        if ($loc === self::OUTPUT && !$this->outputCss) {
            $this->formatted();
        }

        if ($loc === self::INPUT) {
            return (strlen(gzencode($this->inputCss, $level)) / 1000);
        } else if ($loc === self::OUTPUT) {
            return (strlen(gzencode($this->outputCssPlain, $level)) / 1000);
        } else {
            throw new \InvalidArgumentException("Loc must be Output::INPUT or Output::OUTPUT constant, '$loc' given");
        }
    }

    /**
     * Returns the formatted CSS Code and saves it into $this->output_css and $this->output_css_plain
     * @param string $defaultMedia default @media to add to selectors without any @media
     * @version 2.0
     */
    protected function generate($defaultMedia = null)
    {
        if ($this->outputCss && $this->outputCssPlain) {
            return;
        }

        $this->convertRawCss($defaultMedia);

        $template = $this->configuration->getTemplate();

        if ($this->configuration->getAddTimestamp()) {
            array_unshift(
                $this->tokens,
                array(self::COMMENT, ' Tools.bm8.com.cn ' . CSSTidy::getVersion() . ': ' . date('r') . ' ')
            );
        }

        $this->outputCss = $this->tokensToCss($template);

        $this->outputCssPlain = strip_tags($this->outputCss);
        $this->outputCssPlain = htmlspecialchars_decode($this->outputCssPlain, ENT_QUOTES);

        // If using spaces in the template, don't want these to appear in the plain output
        $this->outputCssPlain = str_replace('&#160;', '', $this->outputCssPlain);
    }

    /**
     * @param Template $template
     * @return string
     */
    protected function tokensToCss(Template $template)
    {
        $output = '';

        if (!empty($this->parsed->charset)) {
            // After '@charset' must be single space!
            $output .= "{$template->beforeAtRule}@charset {$template->beforeValue}{$this->parsed->charset}{$template->afterValueWithSemicolon}";
        }

        foreach ($this->parsed->import as $import) {
            $importValue = $import->getValue();
            $output .= "{$template->beforeAtRule}@import{$template->beforeValue}{$importValue}{$template->afterValueWithSemicolon}";
        }

        foreach ($this->parsed->namespace as $namespace) {
            $namespaceValue = $namespace->getValue();
            $output .= "{$template->beforeAtRule}@namespace{$template->beforeValue}{$namespaceValue}{$template->afterValueWithSemicolon}";
        }

        $output .= $template->lastLineInAtRule;
        $inAtOut = '';
        $out = &$output;

        foreach ($this->tokens as $key => $token) {
            switch ($token[0]) {
                case self::PROPERTY:
                    if ($this->configuration->getCaseProperties() === Configuration::LOWERCASE) {
                        $token[1] = strtolower($token[1]);
                    } else if ($this->configuration->getCaseProperties() === Configuration::UPPERCASE) {
                        $token[1] = strtoupper($token[1]);
                    }
                    $out .= $template->beforeProperty . $this->htmlsp($token[1]) . ':' . $template->beforeValue;
                    break;

                case self::VALUE:
                    $out .= $this->htmlsp($token[1]);
                    $nextToken = $this->seekNoComment($key);
                    if (($nextToken === self::SEL_END || $nextToken === self::AT_END) && $this->configuration->getRemoveLastSemicolon()) {
                        $out .= str_replace(';', '', $template->afterValueWithSemicolon);
                    } else {
                        $out .= $template->afterValueWithSemicolon;
                    }
                    break;

                case self::SEL_START:
                    if ($this->configuration->getLowerCaseSelectors()) {
                        $token[1] = strtolower($token[1]);
                    }
                    $out .= $template->beforeSelector . $this->htmlsp($token[1]) . $template->selectorOpeningBracket;
                    break;

                case self::SEL_END:
                    $out .= $template->selectorClosingBracket;
                    if ($this->seekNoComment($key) !== self::AT_END) {
                        $out .= $template->spaceBetweenBlocks;
                    }
                    break;

                case self::AT_START:
                    $out .= $template->beforeAtRule . $this->htmlsp($token[1]) . $template->bracketAfterAtRule;
                    $out = &$inAtOut;
                    break;

                case self::AT_END:
                    $out = &$output;
                    $out .= $template->indentInAtRule . str_replace("\n", "\n" . $template->indentInAtRule, $inAtOut);
                    $inAtOut = '';
                    $out .= $template->atRuleClosingBracket;
                    break;

                case self::COMMENT:
                    $out .= "$template->beforeComment/*{$this->htmlsp($token[1])}*/$template->afterComment";
                    break;

                case self::LINE_AT:
                    $out .= $token[1];
                    break;
            }
        }

        return trim($output);
    }

    /**
     * Gets the next token type, excluding comments
     * @param integer $key current position
     * @return int a token type
     */
    protected function seekNoComment($key)
    {
        while (isset($this->tokens[++$key])) {
            if ($this->tokens[$key][0] === self::COMMENT) {
                continue;
            }

            return $this->tokens[$key][0];
        }

        return 0;
    }

    /**
     * Converts $this->css array to a raw array ($this->tokens)
     * @param string $defaultMedia default @media to add to selectors without any @media
     */
    protected function convertRawCss($defaultMediaIsCurrentlyNotSupported = '')
    {
        $this->tokens = array();

        $this->blockToTokens($this->parsed);
    }

    /**
     * @param Element\Block $block
     */
    protected function blockToTokens(Element\Block $block)
    {
        if ($block instanceof Element\Selector) {
            $this->addToken(self::SEL_START, $block->getName());
        } else if ($block instanceof Element\AtBlock && !$block instanceof Element\Root) {
            $this->addToken(self::AT_START, $block->getName());
        }

        foreach ($block->elements as $element) {
            if ($element instanceof Element\Property) {
                /** @var Element\Property $element */
                $this->addToken(self::PROPERTY, $element->getName());
                $this->addToken(self::VALUE, $element->getValue());
            } else if ($element instanceof Element\Block) {
                /** @var Element\Element $element */
                $this->blockToTokens($element);
            } else if ($element instanceof Element\LineAt) {
                /** @var Element\LineAt $element */
                $this->addToken(self::LINE_AT, $element->__toString());
            } else if ($element instanceof Element\Comment) {
                if ($this->configuration->getPreserveComments()) {
                    $this->addToken(self::COMMENT, $element->__toString());
                }
            } else {
                throw new \Exception("Not supported element " . is_object($element) ? get_class($element) : gettype($element));
            }
        }

        if ($block instanceof Element\Selector) {
            $this->addToken(self::SEL_END);
        } else if ($block instanceof Element\AtBlock && !$block instanceof Element\Root) {
            $this->addToken(self::AT_END);
        }
    }

    /**
     * @param string $string
     * @return string
     */
    protected function htmlsp($string)
    {
        return htmlspecialchars($string, ENT_QUOTES, 'utf-8');
    }

    /**
     * Adds a token to $this->tokens
     * @param int $type
     * @param string $data
     * @return void
     */
    protected function addToken($type, $data = null)
    {
        $this->tokens[] = array($type, $data);
    }
}
?>