<?php

namespace CSSTidy\Optimise;

use CSSTidy\Logger;
use CSSTidy\Parser;
use CSSTidy\Element;

/**
 * CSS Optimising Class
 *
 * This class optimises CSS data generated by csstidy.
 *
 * @package csstidy
 * @author Florian Schmitz (floele at gmail dot com) 2005-2006
 * @version 1.0
 */
class Selector
{
    /** @var \CSSTidy\Logger */
    protected $logger;

    public function __construct(Logger $logger) {
        $this->logger = $logger;
    }

    public function process(Element\Block $block)
    {
        foreach ($block->elements as $element) {
            if ($element instanceof Element\Selector) {
                $this->optimise($element);
            } else if ($element instanceof Element\AtBlock) {
                $this->process($element);
            }
        }
    }

    /**
     * Removes universal selector * if is not necessary and spaces before and after '+' and '>'
     * @param \CSSTidy\Element\Selector $selector
     */
    public function optimise(Element\Selector $selector)
    {
        foreach ($selector->subSelectors as &$subSelector) {
            $optimised = '';

            for ($i = 0, $length = strlen($subSelector); $i < $length; $i++) {
                $current = $subSelector{$i};

                if (($current === '"' || $current === "'") && !Parser::escaped($subSelector, $i)) {
                    $optimised .= $this->skipString($subSelector, $i, $current);
                } else if ($current === '*' && isset($subSelector{$i+1}) && in_array($subSelector{$i+1}, array('.', '#', '[', ':'))) {
                    // remove unnecessary universal selector, FS#147
                } else if ($current === '>' || $current === '+') {
                    $optimised = rtrim($optimised) . $current;
                    if (isset($subSelector{$i+1}) && ctype_space($subSelector{$i+1})) {
                        ++$i; // Skip next space
                    }
                } else {
                    $optimised .= $current;
                }
            }

            $subSelector = $optimised;
        }

        $selector->setName(implode(',', $selector->subSelectors));
    }

    /**
     * @param string $string
     * @param int $i
     * @param string $endsWith
     * @return string Skipped string
     */
    protected function skipString($string, &$i, $endsWith)
    {
        $start = $i;

        do {
            $i = strpos($string, $endsWith, $i+1);
        } while (Parser::escaped($string, $i));

        return substr($string, $start, $i - $start + 1);
    }
}